/*

  libassh - asynchronous ssh2 client/server library.

  Copyright (C) 2013-2020 Alexandre Becoulet <alexandre.becoulet@free.fr>

  This library is free software; you can redistribute it and/or modify
  it under the terms of the GNU Lesser General Public License as
  published by the Free Software Foundation; either version 2.1 of the
  License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
  02110-1301 USA

*/

#define ASSH_PV
#define ASSH_ABI_UNSAFE  /* do not warn */

#include <assh/assh.h>
#include <assh/assh_cipher.h>
#include <assh/assh_context.h>

#include <stdlib.h>
#include <stdint.h>
#include <stdio.h>
#include <string.h>

#include "test.h"

struct cipher_test_s
{
  const char *algo;
  const char *key;
  const char *iv;
  const char *out;
  uint32_t seq;
  int_fast16_t head_size;
  int_fast16_t tail_size;
  int_fast16_t key_size;
  int_fast16_t iv_size;
  uint_fast8_t out_count;
};

const struct cipher_test_s vectors[] =
{
  { .algo = "none",
    .head_size = 16, .tail_size = 24, .out_count = 1,
    .out = "\x00\x01\x02\x03\x04\x05\x06\x07\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f"
    "\x10\x11\x12\x13\x14\x15\x16\x17\x18\x19\x1a\x1b\x1c\x1d\x1e\x1f"
    "\x20\x21\x22\x23\x24\x25\x26\x27" },

  { .algo = "arcfour",
    .head_size = 16, .tail_size = 24, .key_size = 16, .out_count = 2,
    .key = "\x01\xf2\xe3\xd4\xc5\xb6\xa7\x98\x89\x7a\x6b\x5c\x4d\x3e\x2f\x10",
    .out = "\x52\xfe\xd3\x07\x8e\x92\x28\x05\xa2\xb8\xab\x21\x26\xc6\x87\x8c"
    "\xde\xea\xf7\xaf\x70\xdc\x57\x87\xb6\x86\x19\x0b\x3d\xae\x7c\xb6"
    "\x86\x7e\x32\x69\x7c\xde\x73\x74"
    /* 1 */ "\x40\x62\xb7\x4c\xfd\x6a\x43\x34\xb1\xcf\x3b\xf8\x50\x41\x38\xb3"
    "\x4c\xaf\x1e\xae\x05\x5a\x38\x49\x41\x7f\xf4\x44\x37\x26\x3f\x49"
    "\xfb\xd4\x8e\x4e\x3d\x28\x4a\xcc" },
  { .algo = "arcfour128",
    .head_size = 16, .tail_size = 24, .key_size = 16, .out_count = 1,
    .key = "\x01\xf2\xe3\xd4\xc5\xb6\xa7\x98\x89\x7a\x6b\x5c\x4d\x3e\x2f\x10",
    .out = "\x46\x5f\x25\xeb\x80\x3e\xd3\xb1\x58\xb1\x94\xc7\x28\x93\x21\x6e"
    "\x13\x7f\xe7\x03\xb4\x25\xb6\xe4\x7d\x71\x92\x41\x18\x36\x2f\xe9"
    "\xd5\x6e\x43\x1e\x01\x6d\xa6\xf7" },
  { .algo = "arcfour256",
    .head_size = 16, .tail_size = 24, .key_size = 32, .out_count = 1,
    .key = "\xab\xcd\xef\x01\x23\x45\x67\x89\x01\xf2\xe3\xd4\xc5\xb6\xa7\x98"
    "\x89\x7a\x6b\x5c\x4d\x3e\x2f\x10\xab\xcd\xef\x01\x23\x45\x67\x89",
    .out = "\xc7\x54\xff\x60\x29\x4e\xd9\x67\xa6\xb5\x18\xa3\x80\x22\xb5\x33"
    "\x48\x57\x7b\x69\xc2\x3e\x9b\xc7\xb2\xf0\x17\xd2\xe8\xfc\xec\xde"
    "\x7c\xb4\xac\x97\x31\x3f\x05\x25" },

  { .algo = "3des-cbc",
    .head_size = 16, .tail_size = 24, .key_size = 24, .iv_size = 8, .out_count = 2,
    .key = "\x01\xf2\xe3\xd4\xc5\xb6\xa7\x98\x89\x7a\x6b\x5c\x4d\x3e\x2f\x10"
    "\xab\xcd\xef\x01\x23\x45\x67\x89",
    .iv =  "\x00\x11\x22\x33\x44\x55\x66\x77",
    .out = "\x20\x05\xab\x07\x2b\xd1\x44\x5e\x03\x04\x5a\x0f\x67\x0f\x2b\x3d"
    "\x28\x2c\x55\x84\xe1\xc0\x28\xe6\xf8\xa3\x19\xdf\x46\xb9\xd9\x06"
    "\x5b\xff\xfb\xf2\x83\x98\x08\x54"
    /* 1 */ "\x13\x10\x19\xa0\x3f\x14\xfb\xce\x5e\xf1\x40\x47\x8b\x4d\x52\x8a"
    "\x8c\xe7\xbb\xcb\xfb\xe6\xd7\x92\xdb\x92\x4c\xb9\x2a\x4b\x19\x7c"
    "\xe8\x31\x4a\x5a\xe3\x77\xb4\x4e"},

  { .algo = "3des-ctr",
    .head_size = 16, .tail_size = 24, .key_size = 24, .iv_size = 8, .out_count = 1,
    .key = "\x01\xf2\xe3\xd4\xc5\xb6\xa7\x98\x89\x7a\x6b\x5c\x4d\x3e\x2f\x10"
    "\xab\xcd\xef\x01\x23\x45\x67\x89",
    .iv =  "\x00\x11\x22\x33\x44\x55\x66\x77",
    .out = "\x7e\x58\x85\x30\x63\x6d\x27\xa7\x7f\xda\xd4\x13\x5c\x81\xc3\x30"
    "\x72\x98\x35\xa1\xbb\xe2\xd9\xcd\x37\xf3\x3b\x0c\xeb\x78\xde\x9a"
    "\xde\xdc\xed\xc4\xc3\x35\xd0\xf2" },

  { .algo = "cast128-cbc",
    .head_size = 16, .tail_size = 24, .key_size = 16, .iv_size = 8, .out_count = 1,
    .key = "\x01\xf2\xe3\xd4\xc5\xb6\xa7\x98\x89\x7a\x6b\x5c\x4d\x3e\x2f\x10",
    .iv =  "\x00\x11\x22\x33\x44\x55\x66\x77",
    .out = "\x00\x96\x92\x6f\xd1\xf8\x80\xb7\x00\xd5\x9e\x35\x94\x50\xc9\x87"
    "\x8a\x92\xaa\x17\xdf\x80\xea\xad\xb1\xcd\x88\xec\x78\x94\xe4\x99"
    "\x67\x55\x3f\x5d\xc9\x24\x67\xd7" },

  { .algo = "cast128-ctr",
    .head_size = 16, .tail_size = 24, .key_size = 16, .iv_size = 8, .out_count = 1,
    .key = "\x01\xf2\xe3\xd4\xc5\xb6\xa7\x98\x89\x7a\x6b\x5c\x4d\x3e\x2f\x10",
    .iv =  "\x00\x11\x22\x33\x44\x55\x66\x77",
    .out = "\x1e\x47\x5a\x1c\x7a\x92\x70\x20\xde\x0e\x0e\xff\x2c\xe0\x02\xc9"
    "\x5c\x52\xe7\x56\x05\x7c\x01\x6d\xc2\x1f\x56\xc2\x5b\xcd\x17\x55"
    "\xa4\xe7\xa6\x36\x55\x9b\x03\xa5" },

  { .algo = "idea-cbc",
    .head_size = 16, .tail_size = 24, .key_size = 16, .iv_size = 8, .out_count = 1,
    .key = "\x01\xf2\xe3\xd4\xc5\xb6\xa7\x98\x89\x7a\x6b\x5c\x4d\x3e\x2f\x10",
    .iv =  "\x00\x11\x22\x33\x44\x55\x66\x77",
    .out = "\x37\x59\xe9\xc8\x27\xa1\x76\x92\xcc\xf3\x79\xe1\x73\x44\xe7\x14"
    "\xb8\xde\x6d\x56\x24\xa0\x92\x29\x5f\x53\xd9\x6a\xc2\x0e\x62\xb8"
    "\x1b\xf5\xf3\xbf\x60\x91\x20\x71" },

  { .algo = "idea-ctr",
    .head_size = 16, .tail_size = 24, .key_size = 16, .iv_size = 8, .out_count = 1,
    .key = "\x01\xf2\xe3\xd4\xc5\xb6\xa7\x98\x89\x7a\x6b\x5c\x4d\x3e\x2f\x10",
    .iv =  "\x00\x11\x22\x33\x44\x55\x66\x77",
    .out = "\xb7\x84\x68\xd0\x7d\x75\x61\xd1\xff\x90\xe6\xc8\x03\xe6\x45\x4a"
    "\x85\xb6\x3d\x29\x8c\x6b\x8e\xe6\xe5\x04\x21\xa8\x21\xe8\x04\x32"
    "\x0a\x32\x27\x28\xd9\xf7\xfe\x2c" },

  { .algo = "blowfish-cbc",
    .head_size = 16, .tail_size = 24, .key_size = 16, .iv_size = 8, .out_count = 1,
    .key = "\x01\xf2\xe3\xd4\xc5\xb6\xa7\x98\x89\x7a\x6b\x5c\x4d\x3e\x2f\x10",
    .iv =  "\x00\x11\x22\x33\x44\x55\x66\x77",
    .out = "\xfe\x15\xcb\x8d\x95\x6b\xef\xcb\x66\xf9\xc3\x81\xb7\x2e\x40\xc0"
    "\x05\x1c\x7d\x88\x0f\x9e\xf2\x61\xb0\x96\xf3\x1c\xe6\x1b\x45\x33"
    "\x86\xde\x22\xd5\xe6\xf2\xfc\x13" },

  { .algo = "blowfish-ctr",
    .head_size = 16, .tail_size = 24, .key_size = 32, .iv_size = 8, .out_count = 1,
    .key = "\xab\xcd\xef\x01\x23\x45\x67\x89\x01\xf2\xe3\xd4\xc5\xb6\xa7\x98"
    "\x89\x7a\x6b\x5c\x4d\x3e\x2f\x10\xab\xcd\xef\x01\x23\x45\x67\x89",
    .iv =  "\x00\x11\x22\x33\x44\x55\x66\x77",
    .out = "\xfa\xdb\x7d\xca\xee\x5d\xdf\x6d\xb6\xe5\x43\x25\x26\xec\xcd\x8f"
    "\xce\x44\x21\x88\x76\x79\xf4\x4f\x73\xb8\x38\xc5\x25\x3b\x78\x5b"
    "\x67\x93\xa0\x34\x0a\x06\x67\x6c" },

  { .algo = "aes128-cbc",
    .head_size = 16, .tail_size = 32, .key_size = 16, .iv_size = 16, .out_count = 2,
    .key = "\x01\xf2\xe3\xd4\xc5\xb6\xa7\x98\x89\x7a\x6b\x5c\x4d\x3e\x2f\x10",
    .iv =  "\x00\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xdd\xee\xff",
    .out = "\x71\x4e\xcd\x44\xf2\xf5\x94\xdb\x9a\xdb\x54\xbf\x30\x5c\x91\x38"
    "\xe0\x40\x47\xb8\x91\x74\x3c\xe4\xc2\x93\xbd\xbe\x22\x62\xd8\x97"
    "\x81\xaa\xad\x9c\xba\xf3\xa6\x14\x84\x1e\x06\x44\xc3\x9b\x1d\xf1"
    /* 1 */ "\x74\x9b\xe4\x5f\x12\xc7\x82\x0f\xf3\x9d\x4b\x6e\xcb\xc8\x18\x2d"
    "\x86\xb0\x57\x2a\xd3\x04\xed\x67\x7a\xf5\xf1\x3c\x53\x33\x01\x0d"
    "\x65\xcd\xef\x9f\xb4\x15\x65\x3b\xab\xf5\xa0\x6b\x0d\x2f\xa1\x35" },

  { .algo = "aes192-cbc",
    .head_size = 16, .tail_size = 32, .key_size = 24, .iv_size = 16, .out_count = 1,
    .key = "\x01\xf2\xe3\xd4\xc5\xb6\xa7\x98\x89\x7a\x6b\x5c\x4d\x3e\x2f\x10"
    "\xab\xcd\xef\x01\x23\x45\x67\x89",
    .iv =  "\x00\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xdd\xee\xff",
    .out = "\x0f\x63\x3f\x8d\xfb\x58\xed\x52\xce\xba\x41\x19\x7b\x5f\xc7\xcc"
    "\x10\x23\x66\x0d\xc7\x3c\x93\x9e\xe0\xe7\x8c\x89\x3b\x67\xec\x94"
    "\xf9\x0b\x0d\xd8\x7f\x54\x92\x87\x0f\x45\xba\x9f\xe6\x23\x66\x55" },

  { .algo = "aes256-cbc",
    .head_size = 16, .tail_size = 32, .key_size = 32, .iv_size = 16, .out_count = 1,
    .key = "\xab\xcd\xef\x01\x23\x45\x67\x89\x01\xf2\xe3\xd4\xc5\xb6\xa7\x98"
    "\x89\x7a\x6b\x5c\x4d\x3e\x2f\x10\xab\xcd\xef\x01\x23\x45\x67\x89",
    .iv =  "\x00\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xdd\xee\xff",
    .out = "\x75\xa1\xe6\xa7\x80\xd8\x02\x5d\xa3\xb3\x95\x0a\xd6\x79\x89\xf0"
    "\xdb\x8c\x67\x15\xe7\x1d\x7e\xc8\x4d\x3b\xb4\x28\xf4\x20\x89\xfa"
    "\xce\xf0\x01\xd7\x89\xa8\x49\xcf\xa5\x23\xd3\x0f\x30\x11\x45\x48" },

  { .algo = "aes128-ctr",
    .head_size = 16, .tail_size = 32, .key_size = 16, .iv_size = 16, .out_count = 2,
    .key = "\x01\xf2\xe3\xd4\xc5\xb6\xa7\x98\x89\x7a\x6b\x5c\x4d\x3e\x2f\x10",
    .iv =  "\x00\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xdd\xee\xff",
    .out = "\xd6\x31\xdb\xb9\xb1\x82\xba\x96\x43\x8e\x98\x12\xbf\xf6\xeb\x22"
    "\x76\x98\xfb\x50\x93\x68\x00\x65\x51\x85\x3f\xc6\x5e\xd1\xc3\x33"
    "\xa3\x58\x51\xcb\x4f\x76\x07\x28\xa4\x8f\xa6\x28\x79\xf5\x3a\x94"
    /* 1 */ "\x8e\x94\x17\x61\x6e\x1c\x30\x95\x8e\x54\x3b\xbe\x44\x9d\xe7\xe5"
    "\x0b\x4f\x90\x7f\x6a\x40\xb9\x4c\xcb\x6c\x87\xc9\x77\x98\xc7\x45"
    "\xf7\x96\x56\x63\x4c\x89\x11\xec\x43\x6f\x86\x60\x42\x2d\x5d\xf5" },

  { .algo = "aes192-ctr",
    .head_size = 16, .tail_size = 32, .key_size = 24, .iv_size = 16, .out_count = 1,
    .key = "\x01\xf2\xe3\xd4\xc5\xb6\xa7\x98\x89\x7a\x6b\x5c\x4d\x3e\x2f\x10"
    "\xab\xcd\xef\x01\x23\x45\x67\x89",
    .iv =  "\x00\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xdd\xee\xff",
    .out = "\x81\xaa\x1c\xd5\x48\xa4\x7c\x80\xc6\x56\x3c\x3b\xb3\xcd\xb6\xdb"
    "\xfa\x70\x2d\x77\xe6\xcd\xfd\xdd\x47\x6d\x2f\xec\xde\x34\xa4\xbd"
    "\xe0\x1a\xf9\x0b\xad\x1c\x73\xfb\xc8\xa8\xd9\x04\x03\x3a\x1e\xa5" },

  { .algo = "aes256-ctr",
    .head_size = 16, .tail_size = 32, .key_size = 32, .iv_size = 16, .out_count = 1,
    .key = "\xab\xcd\xef\x01\x23\x45\x67\x89\x01\xf2\xe3\xd4\xc5\xb6\xa7\x98"
    "\x89\x7a\x6b\x5c\x4d\x3e\x2f\x10\xab\xcd\xef\x01\x23\x45\x67\x89",
    .iv =  "\x00\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xdd\xee\xff",
    .out = "\xac\xd9\x74\xa3\x42\x88\xd2\x52\x8d\xbf\x8d\x5c\x4b\xd1\xee\xd4"
    "\x9e\x4b\x6f\x2a\x54\x27\x8b\x51\x5a\x27\x4c\xc3\x48\xfc\x8c\x38"
    "\x15\x25\x52\xa0\x67\x54\xff\x85\xd4\x25\xe2\x34\x14\x52\x8b\x52" },

  { .algo = "aes128-gcm@openssh.com",
    .head_size = 0, .tail_size = 32 + 4, .key_size = 16, .iv_size = 12, .out_count = 2,
    .key = "\x01\xf2\xe3\xd4\xc5\xb6\xa7\x98\x89\x7a\x6b\x5c\x4d\x3e\x2f\x10",
    .iv =  "\x00\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb",
    .out = "\x00\x01\x02\x03\x42\x11\x00\xe3\x3f\x8d\x48\x21\xcb\x9f\x6d\x14"
    "\x29\xc1\x73\xd8\x9d\x25\xe9\xfe\xe4\x4a\x08\x36\xd1\xfd\x7d\x4d"
    "\xd1\xef\x26\x4b"
    /* 1 */ "\x00\x01\x02\x03\x35\x80\x71\x91\x25\xd0\xc2\xfa\xfc\xec\x97\xfd"
    "\x16\xb1\xdd\x2a\x5f\x1b\x05\xc1\x6e\xe0\xb8\x84\x50\xbc\x06\x1f"
    "\xfd\x52\x88\xc6" },

  { .algo = "aes256-gcm@openssh.com",
    .head_size = 0, .tail_size = 32 + 4, .key_size = 32, .iv_size = 12, .out_count = 1,
    .key = "\xab\xcd\xef\x01\x23\x45\x67\x89\x01\xf2\xe3\xd4\xc5\xb6\xa7\x98"
    "\x89\x7a\x6b\x5c\x4d\x3e\x2f\x10\xab\xcd\xef\x01\x23\x45\x67\x89",
    .iv =  "\x00\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb",
    .out = "\x00\x01\x02\x03\x72\x6f\x20\xef\x5a\x49\x4d\x73\xc9\xbd\x91\x2f"
    "\xc6\xdd\xd3\x77\x08\xc4\x04\xeb\x9c\xf1\x6b\x28\x2b\x66\x9f\xae"
    "\x5c\x26\xe5\x11" },

  { .algo = "aes128-ocb@libassh.org",
    .head_size = 0, .tail_size = 32 + 4, .key_size = 16, .iv_size = 12, .out_count = 4,
    .key = "\x01\xf2\xe3\xd4\xc5\xb6\xa7\x98\x89\x7a\x6b\x5c\x4d\x3e\x2f\x10",
    .iv =  "\x00\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xfe",
    .out = "\x00\x01\x02\x03\x5f\xa0\x12\xf2\x6c\x52\xbc\x53\x34\xdf\xd7\x69"
    "\x77\xe0\xbe\x15\xc5\x08\xa2\xf1\xcf\x8b\xee\xed\xad\xab\x16\xef"
    "\x16\xad\x64\xbe"
    /* 2 */ "\x00\x01\x02\x03\x00\x54\xbf\xb3\x6e\x4a\x5d\xa6\xf6\xde\xd1\x73"
    "\x67\x39\x08\xf2\xf1\x06\x33\xb5\xc3\xc7\xef\x26\xc6\x41\x8f\x04"
    "\xbb\xfa\xa1\xdb"
    /* 3 */ "\x00\x01\x02\x03\x51\xd7\xaf\xb9\xe1\x30\x80\x9a\x9b\xfc\x93\x4a"
    "\xdc\xd7\x35\xe2\x67\xfe\x3b\x09\xa5\xdf\xfa\x10\xbe\xdd\xb3\xb7"
    "\xe3\xc3\x6f\x0a"
    /* 4 */ "\x00\x01\x02\x03\x81\x62\xf3\x55\xef\x54\xd0\x8e\xae\x8b\xc9\x10"
    "\x5d\x27\x61\x41\x7a\xf8\x9a\x8c\x4f\x89\x5f\x37\x60\x5a\x69\x5b"
    "\x3c\xb2\xcd\xb1" },

  { .algo = "aes128-ocb@libassh.org",
    .head_size = 0, .tail_size = 48 + 4, .key_size = 16, .iv_size = 12, .out_count = 1,
    .key = "\x01\xf2\xe3\xd4\xc5\xb6\xa7\x98\x89\x7a\x6b\x5c\x4d\x3e\x2f\x10",
    .iv =  "\x00\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xfe",
    .out = "\x00\x01\x02\x03\x5f\xa0\x12\xf2\x6c\x52\xbc\x53\x34\xdf\xd7\x69"
    "\x77\xe0\xbe\x15\xf1\x69\x2a\x12\xc6\x37\xfe\xa7\x96\x05\x2a\x5e"
    "\xf3\xa9\x68\xf7\xef\x5e\xca\x93\xe2\x66\xe8\x82\xa9\x11\xbe\x1e"
    "\x50\xa9\x53\x17" },

  { .algo = "aes128-ocb@libassh.org",
    .head_size = 0, .tail_size = 64 + 4, .key_size = 16, .iv_size = 12, .out_count = 1,
    .key = "\x01\xf2\xe3\xd4\xc5\xb6\xa7\x98\x89\x7a\x6b\x5c\x4d\x3e\x2f\x10",
    .iv =  "\x00\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xfe",
    .out = "\x00\x01\x02\x03\x5f\xa0\x12\xf2\x6c\x52\xbc\x53\x34\xdf\xd7\x69"
    "\x77\xe0\xbe\x15\xf1\x69\x2a\x12\xc6\x37\xfe\xa7\x96\x05\x2a\x5e"
    "\xf3\xa9\x68\xf7\xf4\xfd\xd1\x74\xd0\xbd\xaa\x3a\x25\x30\xcc\x16"
    "\x7b\xbd\x7a\x97\x19\x24\xe9\xeb\xc4\x6a\xde\x2f\x69\xc6\xe5\x3b"
    "\x9e\x26\x31\x25" },

  { .algo = "aes256-ocb@libassh.org",
    .head_size = 0, .tail_size = 32 + 4, .key_size = 32, .iv_size = 12, .out_count = 1,
    .key = "\xab\xcd\xef\x01\x23\x45\x67\x89\x01\xf2\xe3\xd4\xc5\xb6\xa7\x98"
    "\x89\x7a\x6b\x5c\x4d\x3e\x2f\x10\xab\xcd\xef\x01\x23\x45\x67\x89",
    .iv =  "\x00\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xff",
    .out = "\x00\x01\x02\x03\x22\x14\xc9\x23\x52\x12\xb5\xe9\x09\xd5\x41\x5c"
    "\x56\x88\xb4\xb5\x0d\x3e\x57\xce\x09\xbd\x58\x70\x88\x04\xe7\x8b"
    "\xe1\xba\x1b\xf2" },

  { .algo = "twofish128-cbc",
    .head_size = 16, .tail_size = 32, .key_size = 16, .iv_size = 16, .out_count = 1,
    .key = "\x01\xf2\xe3\xd4\xc5\xb6\xa7\x98\x89\x7a\x6b\x5c\x4d\x3e\x2f\x10",
    .iv =  "\x00\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xdd\xee\xff",
    .out = "\xe4\xd6\x07\x47\xf0\xae\xdf\x39\xb6\x3a\x01\xb1\xa4\x74\xa2\xca"
    "\xdf\xba\x82\xf0\x0f\x4e\x85\x62\xfe\x79\x30\xf4\xb5\xe1\xba\x9f"
    "\xa1\x3a\x61\x53\x0b\x6f\x92\x33\x4a\x41\x03\x52\xca\x37\x7b\x1c" },

  { .algo = "twofish256-cbc",
    .head_size = 16, .tail_size = 32, .key_size = 32, .iv_size = 16, .out_count = 1,
    .key = "\xab\xcd\xef\x01\x23\x45\x67\x89\x01\xf2\xe3\xd4\xc5\xb6\xa7\x98"
    "\x89\x7a\x6b\x5c\x4d\x3e\x2f\x10\xab\xcd\xef\x01\x23\x45\x67\x89",
    .iv =  "\x00\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xdd\xee\xff",
    .out = "\x96\x30\x7a\xee\x2c\x39\x95\x84\x57\x27\x29\x70\xfe\x28\x97\xa7"
    "\x19\x73\x30\x1e\x3b\x79\x5b\x42\x80\x8c\x81\xab\x69\x94\xea\xdf"
    "\x4b\x78\x12\x3e\x5b\x54\x8e\x94\x99\x6d\xe4\x93\x7f\x0e\x94\x4a" },

  { .algo = "twofish128-ctr",
    .head_size = 16, .tail_size = 32, .key_size = 16, .iv_size = 16, .out_count = 1,
    .key = "\x01\xf2\xe3\xd4\xc5\xb6\xa7\x98\x89\x7a\x6b\x5c\x4d\x3e\x2f\x10",
    .iv =  "\x00\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xdd\xee\xff",
    .out = "\x41\x2b\x7d\xea\xa2\x4e\xdb\xed\x2a\x0d\x03\xc6\x7a\x02\xc7\xaf"
    "\xe6\xfe\xc5\xf7\x4d\x46\xcf\xab\xf7\xc5\x0e\x88\x95\x1b\x8e\xc0"
    "\x44\x84\xb0\x25\x66\x75\x0b\x41\x14\xbc\xd5\xb8\x22\x59\x2a\xbf" },

  { .algo = "twofish256-ctr",
    .head_size = 16, .tail_size = 32, .key_size = 32, .iv_size = 16, .out_count = 1,
    .key = "\xab\xcd\xef\x01\x23\x45\x67\x89\x01\xf2\xe3\xd4\xc5\xb6\xa7\x98"
    "\x89\x7a\x6b\x5c\x4d\x3e\x2f\x10\xab\xcd\xef\x01\x23\x45\x67\x89",
    .iv =  "\x00\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xdd\xee\xff",
    .out = "\xc4\x97\x9c\x49\xef\xec\x81\xdc\xea\x61\x3a\x4b\xfb\x43\xdf\xc2"
    "\x0b\x8a\x8e\x51\x48\xaa\x8b\x4f\x2f\x50\x4d\x72\xc4\xb8\x15\x7c"
    "\x85\x8b\x8f\xa5\xf1\xf9\x98\x8f\xcf\x6e\x0e\x52\x88\xa3\x94\x85" },

  { .algo = "twofish128-gcm@libassh.org",
    .head_size = 0, .tail_size = 32 + 4, .key_size = 16, .iv_size = 12, .out_count = 1,
    .key = "\x01\xf2\xe3\xd4\xc5\xb6\xa7\x98\x89\x7a\x6b\x5c\x4d\x3e\x2f\x10",
    .iv =  "\x00\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb",
    .out = "\x00\x01\x02\x03\x08\x67\x39\xff\x94\x18\xb8\xf0\x9a\xac\xdb\x54"
    "\x4f\x4a\x1d\x64\x19\x5f\x1d\x1e\x22\x1e\x9a\x0b\x93\x9a\x2b\x18"
    "\x42\x81\xd5\x77" },

  { .algo = "twofish256-gcm@libassh.org",
    .head_size = 0, .tail_size = 32 + 4, .key_size = 32, .iv_size = 12, .out_count = 1,
    .key = "\xab\xcd\xef\x01\x23\x45\x67\x89\x01\xf2\xe3\xd4\xc5\xb6\xa7\x98"
    "\x89\x7a\x6b\x5c\x4d\x3e\x2f\x10\xab\xcd\xef\x01\x23\x45\x67\x89",
    .iv =  "\x00\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb",
    .out = "\x00\x01\x02\x03\x29\x1c\xef\x8c\xbb\x8b\x07\xd2\x77\x9e\x5c\x94"
    "\x29\x96\xec\xda\x5f\x47\x2e\x6d\x47\x8e\x96\x8b\xe1\x1a\xa2\x9f"
    "\x71\x8c\x0d\xf0" },

  { .algo = "serpent128-cbc",
    .head_size = 16, .tail_size = 32, .key_size = 16, .iv_size = 16, .out_count = 1,
    .key = "\x01\xf2\xe3\xd4\xc5\xb6\xa7\x98\x89\x7a\x6b\x5c\x4d\x3e\x2f\x10",
    .iv =  "\x00\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xdd\xee\xff",
    .out = "\x79\x5f\x04\xd8\xab\xe5\x19\x65\x89\xb5\x69\xc0\x5a\xb7\x47\x8a"
    "\x04\x0f\x39\x72\xae\x27\x12\xaa\x01\x1a\xf0\xa3\x5c\xd9\xb2\x0a"
    "\xe4\x71\xc9\xbd\xcd\x07\x3e\x85\x7c\x47\xfd\x42\xa3\x77\x0c\x8b" },

  { .algo = "serpent192-cbc",
    .head_size = 16, .tail_size = 32, .key_size = 24, .iv_size = 16, .out_count = 1,
    .key = "\x01\xf2\xe3\xd4\xc5\xb6\xa7\x98\x89\x7a\x6b\x5c\x4d\x3e\x2f\x10"
    "\xab\xcd\xef\x01\x23\x45\x67\x89",
    .iv =  "\x00\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xdd\xee\xff",
    .out = "\xde\x58\x11\xad\xca\x08\xd2\xd1\xdf\x20\x88\xf0\x2d\xc0\xcb\x69"
    "\x3b\xff\x41\xa0\x25\xea\xe8\x2a\x89\x39\x06\x9e\x27\xef\x93\x56"
    "\xc7\x0f\xbe\x23\x1e\x42\x98\xfc\xf7\x7d\xbc\xe1\x0d\x10\xaa\xfa" },

  { .algo = "serpent256-cbc",
    .head_size = 16, .tail_size = 32, .key_size = 32, .iv_size = 16, .out_count = 1,
    .key = "\xab\xcd\xef\x01\x23\x45\x67\x89\x01\xf2\xe3\xd4\xc5\xb6\xa7\x98"
    "\x89\x7a\x6b\x5c\x4d\x3e\x2f\x10\xab\xcd\xef\x01\x23\x45\x67\x89",
    .iv =  "\x00\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xdd\xee\xff",
    .out = "\x5b\x6f\x35\xc3\x8a\x04\x11\x71\xe8\xaa\x32\x4d\x37\x57\x03\xe1"
    "\x74\xff\x32\xbf\xb2\x7f\x0b\x32\x4f\x3f\x20\xca\xa9\x67\xef\x14"
    "\x95\x33\xae\x46\xba\x87\x96\x54\xf1\x0a\xdc\xd7\x7c\x25\x5a\x5c" },

  { .algo = "serpent128-ctr",
    .head_size = 16, .tail_size = 32, .key_size = 16, .iv_size = 16, .out_count = 1,
    .key = "\x01\xf2\xe3\xd4\xc5\xb6\xa7\x98\x89\x7a\x6b\x5c\x4d\x3e\x2f\x10",
    .iv =  "\x00\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xdd\xee\xff",
    .out = "\x84\x01\x7b\x45\x1e\x17\x92\x70\x14\x54\x67\xcc\x64\x8f\xdd\x26"
    "\x5f\xaa\x76\x2e\x52\xe6\x76\x9d\xa1\x80\x72\x7c\xbe\xf4\x76\x5e"
    "\x65\x0d\xd1\x2e\xa3\x76\x5c\xe1\xf3\xdc\xab\x07\x6f\xe7\x99\x46" },

  { .algo = "serpent192-ctr",
    .head_size = 16, .tail_size = 32, .key_size = 24, .iv_size = 16, .out_count = 1,
    .key = "\x01\xf2\xe3\xd4\xc5\xb6\xa7\x98\x89\x7a\x6b\x5c\x4d\x3e\x2f\x10"
    "\xab\xcd\xef\x01\x23\x45\x67\x89",
    .iv =  "\x00\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xdd\xee\xff",
    .out = "\x5a\xc8\x70\xdf\x4f\xe0\x15\x8c\xa3\x9c\x92\x3a\xfe\x48\xf8\xc1"
    "\x51\xca\x56\xd1\xbf\xa1\xde\x42\x7d\xd0\xd6\x6d\x89\x9a\x35\x87"
    "\x12\xa4\x7a\xd0\xeb\x0c\xb2\xb9\x57\x8e\x59\x82\x5c\x95\x1c\x5f" },

  { .algo = "serpent256-ctr",
    .head_size = 16, .tail_size = 32, .key_size = 32, .iv_size = 16, .out_count = 1,
    .key = "\xab\xcd\xef\x01\x23\x45\x67\x89\x01\xf2\xe3\xd4\xc5\xb6\xa7\x98"
    "\x89\x7a\x6b\x5c\x4d\x3e\x2f\x10\xab\xcd\xef\x01\x23\x45\x67\x89",
    .iv =  "\x00\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xdd\xee\xff",
    .out = "\xc7\xbf\xdf\xc6\x36\x3e\xed\x9c\xc9\xd5\x63\x09\x2e\x4c\x64\x24"
    "\x04\xb8\x51\x60\x2f\xc0\x8f\x67\xcb\xeb\x9f\xf5\x4e\x20\x05\xb0"
    "\xc8\xfd\x7e\x75\x0e\x67\x44\xe8\x20\xa6\x56\x78\xeb\xee\x95\x25" },

  { .algo = "serpent128-gcm@libassh.org",
    .head_size = 0, .tail_size = 32 + 4, .key_size = 16, .iv_size = 12, .out_count = 2,
    .key = "\x01\xf2\xe3\xd4\xc5\xb6\xa7\x98\x89\x7a\x6b\x5c\x4d\x3e\x2f\x10",
    .iv =  "\x00\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb",
    .out = "\x00\x01\x02\x03\x8f\x25\x5a\xd2\xb4\xce\x7c\x9c\x9f\x3b\x3a\xe5"
    "\xcf\xe2\x4e\x17\x1c\xdd\xbe\x48\x8b\xfd\xe3\x5c\x96\x37\x26\xf7"
    "\xd8\x16\xd1\xf9"
    /* 1 */ "\x00\x01\x02\x03\xdf\xab\xdb\x66\xa5\x2d\xaf\x3e\x2d\x4b\xe6\xf4"
    "\x86\xdf\x04\x02\x25\x0f\x20\x12\xea\xd8\xe5\xca\x82\x36\x55\x03"
    "\x81\x4a\x53\x1c" },

  { .algo = "serpent256-gcm@libassh.org",
    .head_size = 0, .tail_size = 32 + 4, .key_size = 32, .iv_size = 12, .out_count = 1,
    .key = "\xab\xcd\xef\x01\x23\x45\x67\x89\x01\xf2\xe3\xd4\xc5\xb6\xa7\x98"
    "\x89\x7a\x6b\x5c\x4d\x3e\x2f\x10\xab\xcd\xef\x01\x23\x45\x67\x89",
    .iv =  "\x00\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb",
    .out = "\x00\x01\x02\x03\xbf\xf8\x8c\x32\x6f\x4c\x9a\x3a\xfb\x2e\x2a\x31"
    "\x57\x1f\xf8\x6f\xde\x3d\x94\x26\xb7\x58\xc2\x72\xb1\xfb\x86\x1d"
    "\x8d\x64\x15\x44" },

  { .algo = "camellia128-cbc",
    .head_size = 16, .tail_size = 32, .key_size = 16, .iv_size = 16, .out_count = 1,
    .key = "\x01\xf2\xe3\xd4\xc5\xb6\xa7\x98\x89\x7a\x6b\x5c\x4d\x3e\x2f\x10",
    .iv =  "\x00\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xdd\xee\xff",
    .out = "\x76\xbc\x78\x71\x73\x34\x05\x7d\x58\x6e\x17\xb3\x7d\x53\x5e\xd7"
    "\x0e\xa2\x61\x5d\x21\xd4\xd3\xfe\x2d\x75\x7b\xb4\x24\xe5\x43\x21"
    "\x18\x0c\xf6\xaa\x62\xae\x30\x18\x9a\xe9\x89\x1b\x77\x5e\x11\x0e" },

  { .algo = "camellia192-cbc",
    .head_size = 16, .tail_size = 32, .key_size = 24, .iv_size = 16, .out_count = 1,
    .key = "\x01\xf2\xe3\xd4\xc5\xb6\xa7\x98\x89\x7a\x6b\x5c\x4d\x3e\x2f\x10"
    "\xab\xcd\xef\x01\x23\x45\x67\x89",
    .iv =  "\x00\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xdd\xee\xff",
    .out = "\x77\x5c\xec\xd6\x01\x22\x8a\xa8\xcd\x93\x66\x53\x18\x70\x9b\x8c"
    "\xcf\x9d\xfe\x02\x44\x01\x21\x5d\x1c\x89\x1a\xba\x28\x7c\xc8\xba"
    "\x8f\x7f\x2f\x68\xdc\x56\xc0\x8d\x72\x60\x9c\xfa\x39\x1d\x5b\x8e" },

  { .algo = "camellia256-cbc",
    .head_size = 16, .tail_size = 32, .key_size = 32, .iv_size = 16, .out_count = 1,
    .key = "\xab\xcd\xef\x01\x23\x45\x67\x89\x01\xf2\xe3\xd4\xc5\xb6\xa7\x98"
    "\x89\x7a\x6b\x5c\x4d\x3e\x2f\x10\xab\xcd\xef\x01\x23\x45\x67\x89",
    .iv =  "\x00\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xdd\xee\xff",
    .out = "\xf3\xb7\x38\x7c\x83\x15\x15\xed\xf0\xf3\xb7\xd5\xab\x9e\x52\x34"
    "\x7b\xae\xdf\x9a\x8b\x3a\xcf\x0e\x61\x9e\xf2\x27\x02\xdf\x54\x3b"
    "\xcd\x01\x79\xdb\xbc\x1e\x15\xe4\x93\x82\x04\xc3\xdd\x5f\xc3\xe3" },

  { .algo = "camellia128-ctr",
    .head_size = 16, .tail_size = 32, .key_size = 16, .iv_size = 16, .out_count = 1,
    .key = "\x01\xf2\xe3\xd4\xc5\xb6\xa7\x98\x89\x7a\x6b\x5c\x4d\x3e\x2f\x10",
    .iv =  "\x00\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xdd\xee\xff",
    .out = "\xe8\x1b\x8d\xb1\xf6\xa1\x70\xd1\xbe\x7d\x13\xbb\x36\xe4\x8d\xbc"
    "\x63\x7d\x1f\x80\x9c\x01\x86\x6d\x92\x3d\x53\xa9\x7d\x8d\xef\x58"
    "\x0d\x76\xbc\x64\x76\x41\x55\x0e\xd9\x6f\x67\xea\xd5\x09\x1b\x0f" },

  { .algo = "camellia192-ctr",
    .head_size = 16, .tail_size = 32, .key_size = 24, .iv_size = 16, .out_count = 1,
    .key = "\x01\xf2\xe3\xd4\xc5\xb6\xa7\x98\x89\x7a\x6b\x5c\x4d\x3e\x2f\x10"
    "\xab\xcd\xef\x01\x23\x45\x67\x89",
    .iv =  "\x00\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xdd\xee\xff",
    .out = "\x93\x9c\x2c\x91\x55\xdf\x42\xb4\x95\x27\x1b\xe6\xab\x8c\x13\x28"
    "\x72\x13\x91\xeb\x69\xa4\xae\x8b\xd8\x13\xb8\x19\xea\x0f\x07\x95"
    "\xc3\x58\x33\x92\x42\xed\x59\xa4\xe5\xf2\xef\x0d\xfc\x0a\x25\x60" },

  { .algo = "camellia256-ctr",
    .head_size = 16, .tail_size = 32, .key_size = 32, .iv_size = 16, .out_count = 1,
    .key = "\xab\xcd\xef\x01\x23\x45\x67\x89\x01\xf2\xe3\xd4\xc5\xb6\xa7\x98"
    "\x89\x7a\x6b\x5c\x4d\x3e\x2f\x10\xab\xcd\xef\x01\x23\x45\x67\x89",
    .iv =  "\x00\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xdd\xee\xff",
    .out = "\xb0\x1b\xfc\x02\xf7\x2e\x42\x23\x5a\x25\x31\x5a\x0f\xc7\x63\x23"
    "\x11\x46\x47\xa6\x01\xa7\xa5\xeb\x0d\x7d\x51\x79\x62\x18\x70\xcb"
    "\x9d\xd2\x8b\x6e\xf3\x54\x4b\xe4\xf2\xe3\xa9\x95\x14\x31\xe1\xb8" },

  { .algo = "chacha20-poly1305@openssh.com",
    .head_size = 4, .tail_size = 44, .key_size = 64, .out_count = 2, .seq = 101,
    .key = "\xab\xcd\xef\x01\x23\x45\x67\x89\x01\xf2\xe3\xd4\xc5\xb6\xa7\x98"
    "\x89\x7a\x6b\x5c\x4d\x3e\x2f\x10\xab\xcd\xef\x01\x23\x45\x67\x89"
    "\xab\xcd\xef\x10\x23\x45\x76\x89\x01\x2f\xe3\xd4\x5c\xb6\xa7\x98"
    "\x89\x7a\x6b\xc5\x4d\x3e\x2f\x10\xba\xcd\xef\x01\x32\x45\x67\x89",
    .out = "\xef\x25\xbf\xb1\xde\x08\x41\x88\x99\x1a\x8f\x50\x4c\xce\x25\x0e"
    "\x1a\x35\x51\x12\x94\xfd\x30\x28\x54\xb0\xfd\x99\xb2\x2c\x1a\x5e"
    "\xa8\xab\x95\x1c\x4f\xcf\xce\x31\xca\xce\xfd\x6c\x0f\x14\x3a\x0a"
    /* 1 */ "\x8a\x4b\xa1\x64\x31\x50\x8d\x55\x28\x31\x48\x38\x45\x55\xcb\x2d"
    "\x57\x32\xa6\xf7\xac\x82\xae\xf2\x84\x57\x74\x2a\x40\x7c\xc6\xa0"
    "\x94\xcf\xc7\xad\x25\xf3\x81\x72\x48\x0f\xcc\x29\x15\x23\x94\x40" },

  { 0 }
};

void test_cipher(const struct cipher_test_s *t,
		 const struct assh_algo_cipher_s *ca)
{
  struct assh_context_s context;

  if (assh_context_init(&context, ASSH_CLIENT_SERVER, test_leaks_allocator,
			NULL, NULL, NULL))
    TEST_FAIL("context init");

  if (t->iv_size != ca->iv_size)
    TEST_FAIL("iv size");
  if (t->key_size != ca->key_size)
    TEST_FAIL("key size");

  if (!t->out_count)
    TEST_FAIL("no output");

  size_t size = t->head_size + t->tail_size;

  uint8_t buf[size];
  uint8_t buf2[size];
  for (int i = 0; i < size; i++)
    buf[i] = i;

  printf("testing %s, %s: ",
	  t->algo, ca->algo.implem);

  void *ctx, *ctx2, *ctx3;

  if (posix_memalign(&ctx, 32, ca->ctx_size) ||
      posix_memalign(&ctx2, 32, ca->ctx_size) ||
      posix_memalign(&ctx3, 32, ca->ctx_size))
    TEST_FAIL("malloc");

  if (ca->f_init(&context, ctx, (const uint8_t*)t->key,
		 t->iv_size ? (const uint8_t*)t->iv : NULL, 1))
    TEST_FAIL("encrypt init");
  if (ca->f_init(&context, ctx2, (const uint8_t*)t->key,
		 t->iv_size ? (const uint8_t*)t->iv : NULL, 0))
    TEST_FAIL("decrypt init");

  uint32_t seq = t->seq;
  const size_t count = 4;
  uint_fast8_t  i;

  for (i = 0; i < count; i++, seq++)
    {
      assh_bool_t check_output = (i < t->out_count);

      /* encrypt */
      putchar('E');
      if (ca->f_process(ctx, buf, size, ASSH_CIPHER_PCK_TAIL, seq))
	TEST_FAIL("encrypt %u", i);

      memcpy(buf2, buf, size);

      if (check_output)
	{
	  putchar('Q');
	  if (memcmp(buf, t->out + size * i, size))
	    {
	      assh_hexdump(stderr, "output", buf, size);
	      assh_hexdump(stderr, "expected   ", t->out + size * i, size);
	      TEST_FAIL("encrypt output %u", i);
	    }
	}

      /* decrypt */
      putchar('d');
      if (ca->f_process(ctx2, buf, t->head_size, ASSH_CIPHER_PCK_HEAD, seq))
	TEST_FAIL("decrypt head %u", i);

      size_t j;

      putchar('D');
      if (ca->auth_size)
	{
	  if (ca->f_process(ctx2, buf, size, ASSH_CIPHER_PCK_TAIL, seq))
	    TEST_FAIL("decrypt tail %u", i);

	  putchar('t');

	  for (j = 0; j < size; j++)
	    {
	      buf2[j] ^= 1 << (rand() % 8);

	      if (ca->f_init(&context, ctx3, (const uint8_t*)t->key,
			     t->iv_size ? (const uint8_t*)t->iv : NULL, 0))
		TEST_FAIL("auth decrypt init");

	      if (!ca->f_process(ctx3, buf2, t->head_size, ASSH_CIPHER_PCK_HEAD, seq) &&
		  !ca->f_process(ctx3, buf2, size, ASSH_CIPHER_PCK_TAIL, seq))
		TEST_FAIL("auth %u", i);

	      ca->f_cleanup(&context, ctx3);
	    }
	}
      else
	{
	  if (ca->f_process(ctx2, buf + t->head_size, t->tail_size,
			    ASSH_CIPHER_PCK_TAIL, seq))
	    TEST_FAIL("decrypt tail %u", i);
	}

      putchar('q');
      for (j = 0; j < size - ca->auth_size; j++)
	if (buf[j] != j)
	  {
	    assh_hexdump(stderr, "output", buf, size);
	    TEST_FAIL("decrypt output %u", i);
	  }
    }

  printf("\n");

  ca->f_cleanup(&context, ctx);
  free(ctx);

  ca->f_cleanup(&context, ctx2);
  free(ctx2);

  free(ctx3);

  assh_context_cleanup(&context);
}

int
main(int argc, char **argv)
{
  setvbuf(stdout, NULL, _IONBF, 0);

  if (assh_deps_init())
    return -1;

  uint_fast16_t i;
  for (i = 0; vectors[i].algo != NULL; i++)
    {
      const struct cipher_test_s *t = &vectors[i];

      if (!strcmp(t->algo, "none"))
	{
	  test_cipher(t, &assh_cipher_none);
	  continue;
	}

      assh_bool_t done = 0;
      const struct assh_algo_s **a;

      for (a = assh_algo_table; *a; a++)
	{
	  if (!assh_algo_name_match(*a, ASSH_ALGO_CIPHER,
				    t->algo, strlen(t->algo)))
	    continue;

	  if (!assh_algo_supported(*a))
	    continue;

	  done = 1;
	  test_cipher(t, assh_algo_cipher(*a));

	  if (test_alloc_size != 0)
	    TEST_FAIL("memory leak detected, %zu bytes allocated\n", test_alloc_size);
	}

      if (!done)
	printf("skipping %s, no implementation\n", t->algo);
    }

  puts("\nTest passed");
  return 0;
}
