#!/usr/bin/env python
# -*- coding: iso-8859-1

# Copyright (C) 2005 Andre Kloss
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

__title__="shuffle Last Played"
__version__="0.3"
__author__="Andre Kloss"
__email__="andre.kloss@gmx.net"

"""
VERSION HISTORY:
0.1 2005-08-02 initial release
0.2 2005-08-18 automatic iPod shuffle detection
0.3 2005-09-03 shuffle detection for windows (untested), sequence mode detection
"""

# some constants
headerSize = 18
songSize = 558
titleAddress = 33
titleLength = 522
shufflePosAddress = 3
mtab = "/etc/mtab" # POSIX standard position
fstab = "/etc/fstab"
base = "iPod_Control/iTunes/" # assuming you run from shuffle root

import os
import re

def int24(s):
	"""read a 24-bit(=3 char) string, return the integer value"""
	return ord(s[0]) + (ord(s[1])<<8) + (ord(s[2])<<16)

def string16(s):
	"""read the title strings (does not handle unicode yet)"""
	return "".join([x for x in s[0::2] if x != '\0'])

def getSong(iTunesSD, trackNo):
	"""for a position in the linear sequence, return the filename"""
	iTunesSD.seek(headerSize + (trackNo * songSize) + titleAddress)
	return string16(iTunesSD.read(titleLength))

def getSongShuffle(iTunesSD, iTunesShuffle, shufflePos):
	"""for a position in the shuffle sequence, return the filename"""
	iTunesShuffle.seek(shufflePos * 3)
	trackNo = int24(iTunesShuffle.read(3))
	return getSong(iTunesSD, trackNo)

def lastNplayed(n):
	"""return a string with newline-separated filenames of the n songs last played"""
	pState = file(base+"iTunesPState", "r")
	pState.seek(shufflePosAddress)
	pStateData = pState.read(9)
	sPos = int24(pStateData[0:3])
	nPos = int24(pStateData[3:6])
	isShuffle = int24(pStateData[6:9])
	pState.close()

	sd = file(base+"iTunesSD", "r")
	if isShuffle:
		shuffle = file(base+"iTunesShuffle", "r")
		result = [getSongShuffle(sd, shuffle, no) for no in xrange(max(sPos-n, 0), sPos + 1)]
		shuffle.close()
	else:
		result = [getSong(sd, no) for no in xrange(max(nPos-n, 0), nPos + 1)]
	sd.close()
	result.reverse()

	return "\n".join(result)

_reSpaceSplit = re.compile(r"[\s\t]+")
def spacesSplit(line, *args):
	"""split a line by space or tabs"""
	return _reSpaceSplit.split(line, *args)

def findShuffle():
	"""return the base directory of a (mounted) iPod shuffle"""
	if os.path.exists(base): # called from shuffle root
		 return (base, lambda : None)
	
	if os.path.exists(mtab): # posix
		mtabfile = file(mtab, "r")
		mountpoint = ""
		for line in mtabfile:
			mountpoint = spacesSplit(line, 3)[1]
			if os.path.isdir(os.path.join(mountpoint, base)):
				break
			mountpoint = "" 
		mtabfile.close()
		if mountpoint: 
			return (mountpoint, lambda : None)
		
		raise Exception("Could not find any mount point for your iPod shuffle.")
	# will someone using a different system please enlighten me how to find a shuffle?
	try: # windows, yet untested
		import win32api
		mountpoint = ""
		for mountpoint in win32api.GetLogicalDriveStrings().split("\000"):
			if mountpoint in ("A:", "C:"): continue # disk and hard disk
			if os.path.isdir(os.path.join(mountpoint, base)):
				break
			mountpoint = ""
		if mountpoint:
			return (mountpoint, lambda : None)
		raise Exception("Sorry, I could not find your shuffle.")
	except: raise
	raise Exception("Sorry, I do not know how to find a shuffle on your system.\n"
		"Please run this script from your shuffle's root directory.")

if __name__ == '__main__':
	import sys
	# find a mounted iPod shuffle
	try: 
		(prebase, umount) = findShuffle()
		base = os.path.join(prebase, base)
	except Exception, e: 
		print e
		sys.exit(1)
	try: 
		n = int(sys.argv[1])
	except: 
		n = 5
	print lastNplayed(n)
	try:
		umount()
	except:
		pass
