/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *\
 * Kropki -- some classes for manipulating sgf                       *
 *                                                                   *
 * This file is based on sgftree.c, which is part                    *
 * of GNU Go, a Go program. Contact gnugo@gnu.org, or see            *
 * http://www.gnu.org/software/gnugo/ for more information.          *
 *                                                                   *
 * Copyright 1999, 2000, 2001, 2002, 2003 and 2004                   *
 * by the Free Software Foundation.                                  *
 * Copyright 2008 by Bartek Dyda <kropki@yahoo.co.uk>                *
 *                                                                   *
 * This program is free software; you can redistribute it and/or     *
 * modify it under the terms of the GNU General Public License as    *
 * published by the Free Software Foundation - version 2             *
 *                                                                   *
 * This program is distributed in the hope that it will be useful,   *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of    *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the     *
 * GNU General Public License in file COPYING for more details.      *
 *                                                                   *
 * You should have received a copy of the GNU General Public         *
 * License along with this program; if not, write to the Free        *
 * Software Foundation, Inc., 59 Temple Place - Suite 330,           *
 * Boston, MA 02111, USA.                                            *
\* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include <assert.h>
#include "sgftree.h"

void
SGFTree::sgftree_clear()
{
  root = NULL;
  lastnode = NULL;
}

int
SGFTree::sgftree_readfile(const char *infilename)
{
  SGFNode *savetree = root;
  SGFParser *parser = new SGFParser;
  root = parser->readsgffile(infilename);
  delete parser;
  if (root == NULL) {
    root = savetree;
    return 0;
  }
  
  delete savetree;
  lastnode = NULL;
  return 1;
}


/* Go back one node in the tree. If lastnode is NULL, go to the last
 * node (the one in main variant which has no children).
 */

int
SGFTree::sgftreeBack()
{
  if (lastnode) {
    if (lastnode->parent)
      lastnode = lastnode->parent;
    else
      return 0;
  }
  else
    while (sgftreeForward())
      ;
  
  return 1;
}


/* Go forward one node in the tree. If lastnode is NULL, go to the
 * tree root.
 */

int
SGFTree::sgftreeForward()
{
  if (lastnode) {
    if (lastnode->child)
      lastnode = lastnode->child;
    else
      return 0;
  }
  else
    lastnode = root;
  
  return 1;
}

SGFTree::SGFTree()
{
  root = lastnode = NULL;
}

SGFTree::~SGFTree()
{
  delete root;
  root = lastnode = NULL;
}

/* ================================================================ */
/*                        High level functions                      */
/* ================================================================ */

/*
 * Returns the node to modify. Use lastnode if available, otherwise
 * follow the main variation to the current end of the game.
 */

SGFNode *
SGFTree::sgftreeNodeCheck()
{
  SGFNode *node = NULL;
  assert(root);

  if (lastnode)
    node = lastnode;
  else {
    node = root;
    while (node->child)
      node = node->child;
  }

  return node;
}


/*
 * Add a stone to the current or the given node.
 * Return the node where the stone was added.
 */

void
SGFTree::sgftreeAddStone(int color, int movex, int movey)
{
  SGFNode *node = sgftreeNodeCheck();
  node->sgfAddStone(color, movex, movey);
}


/*
 * Add a move to the gametree.
 */

void
SGFTree::sgftreeAddPlay(int color, int movex, int movey)
{
  SGFNode *node = sgftreeNodeCheck();
  lastnode = node->sgfAddPlay(color, movex, movey);
}

void
SGFTree::sgftreeAddPlay(int color, char *move)
{
  SGFNode *node = sgftreeNodeCheck();
  lastnode = node->sgfAddPlay(color, move);
}


/*
 * Add a move to the gametree. New variations are added after the old
 * ones rather than before.
 */

void
SGFTree::sgftreeAddPlayLast(int color, int movex, int movey)
{
  SGFNode *node = sgftreeNodeCheck();
  lastnode = node->sgfAddPlayLast(color, movex, movey);
}


void
SGFTree::sgftreeCreateHeaderNode(int boardsizex, int boardsizey, int rules)
{
  SGFNode *nroot = new SGFNode;

  if (boardsizex!= boardsizey)      
    nroot->sgfAddPropertyInt("SZ", boardsizex, boardsizey);
  else
    nroot->sgfAddPropertyInt("SZ", boardsizex);
  nroot->sgfAddPropertyInt("RU", rules);
  root = nroot;
  lastnode = nroot;
}


/*
 * Add a comment to a gametree.
 */

void
SGFTree::sgftreeAddComment(const char *comment)
{
  SGFNode *node;
  assert(this && root);

  node = sgftreeNodeCheck();
  node->sgfAddComment(comment);
}


/*
 * Place text on the board at position (i, j).
 */

void
SGFTree::sgftreeBoardText(int i, int j, const char *text)
{
  SGFNode *node;
  assert(root);

  node = sgftreeNodeCheck();
  node->sgfBoardText(i, j, text);
}


/*
 * Place a character on the board at position (i, j).
 */

void
SGFTree::sgftreeBoardChar(int i, int j, char c)
{
  SGFNode *node;
  assert(root);

  node = sgftreeNodeCheck();
  node->sgfBoardChar(i, j, c);
}


/*
 * Place a number on the board at position (i, j).
 */

void
SGFTree::sgftreeBoardNumber(int i, int j, int number)
{
  SGFNode *node = sgftreeNodeCheck();
  node->sgfBoardNumber(i, j, number);
}


/*
 * Place a circle mark on the board at position (i, j).
 */

void
SGFTree::sgftreeTriangle(int i, int j)
{
  SGFNode *node = sgftreeNodeCheck();
  node->sgfTriangle(i, j);
}


/*
 * Place a circle mark on the board at position (i, j).
 */

void
SGFTree::sgftreeCircle(int i, int j)
{
  SGFNode *node = sgftreeNodeCheck();
  node->sgfCircle(i, j);
}


/*
 * Place a square mark on the board at position (i, j).
 */

void
SGFTree::sgftreeSquare(int i, int j)
{
  SGFNode *node = sgftreeNodeCheck();
  node->sgfSquare(i, j);
}


/*
 * Place a (square) mark on the board at position (i, j).
 */

void
SGFTree::sgftreeMark(int i, int j)
{
  SGFNode *node = sgftreeNodeCheck();
  node->sgfMark(i, j);
}


/*
 * Start a new variant.
 */

void
SGFTree::sgftreeStartVariant()
{
  SGFNode *node = sgftreeNodeCheck();
  lastnode = node->sgfStartVariant();
}


/*
 * Start a new variant as first child.
 */

void
SGFTree::sgftreeStartVariantFirst()
{
  SGFNode *node = sgftreeNodeCheck();
  lastnode = node->sgfStartVariantFirst();
}


/*
 * Write result of the game to the game tree.
 */

void
SGFTree::sgftreeWriteResult(float score, int overwrite)
{
  assert(root);
  root->sgfWriteResult(score, overwrite);
}


void
SGFTree::sgftreeSetLastNode(SGFNode *last_node)
{
  lastnode = last_node;
}
/* ================================================================ */
/*    New (kropki-specific) high level functions                    */
/* ================================================================ */

void
SGFTree::sgfPlayBack(int n)
{
  SGFNode *node = sgftreeNodeCheck();
  while (n && node) {
    if (node->is_move_node()) n--;
    node = node->parent;
  }
  lastnode = node;
}

void
SGFTree::sgftreeSaveNode()
{
  saved_node = sgftreeNodeCheck();
}

void
SGFTree::sgftreeRestoreNode()
{
  lastnode = saved_node;
}


/*
 * Local Variables:
 * tab-width: 8
 * c-basic-offset: 2
 * End:
 */

