/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *\
 * Kropki -- some classes for manipulating sgf                       *
 *                                                                   *
 * This file is based on sgfnode.c and sgf_utils.c, which are part   *
 * of GNU Go, a Go program. Contact gnugo@gnu.org, or see            *
 * http://www.gnu.org/software/gnugo/ for more information.          *
 *                                                                   *
 * Copyright 1999, 2000, 2001, 2002, 2003 and 2004                   *
 * by the Free Software Foundation.                                  *
 * Copyright 2008, 2012 by Bartek Dyda <kropki@yahoo.co.uk>          *
 *                                                                   *
 * This program is free software; you can redistribute it and/or     *
 * modify it under the terms of the GNU General Public License as    *
 * published by the Free Software Foundation - version 2             *
 *                                                                   *
 * This program is distributed in the hope that it will be useful,   *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of    *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the     *
 * GNU General Public License in file COPYING for more details.      *
 *                                                                   *
 * You should have received a copy of the GNU General Public         *
 * License along with this program; if not, write to the Free        *
 * Software Foundation, Inc., 59 Temple Place - Suite 330,           *
 * Boston, MA 02111, USA.                                            *
\* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/*  Parts of this code were given to us by Tommy Thorn */

#define VERSION "0084.12"

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <string.h>
#include <assert.h>


#if TIME_WITH_SYS_TIME
# include <sys/time.h>
# include <time.h>
#else
# if HAVE_SYS_TIME_H
#  include <sys/time.h>
# else
#  include <time.h>
# endif
#endif

#include "sgftree.h"

#define STRICT_SGF 's'
#define LAX_SGF    'l'

/* Set this to 1 if you want warnings for missing GM and FF properties. */
#define VERBOSE_WARNINGS 0

/* ================================================================ */
/*                     Some utility functions.                      */
/* ================================================================ */

/*
 * Utility: a checking, initializing malloc
 */

void *
xalloc(unsigned int size)
{
  void *pt = malloc(size);

  if (!pt) {
    fprintf(stderr, "xalloc: Out of memory!\n");
    exit(EXIT_FAILURE);
  }

  memset(pt, 0, (unsigned long) size);
  return pt;
}

void *
xrealloc(void *pt, unsigned int size)
{
  void *ptnew = realloc(pt, size);

  if (!ptnew) {
    fprintf(stderr, "xrealloc: Out of memory!\n");
    exit(EXIT_FAILURE);
  }
  return ptnew;
}

/* ================================================================ */
/*                 Some utility functions from sgf_utils.c          */
/* ================================================================ */

/* 
 * Return the integer X move.
 */

int
SGFProperty::get_moveX(int boardsize)
{
  int i;
  if (strlen(value) < 2)
    return -1;

  if (isupper(value[1]))
    i = value[1] - 'A' + 26;
  else if (islower(value[1]))
    i = value[1] - 'a';
  else return -1;
  return (i < boardsize) ? i : -1;
}

/* 
 * Return the integer Y move.
 */

int
SGFProperty::get_moveY(int boardsize)
{
  int j;
  if (strlen(value) < 2)
    return -1;
 
  if (isupper(value[0]))
    j = value[0] - 'A' + 26;
  else if (islower(value[0]))
    j = value[0] - 'a';
  else return -1;
  return (j < boardsize) ? j : -1;
}


/* Fills (*i, *j) from the property value, in GNU Go co-ords.
 * Note that GNU Go uses different conventions from sgf for
 * co-ordinates been called. 
 *
 * Returns 1 for a move, 0 for a pass.
 */

int
SGFProperty::get_moveXY(int *i, int *j, int boardsize)
{
  *i = get_moveX(boardsize);
  *j = get_moveY(boardsize);
  
  if (*i == -1 && *j == -1)
    return 0;

  return 1;
}


/* 
 * Debugging function to print properties as they are traversed.
 */

int
SGFNode::show_sgf_properties()
{
  SGFProperty *sgf_prop;
  int propcount;

  propcount = 0;

  printf("P: ");
  if (!props) {
    printf("None\n");
    return propcount;
  }
  else {
    sgf_prop = props;
    while (sgf_prop) {
      printf("%c%c ", sgf_prop->name & 0x00FF, (sgf_prop->name & 0xFF00)>>8);
      sgf_prop = sgf_prop->next;
      propcount++;
    }

    printf("(%d) ", propcount);
    if (next)
      printf("n");
    if (child)
      printf("c");
    printf("\n");
  }

  return propcount;
}


/*
 * Recursively traverse each node showing all properties.
 */

int
SGFNode::show_sgf_tree()
{
  int n = 0; /* number of nodes */
  
  n++;
  show_sgf_properties();

  /* must search depth first- siblings are equal! */
  if (child)
    n += child->show_sgf_tree();

  if (next)
    n += next->show_sgf_tree();
  
  return n;
}


/*
 * Determine if a node has a mark property in it.
 */

int
SGFNode::is_markup_node()
{
  SGFProperty *sgf_prop;
  
  /* If the node has no properties, there's nothing to do.
     This should have been checked by the caller, but it can't hurt. */
  if (!props)
    return 0;

  sgf_prop = props;
  while (sgf_prop) {
    switch (sgf_prop->name) {
    case SGFCR: 
    case SGFSQ: /* Square */
    case SGFTR: /* Triangle */
    case SGFMA: /* Mark */
    case SGFBM: /* bad move */
    case SGFDO: /* doubtful move */
    case SGFIT: /* interesting move */
    case SGFTE: /* good move */
      return 1;
      break;
    default:
      break;
    }
    sgf_prop = sgf_prop->next;
  }

  /* No markup property found. */
  return 0;
}


/*
 * Determine if the node has a move in it,
 * and if yes, returns BLACK or WHITE.
 */

int
SGFNode::is_move_node()
{
  SGFProperty *sgf_prop;
  
  /* If the node has no properties, there's nothing to do.
     This should have been checked by the caller, but it can't hurt. */
  if (!props)
    return 0;

  sgf_prop = props;
  while (sgf_prop) {
    switch (sgf_prop->name) {
    case SGFB: 
      return BLACK;
    case SGFW: 
      return WHITE;
      break;
    default:
      break;
    }
    sgf_prop = sgf_prop->next;
  }

  return 0;
}


/*
 * Determine if the node has a pass move in it.
 */

int
SGFNode::is_pass_node(int boardsize)
{
  SGFProperty *sgf_prop;
  int i, j;
  
  /* If the node has no properties, there's nothing to do.
     This should have been checked by the caller, but it can't hurt. */
  if (!props)
    return 0;

  sgf_prop = props;
  while (sgf_prop) {
    switch (sgf_prop->name) {
    case SGFB: 
    case SGFW: 
      return !sgf_prop->get_moveXY(&i, &j, boardsize);
      break;
    default:
      break;
    }
    sgf_prop = sgf_prop->next;
  }

  return 0;
}


/*
 * Determine whose move is in the node.
 */

int
SGFNode::find_move()
{
  SGFProperty *sgf_prop;
  
  /* If the node has no properties, there's nothing to do.
     This should have been checked by the caller, but it can't hurt. */
  if (!props)
    return 0;

  sgf_prop = props;
  while (sgf_prop) {
    switch (sgf_prop->name) {
    case SGFB: 
      return BLACK;
      break;
    case SGFW: 
      return WHITE;
      break;
    default:
      break;
    }
    sgf_prop = sgf_prop->next;
  }

  return EMPTY;
}


/* ================================================================ */
/*                           SGF Nodes                              */
/* ================================================================ */


/*
 * Constructor
 */

SGFNode::SGFNode()
{
  next = NULL;
  props = NULL;
  parent = NULL;
  child = NULL;
}

/*
 * Recursively free an sgf node
 */

SGFNode::~SGFNode()
{
  if (next) delete next;
  if (child) delete child;
  if (props) delete props;
}


/*
 * Add a generic text property to an SGF node.
 */

void
SGFNode::sgfAddProperty(const char *name, const char *value)
{
  SGFProperty *prop = props;

  if (prop)
    while (prop->next)
      prop = prop->next;

  sgfMkProperty(name, value, prop);
}

 
/*
 * Add an integer property to an SGF node.
 */

void
SGFNode::sgfAddPropertyInt(const char *name, long val)
{
  char buffer[10];

  snprintf(buffer, 10, "%ld", val);
  sgfAddProperty(name, buffer);
}

void
SGFNode::sgfAddPropertyInt(const char *name, long val, long val2)
{
  char buffer[20];

  snprintf(buffer, 20, "%ld:%ld", val, val2);
  sgfAddProperty(name, buffer);
}

/*
 * Add a float property to an SGF node.
 */

void
SGFNode::sgfAddPropertyFloat(const char *name, float val)
{
  char buffer[10];

  snprintf(buffer, 10, "%3.1f", val);
  sgfAddProperty(name, buffer);
}


/*
 * Read a property as int from an SGF node.
 */

int
SGFNode::sgfGetIntProperty(const char *name, int *value)
{
  SGFProperty *prop;
  short nam = name[0] | name[1] << 8;

  for (prop = props; prop; prop = prop->next)
    if (prop->name == nam) {
      *value = atoi(prop->value);
      return 1;
    }

  return 0;
}


/*
 * Read a property as float from an SGF node.
 */

int
SGFNode::sgfGetFloatProperty(const char *name, float *value)
{
  SGFProperty *prop;
  short nam = name[0] | name[1] << 8;

  for (prop = props; prop; prop = prop->next)
    if (prop->name == nam) {
      *value = (float) atof(prop->value);
      /* MS-C warns of loss of data (double to float) */
      return 1;
    }

  return 0;
}


/*
 * Read a property as text from an SGF node.
 */

int
SGFNode::sgfGetCharProperty(const char *name, char **value)
{
  SGFProperty *prop;
  short nam = name[0] | name[1] << 8;

  for (prop = props; prop; prop = prop->next)
    if (prop->name == nam) {
      *value = prop->value;
      return 1;
    }

  return 0;
}


/*
 * Is there a property of this type in the node?
 */

int
SGFNode::sgfHasProperty(const char *name)
{
  SGFProperty *prop;
  short nam = name[0] | name[1] << 8;

  for (prop = props; prop; prop = prop->next)
    if (prop->name == nam)
      return 1;

  return 0;
}


/*
 * Overwrite a property from an SGF node with text or create a new
 * one if it does not exist.
 */

void
SGFNode::sgfOverwriteProperty(const char *name, const char *text)
{
  SGFProperty *prop;
  short nam = name[0] | name[1] << 8;

  for (prop = props; prop; prop = prop->next)
    if (prop->name == nam) {
      prop->value = (char*) xrealloc(prop->value, strlen(text)+1);
      strcpy(prop->value, text);
      return;
    }

  sgfAddProperty(name, text);
}


/*
 * Overwrite an int property in an SGF node with val or create a new 
 * one if it does not exist.
 */

void
SGFNode::sgfOverwritePropertyInt(const char *name, int val)
{
  SGFProperty *prop;
  short nam = name[0] | name[1] << 8;

  for (prop = props; prop; prop = prop->next)
    if (prop->name == nam) {
      prop->value = (char*) xrealloc(prop->value, 12);
      snprintf(prop->value, 12, "%d", val);
      return;
   }

  sgfAddPropertyInt(name, val);
}


/* 
 * Overwrite a float property in the gametree with val or create
 * a new one if it does not exist.
 */

void
SGFNode::sgfOverwritePropertyFloat(const char *name, float val)
{
  SGFProperty *prop;
  short nam = name[0] | name[1] << 8;

  for (prop = props; prop; prop = prop->next)
    if (prop->name == nam) {
      prop->value = (char*) xrealloc(prop->value, 15);
      snprintf(prop->value, 15, "%3.1f", val);
      return;
    }

  sgfAddPropertyFloat(name, val);
}


/*
 * Goto previous node.
 */

SGFNode *
SGFNode::sgfPrev()
{
  SGFNode *q;
  SGFNode *prev;

  if (!parent)
    return NULL;

  q = parent->child;
  prev = NULL;
  while (q && q != this) {
    prev = q;
    q = q->next;
  }

  return prev;
}


/*
 * Goto root node.
 */

SGFNode *
SGFNode::sgfRoot()
{
  SGFNode *node = this;
  while (node->parent)
    node = node->parent;

  return node;
}


/* ================================================================ */
/*                         SGF Properties                           */
/* ================================================================ */


/*
 * Make an SGF property.
 */
SGFProperty *
SGFNode::do_sgf_make_property(short sgf_name,  const char *value,
		     SGFProperty *last)
{
  SGFProperty *prop = new SGFProperty;
  prop->name = sgf_name;
  prop->value = (char*) xalloc(strlen(value) + 1);
  strcpy(prop->value, value);
  prop->next = NULL;

  if (last == NULL)
    props = prop;
  else
    last->next = prop;

  return prop;
}


/* Make an SGF property.  In case of a property with a range it
 * expands it and makes several properties instead.
 */
SGFProperty *
SGFNode::sgfMkProperty(const char *name, const  char *value,
		       SGFProperty *last)
{
  static const short properties_allowing_ranges[12] = {
    /* Board setup properties. */
    SGFAB, SGFAW, SGFAE,

    /* Markup properties. */
    SGFCR, SGFMA, SGFSQ, SGFTR, SGFDD, SGFSL,

    /* Miscellaneous properties. */
    SGFVW,

    /* Go-specific properties. */
    SGFTB, SGFTW
  };

  int k;
  short sgf_name;

  if (strlen(name) == 1)
    sgf_name = name[0] | (short) (' ' << 8);
  else
    sgf_name = name[0] | name[1] << 8;

  for (k = 0; k < 12; k++) {
    if (properties_allowing_ranges[k] == sgf_name)
      break;
  }

  if (k < 12
      && strlen(value) == 5
      && value[2] == ':') {
    char x1 = value[0];
    char y1 = value[1];
    char x2 = value[3];
    char y2 = value[4];
    char new_value[] = "xy";

    if (x1 <= x2 && y1 <= y2) {
      for (new_value[0] = x1; new_value[0] <= x2; new_value[0]++) {
	for (new_value[1] = y1; new_value[1] <= y2; new_value[1]++)
	  last = do_sgf_make_property(sgf_name, new_value, last);
      }

      return last;
    }
  }

  /* Not a range property. */
  return do_sgf_make_property(sgf_name, value, last);
}


/*
 * Recursively free an SGF property.
 *
 */

SGFProperty::~SGFProperty()
{
  if (next) delete next;
}


/* ================================================================ */
/*                        High level functions                      */
/* ================================================================ */


/*
 * Add a stone to the current or the given node.
 * Return the node where the stone was added.
 */

SGFNode *
SGFNode::sgfAddStone(int color, int movex, int movey)
{
  char move[3];

  sprintf(move, "%c%c", movey + 'a', movex + 'a');
  sgfAddProperty((color == BLACK) ? "AB" : "AW", move);

  return this;
}


/*
 * Add a (simple) move to the gametree.
 */

SGFNode *
SGFNode::sgfAddPlay(int who, int movex, int movey)
{
  char move[3];
  SGFNode *newn;
  
  sprintf(move, "%c%c", movey + 'a', movex + 'a');

  if (child)
    newn = child->sgfStartVariantFirst();
  else {
    newn = new SGFNode();
    child = newn;
    newn->parent = this;
  }
  
  newn->sgfAddProperty((who == BLACK) ? "B" : "W", move);

  return newn;
}

SGFNode *
SGFNode::sgfAddPlay(int who, char *move)
{
  SGFNode *newn;
  
  if (child)
    newn = child->sgfStartVariantFirst();
  else {
    newn = new SGFNode();
    child = newn;
    newn->parent = this;
  }
  
  newn->sgfAddProperty((who == BLACK) ? "B" : "W", move);

  return newn;
}

/*
 * Add a move to the gametree. New variations are added after the old
 * ones rather than before.
 */

SGFNode *
SGFNode::sgfAddPlayLast(int who, int movex, int movey)
{
  char move[3];
  SGFNode *newn;

  sprintf(move, "%c%c", movey + 'a', movex + 'a');

  newn = sgfAddChild();
  newn->sgfAddProperty((who == BLACK) ? "B" : "W", move);

  return newn;
}


SGFNode *
//SGFNode::sgfCreateHeaderNode(int boardsize, float komi)
SGFNode::sgfAddHeaderProperties(int boardsizex, int boardsizey)
{
    if (boardsizex!=boardsizey)
      sgfAddPropertyInt("SZ", boardsizex, boardsizey);
    else sgfAddPropertyInt("SZ", boardsizex);

    return this;
}


/*
 * Add a comment to an SGF node.
 */

SGFNode *
SGFNode::sgfAddComment(const char *comment)
{
  sgfAddProperty("C ", comment);
  return this;
}


/*
 * Place text on the board at position (i, j).
 */

SGFNode *
SGFNode::sgfBoardText(int i, int j, const char *text)
{
  char *str = (char*) xalloc(strlen(text) + 3);

  sprintf(str, "%c%c:%s", j+'a', i+'a', text);
  sgfAddProperty("LB", str);
  free(str);

  return this;
}


/*
 * Place a character on the board at position (i, j).
 */

SGFNode *
SGFNode::sgfBoardChar(int i, int j, char c)
{
  char text[2] = "";

  text[0] = c;
  text[1] = 0;

  return sgfBoardText(i, j, text);
}


/*
 * Place a number on the board at position (i, j).
 */

SGFNode *
SGFNode::sgfBoardNumber(int i, int j, int number)
{
  char text[10];

  snprintf(text, 10, "%c%c:%i", j+'a', i+'a', number);
  sgfAddProperty("LB", text);

  return this;
}


/*
 * Place a triangle mark on the board at position (i, j).
 */

SGFNode *
SGFNode::sgfTriangle(int i, int j)
{
  char text[3];

  snprintf(text, 3, "%c%c", j+'a', i+'a');
  sgfAddProperty("TR", text);

  return this;
}


/*
 * Place a label on the board at position (i, j).
 */

SGFNode *
SGFNode::sgfLabel(const char *label, int i, int j)
{
  /* allows 12 chars labels - more than enough */
  char text[16];

  snprintf(text, 16, "%c%c:%s", j+'a', i+'a', label);
  sgfAddProperty("LB", text);

  return this;
}


/*
 * Place a numeric label on the board at position (i, j).
 */

SGFNode *
SGFNode::sgfLabelInt(int num, int i, int j)
{
  char text[16];

  snprintf(text, 16, "%c%c:%d", j+'a', i+'a', num);
  sgfAddProperty("LB", text);

  return this;
}


/*
 * Place a circle mark on the board at position (i, j).
 */

SGFNode *
SGFNode::sgfCircle(int i, int j)
{
  char text[3];

  snprintf(text, 3, "%c%c", j+'a', i+'a');
  sgfAddProperty("CR", text);

  return this;
}


/*
 * Place a square mark on the board at position (i, j).
 */

SGFNode *
SGFNode::sgfSquare(int i, int j)
{
  return sgfMark(i, j);   /* cgoban 1.9.5 does not understand SQ */
}


/*
 * Place a (square) mark on the board at position (i, j).
 */

SGFNode *
SGFNode::sgfMark(int i, int j)
{
  char text[3];

  snprintf(text, 3, "%c%c", j+'a', i+'a');
  sgfAddProperty("MA", text);

  return this;
}


/*
 * Start a new variant. Returns a pointer to the new node.
 */

SGFNode *
SGFNode::sgfStartVariant()
{
  assert(parent);

  SGFNode *node=this;

  while (node->next)
    node = node->next;
  node->next = new SGFNode();
  node->next->parent = node->parent;

  return node->next;
}


/*
 * Start a new variant as first child. Returns a pointer to the new node.
 */

SGFNode *
SGFNode::sgfStartVariantFirst()
{
  SGFNode *old_first_child = this;
  SGFNode *new_first_child = new SGFNode();

  assert(parent);

  new_first_child->next = old_first_child;
  new_first_child->parent = old_first_child->parent;

  new_first_child->parent->child = new_first_child;

  return new_first_child;
}


/*
 * If no child exists, add one. Otherwise add a sibling to the
 * existing children. Returns a pointer to the new node.
 */

SGFNode *
SGFNode::sgfAddChild()
{
  SGFNode *new_node = new SGFNode();

  new_node->parent = this;
  
  if (!child)
    child = new_node;
  else {
    SGFNode *node = child;
    while (node->next)
      node = node->next;
    node->next = new_node;
  }

  return new_node;
}


/*
 * Write result of the game to the game tree.
 */

void
SGFNode::sgfWriteResult(float score, int overwrite)
{
  char text[8];
  char winner;
  float s;
  int dummy;

  /* If not overwriting and there already is a result property, return. */
  if (!overwrite)
    if (sgfGetIntProperty("RE", &dummy))
      return;

  if (score > 0.0) {
    winner = 'W';
    s = score;
  }
  else if (score < 0.0) {
    winner = 'B';
    s = -score;
  }
  else {
    winner = '0';
    s = 0;
  }

  if (winner == '0')
    snprintf(text, 8, "0");
  else if (score < 10000.0 && score > -10000.0)
    snprintf(text, 8, "%c+%3.1f", winner, s);
  else
    snprintf(text, 8, "%c+%c", winner, 'R');
  sgfOverwriteProperty("RE", text);
}


void
SGFNode::sgf_write_header_reduced(int overwrite)
// this == root node
{
  time_t curtime = time(NULL);
  struct tm *loctime = localtime(&curtime);
  char str[128];
  int dummy;

  snprintf(str, 128, "%4.4i-%2.2i-%2.2i",
	   loctime->tm_year+1900, loctime->tm_mon+1, loctime->tm_mday);
  if (overwrite || !sgfGetIntProperty("DT", &dummy))
    sgfOverwriteProperty("DT", str);
  if (overwrite || !sgfGetIntProperty("AP", &dummy))
    sgfOverwriteProperty("AP", "Kropki:"VERSION);
  sgfOverwriteProperty("FF", "4");
}


void
SGFNode::sgf_write_header(int overwrite, int seed, float komi,
			  int level, int rules)
// this == root node
{
  char str[128];
  int dummy;

  snprintf(str, 128, "GNU Go %s Random Seed %d level %d", 
	   VERSION, seed, level);
  if (overwrite || !sgfGetIntProperty("GN", &dummy))
    sgfOverwriteProperty("GN", str);
  if (overwrite || !sgfGetIntProperty("RU", &dummy))
    sgfOverwriteProperty("RU", rules ? "Chinese" : "Japanese");

  sgf_write_header_reduced(overwrite);
}


int
SGFNode::sgfMovesBefore()
// returns number of moves before current node + in the current node
{
  SGFNode *node = this;
  int n = 0;
  while (node) {
    if (node->is_move_node()) n++;
    node = node->parent;
  }
  return n;
}


/* ================================================================ */
/*                          Read SGF tree                           */
/* ================================================================ */


#define MAX_FILE_BUFFER 200000 /* buffer for reading SGF file. */

/*
 * SGF grammar:
 *
 * Collection = GameTree { GameTree }
 * GameTree   = "(" Sequence { GameTree } ")"
 * Sequence   = Node { Node }
 * Node       = ";" { Property }
 * Property   = PropIdent PropValue { PropValue }
 * PropIdent  = UcLetter { UcLetter }
 * PropValue  = "[" CValueType "]"
 * CValueType = (ValueType | Compose)
 * ValueType  = (None | Number | Real | Double | Color | SimpleText |
 *               Text | Point  | Move | Stone)
 *
 * The above grammar has a number of simple properties which enables us
 * to write a simpler parser:
 *   1) There is never a need for backtracking
 *   2) The only recursion is on gametree.
 *   3) Tokens are only one character
 * 
 * We will use a global state to keep track of the remaining input
 * and a global char variable, `lookahead' to hold the next token.  
 * The function `nexttoken' skips whitespace and fills lookahead with 
 * the new token.
 */


//static void parse_error(const char *msg, int arg);
//static void nexttoken(void);
//static void match(int expected);


/* ---------------------------------------------------------------- */
/*                       Parsing primitives                         */
/* ---------------------------------------------------------------- */


void SGFParser::parse_error(const char *msg, int arg)
{
  fprintf(stderr, msg, arg);
  fprintf(stderr, "\n");
  exit(EXIT_FAILURE);
}


void
SGFParser::nexttoken()
{
  do
    lookahead = sgf_getch();
  while (isspace(lookahead));
}


void
SGFParser::match(int expected)
{
  if (lookahead != expected)
    parse_error("expected: %c", expected);
  else
    nexttoken();
}

/* ---------------------------------------------------------------- */
/*                        The parser proper                         */
/* ---------------------------------------------------------------- */


void
SGFParser::propident(char *buffer, int size)
{
  if (lookahead == EOF || !isupper(lookahead)) 
    parse_error("Expected an upper case letter.", 0);
  
  while (lookahead != EOF && isalpha(lookahead)) {
    if (isupper(lookahead) && size > 1) {
      *buffer++ = lookahead;
      size--;
    }
    nexttoken();
  }
  *buffer = '\0';
}


void
SGFParser::propvalue(char *buffer, int size)
{
  char *p = buffer;

  match('[');
  while (lookahead != ']' && lookahead != EOF) {
    if (lookahead == '\\') {
      lookahead = sgf_getch();
      /* Follow the FF4 definition of backslash */
      if (lookahead == '\r') {
	lookahead = sgf_getch();
	if (lookahead == '\n') 
	  lookahead = sgf_getch();
      }
      else if (lookahead == '\n') {
	lookahead = sgf_getch();
	if (lookahead == '\r') 
	  lookahead = sgf_getch();
      }
    }
    if (size > 1) {
      *p++ = lookahead;
      size--;
    }
    lookahead = sgf_getch();
  }
  match(']');
  
  /* Remove trailing whitespace */
  --p;
  while (p > buffer && isspace((int) *p))
    --p;
  *++p = '\0';
}


SGFProperty *
SGFParser::property(SGFNode *n, SGFProperty *last)
{
  char name[3];
  char buffer[4000];

  propident(name, sizeof(name));
  do {
    propvalue(buffer, sizeof(buffer));
    last = n->sgfMkProperty(name, buffer, last);
  } while (lookahead == '[');
  return last;
}


void
SGFParser::node(SGFNode *n)
{
  SGFProperty *last = NULL;
  match(';');
  while (lookahead != EOF && isupper(lookahead))
    last = property(n, last);
}


SGFNode *
SGFParser::sequence(SGFNode *n)
{
  node(n);
  while (lookahead == ';') {
    SGFNode *newn = new SGFNode();
    newn->parent = n;
    n->child = newn;
    n = newn;
    node(n);
  }
  return n;
}


void
SGFParser::gametree(SGFNode **p, SGFNode *parent, int mode) 
{
  if (mode == STRICT_SGF)
    match('(');
  else
    for (;;) {
      if (lookahead == EOF) {
	parse_error("Empty file?", 0);
	break;
      }
      if (lookahead == '(') {
	while (lookahead == '(')
	  nexttoken();
	if (lookahead == ';')
	  break;
      }
      nexttoken();
    }

  /* The head is parsed */
  {
    SGFNode *head = new SGFNode();
    SGFNode *last;

    head->parent = parent;
    *p = head;

    last = sequence(head);
    p = &last->child;
    while (lookahead == '(') {
      gametree(p, last, STRICT_SGF);
      p = &((*p)->next);
    }
    if (mode == STRICT_SGF)
      match(')');
  }
}



/*
 * Wrapper around readsgf which reads from a file rather than a string.
 * Returns NULL if file will not open, or some other parsing error.
 */

SGFNode *
SGFParser::readsgffile(const char *filename)
{
  SGFNode *root;
  int tmpi = 0;

  if (strcmp(filename, "-") == 0)
    sgffile = stdin;
  else
    sgffile = fopen(filename, "r");

  if (!sgffile)
    return NULL;


  nexttoken();
  gametree(&root, NULL, LAX_SGF);

  fclose(sgffile);

  if (sgferr) {
    fprintf(stderr, "Parse error: %s at position %d\n", sgferr, sgferrpos);
    return NULL;
  }

  /* perform some simple checks on the file */
  if (!root->sgfGetIntProperty("GM", &tmpi)) {
    if (VERBOSE_WARNINGS)
      fprintf(stderr, "Couldn't find the game type (GM) attribute!\n");
  }
  else if (tmpi != 40) {
    fprintf(stderr, "SGF file might be for game other than Kropki: %d\n", tmpi);
    fprintf(stderr, "Trying to load anyway.\n");
  }

  if (!root->sgfGetIntProperty("FF", &tmpi)) {
    if (VERBOSE_WARNINGS)
      fprintf(stderr, "Can not determine SGF spec version (FF)!\n");
  }
  else if ((tmpi < 3 || tmpi > 4) && VERBOSE_WARNINGS)
    fprintf(stderr, "Unsupported SGF spec version: %d\n", tmpi);

  return root;
}


/*
 * Reads sgf from a given string.
 * Returns NULL in case of some parsing error.
 */

SGFNode *
SGFParser::readsgfstring(const char *str)
{
  SGFNode *root;
  int tmpi = 0;

  if (str == NULL) return NULL;
  int n = strlen(str);
  if (n<3) return NULL;
  sgfstring = new char[n+1];
  strncpy(sgfstring, str, n+1);
  sgffile = NULL;
  sgfind = 0;


  nexttoken();
  gametree(&root, NULL, LAX_SGF);

  if (sgferr) {
    fprintf(stderr, "Parse error: %s at position %d\n", sgferr, sgferrpos);
    return NULL;
  }

  /* perform some simple checks on the file */
  if (!root->sgfGetIntProperty("GM", &tmpi)) {
    if (VERBOSE_WARNINGS)
      fprintf(stderr, "Couldn't find the game type (GM) attribute!\n");
  }
  else if (tmpi != 40) {
    fprintf(stderr, "SGF file might be for game other than Kropki: %d\n", tmpi);
    fprintf(stderr, "Trying to load anyway.\n");
  }

  if (!root->sgfGetIntProperty("FF", &tmpi)) {
    if (VERBOSE_WARNINGS)
      fprintf(stderr, "Can not determine SGF spec version (FF)!\n");
  }
  else if ((tmpi < 3 || tmpi > 4) && VERBOSE_WARNINGS)
    fprintf(stderr, "Unsupported SGF spec version: %d\n", tmpi);

  return root;
}



/* ================================================================ */
/*                          Write SGF tree                          */
/* ================================================================ */


#define OPTION_STRICT_FF4 0

static int sgf_column = 0;

static void
sgf_putc(int c, FILE *file)
{
  if (c == '\n' && sgf_column == 0)
    return;

  fputc(c, file);

  if (c == '\n')
    sgf_column = 0;
  else
    sgf_column++;

  if (c == ']' && sgf_column > 60) {
    fputc('\n', file);
    sgf_column = 0;
  }
}

static void
sgf_puts(const char *s, FILE *file)
{
  for (; *s; s++) {
    if (*s == '[' || *s == ']' || *s == '\\') {
      fputc('\\', file);
      sgf_column++;
    }
    fputc((int) *s, file);
    sgf_column++;
  }
}

/* Print all properties with the given name in a node to file and mark
 * them as printed.
 *
 * If is_comment is 1, multiple properties are concatenated with a
 * newline. I.e. we write
 *
 * C[comment1
 * comment2]
 *
 * instead of
 *
 * C[comment1][comment2]
 *
 * Most other property types should be written in the latter style.
 */

static void
sgf_print_name(FILE *file, short name)
{
  sgf_putc(name & 0xff, file);
  if (name >> 8 != ' ')
    sgf_putc(name >> 8, file);
}

static void
sgf_print_property(FILE *file, SGFNode *node, short name, int is_comment)
{
  int n = 0;
  SGFProperty *prop;

  for (prop = node->props; prop; prop = prop->next) {
    if (prop->name == name) {
      prop->name |= 0x20;  /* Indicate already printed. */
      if (n == 0) {
	sgf_print_name(file, name);
	sgf_putc('[', file);
      }
      else if (is_comment)
	sgf_putc('\n', file);
      else {
	sgf_putc(']', file);
	sgf_putc('[', file);
      }
      
      sgf_puts(prop->value, file);
      n++;
    }
  }

  if (n > 0)
    sgf_putc(']', file);

  /* Add a newline after certain properties. */
  if (name == SGFAB || name == SGFAW || name == SGFAE || (is_comment && n > 1))
    sgf_putc('\n', file);
}

/*
 * Print all remaining unprinted property values at node N to file.
 */

static void
sgfPrintRemainingProperties(FILE *file, SGFNode *node)
{
  SGFProperty *prop;

  for (prop = node->props; prop; prop = prop->next)
    if (!(prop->name & 0x20))
      sgf_print_property(file, node, prop->name, 0);
}


/*
 * Print the property values of NAME at node N and mark it as printed. 
 */

static void
sgfPrintCharProperty(FILE *file, SGFNode *node, const char *name)
{
  short nam = name[0] | name[1] << 8;
  
  sgf_print_property(file, node, nam, 0);
}


/*
 * Print comments from Node node.
 *
 * NOTE: cgoban does not print "C[comment1][comment2]" and I don't know
 *       what the sgfspec says.
 */

static void
sgfPrintCommentProperty(FILE *file, SGFNode *node, const char *name)
{
  short nam = name[0] | name[1] << 8;
  
  sgf_print_property(file, node, nam, 1);
}


static void
unparse_node(FILE *file, SGFNode *node)
{
  sgf_putc(';', file);
  sgfPrintCharProperty(file, node, "B ");
  sgfPrintCharProperty(file, node, "W ");
  sgfPrintCommentProperty(file, node, "N ");
  sgfPrintCommentProperty(file, node, "C ");
  sgfPrintRemainingProperties(file, node);
}


static void
unparse_root(FILE *file, SGFNode *node)
{
  sgf_putc(';', file);
  
  if (node->sgfHasProperty("GM"))
    sgfPrintCharProperty(file, node, "GM");
  else {
    fputs("GM[40]", file);
    sgf_column += 5;
  }
  
  sgfPrintCharProperty(file, node, "FF");
  sgf_putc('\n', file);

  sgfPrintCharProperty(file, node, "SZ");
  sgf_putc('\n', file);
  
  sgfPrintCharProperty(file, node, "GN");
  sgf_putc('\n', file);
  
  sgfPrintCharProperty(file, node, "DT");
  sgf_putc('\n', file);
  
  sgfPrintCommentProperty(file, node, "PB");
  sgfPrintCommentProperty(file, node, "BR");
  sgf_putc('\n', file);
  
  sgfPrintCommentProperty(file, node, "PW");
  sgfPrintCommentProperty(file, node, "WR");
  sgf_putc('\n', file);
  
  sgfPrintCommentProperty(file, node, "N ");
  sgfPrintCommentProperty(file, node, "C ");
  sgfPrintRemainingProperties(file, node);

  sgf_putc('\n', file);
}


/*
 * p->child is the next move.
 * p->next  is the next variation
 */

static void
unparse_game(FILE *file, SGFNode *node, int root)
{
  if (!root)
    sgf_putc('\n', file);
  sgf_putc('(', file);
  if (root)
    unparse_root(file, node);
  else
    unparse_node(file, node);

  node = node->child;
  while (node != NULL && node->next == NULL) {
    unparse_node(file, node);
    node = node->child;
  } 

  while (node != NULL) {
    unparse_game(file, node, 0);
    node = node->next;
  }
  sgf_putc(')', file);
  if (root)
    sgf_putc('\n', file);
}

/* Printed properties are marked by adding the 0x20 bit to the
 * property name (changing an upper case letter to lower case). This
 * function removes this mark so that we can print the property next
 * time too. It recurses to all properties in the linked list.
 */
static void
restore_property(SGFProperty *prop)
{
  if (prop) {
    restore_property(prop->next);
    prop->name &= ~0x20;
  }
}

/* When called with the tree root, recurses to all properties in the
 * tree and removes all print marks.
 */
static void
restore_node(SGFNode *node)
{
  if (node) {
    restore_property(node->props);
    restore_node(node->child);
    restore_node(node->next);
  }
}


/*
 * Opens filename and writes the game stored in the sgf structure.
 */

int
writesgf(SGFNode *root, const char *filename)
{
  FILE *outfile;

  if (strcmp(filename, "-") == 0) 
    outfile = stdout;
  else
    outfile = fopen(filename, "w");

  if (!outfile) {
    fprintf(stderr, "Can not open %s\n", filename);
    return 0;
  }

  root->sgf_write_header_reduced(0);

  sgf_column = 0;
  unparse_game(outfile, root, 1);

  if (strcmp(filename, "-") != 0) 
    fclose(outfile);
  
  /* Remove "printed" marks so that the tree can be written multiple
   * times.
   */
  restore_node(root);
  
  return 1;
}


#ifdef TEST_SGFPARSER
int
main()
{
  static char buffer[25000];
  static char output[25000];
  SGFNode *game;

  sgffile = stdin;

  nexttoken();
  gametree(&game, LAX_SGF);
  if (sgferr) {
    fprintf(stderr, "Parse error:");
    fprintf(stderr, sgferr, sgferrarg);
    fprintf(stderr, " at position %d\n", sgferrpos);
  }
  else {
    unparse_game(stdin, game, 1);
    write(1, output, outputp - output);
  }
}
#endif


/*
int main(int argc, char * argv[])
{
  // test!
  if (argc<=1) return 0;
  SGFParser parser;
  SGFNode *gra = parser.readsgffile(argv[1]);

  gra->show_sgf_tree();


  unparse_game(stdout, gra->sgfRoot(), 1);


  return 1;
}
*/

/*
 * Local Variables:
 * tab-width: 8
 * c-basic-offset: 2
 * End:
 */
