/*
 * $Id: searchinfo.h
 *
 * Copyright (C) 2003 Petter E. Stokke <gibreel@gibreel.net>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * $Log: searchquery.h,v $
 * Revision 1.2  2003/06/13 18:20:02  gibreel
 * Libkmldonkey now uses references instead of pointers everywhere except where
 * it would cause an obvious performance impact, which should lead to less
 * chance of memory leaks and cleaner code in general. Almost everything that
 * should be const is now also const.
 *
 * Revision 1.1  2003/05/24 22:52:04  gibreel
 * Created some classes for constructing search queries in a more flexible
 * manner, and updated the code accordingly where necessary.
 *
 */

#ifndef __libkmldonkey_searchquery_h__
#define __libkmldonkey_searchquery_h__

#include <qstring.h>
#include <qptrlist.h>
#include "donkeymessage.h"


//! Base class for query operations.
class SearchQuery
{
public:
    enum Operation {
        And = 0,
        Or,
        AndNot,
        Module,
        Keywords,
        MinSize,
        MaxSize,
        Format,
        Media,
        Mp3Artist,
        Mp3Title, // 10
        Mp3Album,
        Mp3Bitrate,
        Hidden
    };

    SearchQuery(Operation op);
    virtual ~SearchQuery();

    Operation operation();
    void setOperation(Operation op);

    //! Write this query into a DonkeyMessage.
    /*! \param msg The message to write to. */
    virtual void writeQuery(DonkeyMessage& msg);

private:
    Operation Op;
};



//! A base class for queries containing a list of other queries.
class SearchQueryList : public SearchQuery
{
public:
    SearchQueryList(Operation op);

    //! Append a query to the list.
    /*! Queries appended to the query list are owned by the list, and will be freed along with it.
     *  \param q The query to append.
     *  \return A pointer to the list. Useful for code like <tt>query->append(foo)->append(bar)->append(baz)</tt>.
     */
    SearchQueryList* append(SearchQuery* q);

    //! Returns the size of the attached list of queries.
    /*! \return The number of queries in this query list. */
    uint count();
    //! Returns a pointer to the query at a given index in the list.
    /*! \param index The index of the query to retrieve.
     *  \return Pointer to the query at the given index.
     */
    SearchQuery* at(uint index);
    //! Detaches a query from the list.
    /*! This method removes a query from the query list and returns a pointer to it.
     *  When you remove a query, you are responsible for freeing it. The list will no longer take care of this.
     *  \param index The index of the query to detach.
     *  \return The detached query.
     */
    SearchQuery* take(uint index);

    //! Write this query (with its attached queries) into a DonkeyMessage.
    /*! \param msg The message to write to. */
    virtual void writeQuery(DonkeyMessage& msg);

private:
    QPtrList<SearchQuery> queryList;
};



//! A query containing a list of subqueries that must all match.
class QueryAnd : public SearchQueryList
{
public:
    QueryAnd();
};

//! A query containing a list of subqueries of which at least one must match.
class QueryOr : public SearchQueryList
{
public:
    QueryOr();
};

class QueryHidden : public SearchQueryList
{
public:
    QueryHidden();
};



class QueryAndNot : public SearchQuery
{
public:
    QueryAndNot(SearchQuery* q1, SearchQuery* q2);
    ~QueryAndNot();

    virtual void writeQuery(DonkeyMessage& msg);

private:
    SearchQuery *query1, *query2;
};



class QueryModule : public SearchQuery
{
public:
    QueryModule(QString str, SearchQuery* q);
    ~QueryModule();

    virtual void writeQuery(DonkeyMessage& msg);

private:
    QString string;
    SearchQuery* query;
};



class SearchQueryTwoStrings : public SearchQuery
{
public:
    SearchQueryTwoStrings(Operation op, QString str1, QString str2);

    virtual void writeQuery(DonkeyMessage& msg);

private:
    QString s1, s2;
};

//! A query that specifies keywords to match.
class QueryKeywords : public SearchQueryTwoStrings
{
public:
    QueryKeywords(QString str1, QString str2);
};

//! A query that defines a minimum file size that must be matched.
class QueryMinSize : public SearchQueryTwoStrings
{
public:
    QueryMinSize(QString str1, QString str2);
};

//! A query that defines a maximum file size that must be matched.
class QueryMaxSize : public SearchQueryTwoStrings
{
public:
    QueryMaxSize(QString str1, QString str2);
};

//! A query that specifies a file format to match.
class QueryFormat : public SearchQueryTwoStrings
{
public:
    QueryFormat(QString str1, QString str2);
};

//! A query that specifies a media type to match.
class QueryMedia : public SearchQueryTwoStrings
{
public:
    QueryMedia(QString str1, QString str2);
};

//! A query that specifies an ID3 artist tag for MP3 files.
class QueryMp3Artist : public SearchQueryTwoStrings
{
public:
    QueryMp3Artist(QString str1, QString str2);
};

//! A query that specifies an ID3 title tag for MP3 files.
class QueryMp3Title : public SearchQueryTwoStrings
{
public:
    QueryMp3Title(QString str1, QString str2);
};

//! A query that specifies an ID3 album tag for MP3 files.
class QueryMp3Album : public SearchQueryTwoStrings
{
public:
    QueryMp3Album(QString str1, QString str2);
};

//! A query that specifies a given MP3 bitrate to match.
class QueryMp3Bitrate : public SearchQueryTwoStrings
{
public:
    QueryMp3Bitrate(QString str1, QString str2);
};

#endif
