/*
 * $Id: searchinfo.cpp
 *
 * Copyright (C) 2003 Petter E. Stokke <gibreel@gibreel.net>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * $Log: searchquery.cpp,v $
 * Revision 1.2  2003/06/13 18:20:02  gibreel
 * Libkmldonkey now uses references instead of pointers everywhere except where
 * it would cause an obvious performance impact, which should lead to less
 * chance of memory leaks and cleaner code in general. Almost everything that
 * should be const is now also const.
 *
 * Revision 1.1  2003/05/24 22:52:04  gibreel
 * Created some classes for constructing search queries in a more flexible
 * manner, and updated the code accordingly where necessary.
 *
 */

#include "searchquery.h"


SearchQuery::SearchQuery(Operation op)
{
    setOperation(op);
}

SearchQuery::~SearchQuery()
{
}

SearchQuery::Operation SearchQuery::operation()
{
    return Op;
}

void SearchQuery::setOperation(Operation op)
{
    Op = op;
}

void SearchQuery::writeQuery(DonkeyMessage& msg)
{
    msg.writeInt8((int8)Op);
}



SearchQueryList::SearchQueryList(Operation op)
    : SearchQuery(op)
{
    queryList.setAutoDelete(true);
}

SearchQueryList* SearchQueryList::append(SearchQuery* q)
{
    queryList.append(q);
    return this;
}

uint SearchQueryList::count()
{
    return queryList.count();
}

SearchQuery* SearchQueryList::at(uint index)
{
    return queryList.at(index);
}

SearchQuery* SearchQueryList::take(uint index)
{
    return queryList.take(index);
}

void SearchQueryList::writeQuery(DonkeyMessage& msg)
{
    SearchQuery::writeQuery(msg);
    msg.writeInt16((int16)queryList.count());
    QPtrListIterator<SearchQuery> it(queryList);
    SearchQuery* q;
    while ((q = it.current()) != 0) {
        ++it;
        q->writeQuery(msg);
    }
}



QueryAnd::QueryAnd()
    : SearchQueryList(SearchQuery::And)
{
}

QueryOr::QueryOr()
    : SearchQueryList(SearchQuery::Or)
{
}

QueryHidden::QueryHidden()
    : SearchQueryList(SearchQuery::Hidden)
{
}



QueryAndNot::QueryAndNot(SearchQuery* q1, SearchQuery* q2)
    : SearchQuery(SearchQuery::AndNot)
{
    query1 = q1; query2 = q2;
}

QueryAndNot::~QueryAndNot()
{
    delete query1; delete query2;
}

void QueryAndNot::writeQuery(DonkeyMessage& msg)
{
    SearchQuery::writeQuery(msg);
    query1->writeQuery(msg);
    query2->writeQuery(msg);
}



QueryModule::QueryModule(QString str, SearchQuery* q)
    : SearchQuery(SearchQuery::Module)
    , string(str)
{
    query = q;
}

QueryModule::~QueryModule()
{
    delete query;
}

void QueryModule::writeQuery(DonkeyMessage& msg)
{
    SearchQuery::writeQuery(msg);
    msg.writeString(string);
    query->writeQuery(msg);
}



SearchQueryTwoStrings::SearchQueryTwoStrings(Operation op, QString str1, QString str2)
    : SearchQuery(op)
    , s1(str1)
    , s2(str2)
{
}

void SearchQueryTwoStrings::writeQuery(DonkeyMessage& msg)
{
    SearchQuery::writeQuery(msg);
    msg.writeString(s1);
    msg.writeString(s2);
}



QueryKeywords::QueryKeywords(QString s1, QString s2)
    : SearchQueryTwoStrings(SearchQuery::Keywords, s1, s2)
{
}



QueryMinSize::QueryMinSize(QString s1, QString s2)
    : SearchQueryTwoStrings(SearchQuery::MinSize, s1, s2)
{
}



QueryMaxSize::QueryMaxSize(QString s1, QString s2)
    : SearchQueryTwoStrings(SearchQuery::MaxSize, s1, s2)
{
}



QueryFormat::QueryFormat(QString s1, QString s2)
    : SearchQueryTwoStrings(SearchQuery::Format, s1, s2)
{
}



QueryMedia::QueryMedia(QString s1, QString s2)
    : SearchQueryTwoStrings(SearchQuery::Media, s1, s2)
{
}



QueryMp3Artist::QueryMp3Artist(QString s1, QString s2)
    : SearchQueryTwoStrings(SearchQuery::Mp3Artist, s1, s2)
{
}



QueryMp3Title::QueryMp3Title(QString s1, QString s2)
    : SearchQueryTwoStrings(SearchQuery::Mp3Title, s1, s2)
{
}



QueryMp3Album::QueryMp3Album(QString s1, QString s2)
    : SearchQueryTwoStrings(SearchQuery::Mp3Album, s1, s2)
{
}



QueryMp3Bitrate::QueryMp3Bitrate(QString s1, QString s2)
    : SearchQueryTwoStrings(SearchQuery::Mp3Bitrate, s1, s2)
{
}



