/* -*- c++ -*-
 *
 * $Id: hostmanager.cpp,v 1.2 2003/07/22 21:16:32 gibreel Exp $
 *
 * Copyright (C) 2003 Petter E. Stokke <gibreel@gibreel.net>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * $Log: hostmanager.cpp,v $
 * Revision 1.2  2003/07/22 21:16:32  gibreel
 * Typo fix.
 *
 * Revision 1.1  2003/07/22 21:01:55  gibreel
 * Added the HostManager class for dealing with mldonkeyrc (MLDonkey connection
 * definitions as written by kcmdonkey).
 *
 */

#include <kconfig.h>
#include <kstandarddirs.h>
#include <kdirwatch.h>

#include "hostmanager.h"
#include "hostmanager.moc"

DonkeyHost::DonkeyHost(const QString& h_name, const QString& h_address,
		       int h_telnetPort, int h_guiPort,
		       const QString& h_username, const QString& h_password)
    : m_name(h_name)
    , m_address(h_address)
    , m_username(h_username)
    , m_password(h_password)
    , m_telnetPort(h_telnetPort)
    , m_guiPort(h_guiPort)
{
}

DonkeyHost::DonkeyHost(const DonkeyHost& host)
{
    m_name = host.name();
    m_address = host.address();
    m_telnetPort = host.telnetPort();
    m_guiPort = host.guiPort();
    m_username = host.username();
    m_password = host.password();
}

DonkeyHost::DonkeyHost()
{
    m_telnetPort = m_guiPort = 0;
}

const QString& DonkeyHost::name() const { return m_name; }
const QString& DonkeyHost::address() const { return m_address; }
const int DonkeyHost::telnetPort() const { return m_telnetPort; }
const int DonkeyHost::guiPort() const { return m_guiPort; }
const QString& DonkeyHost::username() const { return m_username; }
const QString& DonkeyHost::password() const { return m_password; }



HostManager::HostManager(QObject* parent, const char* name)
    : QObject(parent, name)
{
    refreshHostList();

    configWatcher = new KDirWatch(this);
    configWatcher->addFile(locateLocal("config", "mldonkeyrc"));

    connect(configWatcher, SIGNAL(dirty(const QString&)), SLOT(fileChanged(const QString&)));
    connect(configWatcher, SIGNAL(created(const QString&)), SLOT(fileChanged(const QString&)));
    connect(configWatcher, SIGNAL(deleted(const QString&)), SLOT(fileChanged(const QString&)));
}

void HostManager::refreshHostList()
{
    m_hosts.clear();
    m_default = QString::null;

    KConfig *config = new KConfig("mldonkeyrc", false, false);

    QStringList list = config->groupList();
    QStringList::iterator it;
    for (it = list.begin(); it != list.end(); ++it) {
	config->setGroup(*it);
	DonkeyHost entry(*it,
			 config->readEntry("DonkeyHost", "localhost"),
			 config->readNumEntry("DonkeyPort", 4000),
			 config->readNumEntry("DonkeyGuiPort", 4001),
			 config->readEntry("DonkeyUsername", "admin"),
			 config->readEntry("DonkeyPassword", ""));
	if (config->readBoolEntry("Default") && m_default.isNull())
	    m_default = entry.name();
	m_hosts.replace(entry.name(), entry);
    }
    if (m_hosts.empty()) {
	DonkeyHost entry("MLDonkey", "localhost", 4000, 4001, "admin", "");
	m_default = entry.name();
	m_hosts.replace(entry.name(), entry);
    }
    if (m_default.isNull()) {
	QMap<QString,DonkeyHost>::Iterator it = m_hosts.begin();
	m_default = it.key();
    }
}

void HostManager::fileChanged(const QString&)
{
    refreshHostList();
    emit hostListUpdated();
}

QStringList HostManager::hostList()
{
    return m_hosts.keys();
}

const QString& HostManager::defaultHostName()
{
    return m_default;
}

const DonkeyHost& HostManager::defaultHost()
{
    return m_hosts[m_default];
}

const DonkeyHost& HostManager::hostProperties(const QString& hostName)
{
    return m_hosts[hostName];
}

bool HostManager::validHostName(const QString& hostName)
{
    return m_hosts.contains(hostName);
}
