/* -*- c++ -*-
 *
 * $Id: donkeyprotocol.h,v 1.29 2003/07/25 15:42:53 dipesh Exp $
 *
 * Copyright (C) 2003 Petter E. Stokke <gibreel@gibreel.net>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * $Log: donkeyprotocol.h,v $
 * Revision 1.29  2003/07/25 15:42:53  dipesh
 * Added the enableNetwork() donkey-function and changed "Configure MLDonkey" to
 * handle enable/disable Networks another way. The way used previous was working
 * as well, but the core didn't send something back to show that that a Network
 * was enabled/disabled. So, to be able to work at the search/network combo always
 * with the actual enabled networks the new function is used now. Stupid
 * core-behaviour anyway.
 *
 * Revision 1.28  2003/07/23 18:06:12  gibreel
 * Added API doc generation using the KDE doc generation build system. Briefly
 * documented as many classes as necessary to produce useful documentation.
 *
 * Revision 1.27  2003/07/22 21:01:55  gibreel
 * Added the HostManager class for dealing with mldonkeyrc (MLDonkey connection
 * definitions as written by kcmdonkey).
 *
 * Revision 1.26  2003/07/20 20:45:04  dipesh
 * Added "Configure MLDonkey" :)
 *
 * Revision 1.25  2003/07/18 17:08:54  gibreel
 * Fixed a bug where DonkeyProtocol's tables weren't cleared on a manual
 * disconnect, and the friend table wasn't ever cleared. Replaced remaining
 * qDebug() calls with kdDebug().
 *
 * Revision 1.24  2003/06/30 23:30:35  gibreel
 * Preliminary friend list support. A ton of updates to the libkmldonkey API to
 * accommodate this, most notably improvements to the search handling
 * necessitated by mldonkey's somewhat awkward reporting of friend shares.
 *
 * Revision 1.23  2003/06/30 14:53:42  gibreel
 * Still hunting bugs in source management: implemented a lot of missing
 * message handlers pertaining to cleanup of client records. Quite a few bug
 * fixes. Protocol object now emits *Removed() signals corresponding to
 * *Updated() signals instead of sending update signals for removed records and
 * leaving the user to determine whether to remove the records.
 *
 * Revision 1.22  2003/06/29 13:39:28  gibreel
 * Tried to optimise the particularly memory hungry file source lists.
 *
 * Revision 1.21  2003/06/28 21:17:00  gibreel
 * The file info dialog now displays file sources. Libkmldonkey received a
 * number of updates and bugfixes to facilitate this.
 *
 * Revision 1.20  2003/06/27 12:43:08  dipesh
 * Added isConnected();
 *
 * Revision 1.19  2003/06/19 21:47:45  gibreel
 * DonkeyProtocol now keeps track of the amount of connected servers, and
 * provides methods for querying this and the total server count.
 *
 * Revision 1.18  2003/06/13 18:20:02  gibreel
 * Libkmldonkey now uses references instead of pointers everywhere except where
 * it would cause an obvious performance impact, which should lead to less
 * chance of memory leaks and cleaner code in general. Almost everything that
 * should be const is now also const.
 *
 * Revision 1.17  2003/06/09 21:08:50  gibreel
 * Added the setOption method for configuring mldonkey.
 *
 * Revision 1.16  2003/05/29 23:47:55  gibreel
 * Added shared file list refresh action. For some reason, though, the core
 * seems to only want to respond to the refresh request once...
 *
 * Revision 1.15  2003/05/24 22:52:04  gibreel
 * Created some classes for constructing search queries in a more flexible
 * manner, and updated the code accordingly where necessary.
 *
 * Revision 1.14  2003/05/21 00:07:42  gibreel
 * Updated libkmldonkey to support protocol 16. Added the addServer method.
 *
 * Revision 1.13  2003/05/18 17:58:22  dipesh
 * Added KAction's Download, ForceDownload and RemoveServers
 *
 * Revision 1.12  2003/05/17 10:52:24  dipesh
 * Added more search-options
 *
 * Revision 1.11  2003/05/12 00:20:36  gibreel
 * Added a method for forgetting searches.
 *
 * Revision 1.10  2003/05/11 08:33:01  dipesh
 * Added initial Search-functionality.
 *
 * Revision 1.9  2003/04/06 21:29:14  gibreel
 * Added some methods for controlling server lists.
 *
 * Revision 1.8  2003/03/23 23:34:59  gibreel
 * A lot of API additions, especially lists keeping track of shared files and
 * clients.
 *
 * Revision 1.7  2003/03/10 14:52:02  gibreel
 * Support for GUI protocol 14. Specifically: Authentication with username, new
 * download file state "Queued", and support for the new message types in the
 * DonkeyProtocol class.
 *
 * Revision 1.6  2003/03/08 22:37:58  gibreel
 * Written some Doxygen documentation for the library API.
 *
 * Revision 1.5  2003/03/08 20:13:20  gibreel
 * Switched readString/writeString in DonkeyMessage from using pointers to
 * QStrings to using references, which leads to far less hassle, shorter code,
 * and less chance of memory leaks.
 *
 * Revision 1.4  2003/03/08 17:08:12  gibreel
 * Lots of minor API changes.
 *
 * Revision 1.3  2003/03/07 21:51:18  gibreel
 * Changed the MD4 storage in the FileInfo object from an array of int to a
 * QByteArray, added conversion functions between that and QString, and methods
 * to search downloaded files by hash. The applet now uses the file hash
 * instead of the fileno when storing and restoring priorities, as the fileno
 * changes between core restarts.
 *
 * Revision 1.2  2003/03/07 20:35:38  gibreel
 * Added submitURL() method and option to disable flushing of the output buffer
 * on disconnect. Removed a ton of debug messages.
 *
 * Revision 1.1.1.1  2003/03/07 11:50:20  gibreel
 * Initial import.
 *
 */

#ifndef __libkmldonkey_donkeyprotocol_h__
#define __libkmldonkey_donkeyprotocol_h__

#include <qintdict.h>
#include <qstring.h>
#include <qmap.h>
#include "donkeysocket.h"
#include "fileinfo.h"
#include "serverinfo.h"
#include "network.h"
#include "clientinfo.h"
#include "shareinfo.h"
#include "searchinfo.h"
#include "searchquery.h"
#include "hostmanager.h"

// Maximum version of GUI protocol supported
#define MAX_PROTOCOL_VERSION 16

// Minimum version of GUI protocol supported
#define MIN_PROTOCOL_VERSION 10

//! A high level abstraction of the mldonkey GUI protocol.
/*!
 * DonkeyProtocol wraps around a DonkeySocket object, and handles all
 * aspects of passing messages back and forth through it. It provides
 * signals and slots for linking your application to the mldonkey core
 * without having to deal with the protocol itself.
 */

class DonkeyProtocol : public QObject

{
    Q_OBJECT

public:

    //! Opcodes for core messages.
    enum FromCore {
	CoreProtocol = 0,
	Options_info,
	DefineSearches = 3,
	Result_info,
	Search_result,
	Search_waiting,
	File_info_v1,
	File_downloaded_v1,
	File_update_availability,
	File_add_source, // 10
	Server_busy,
	Server_user,
	Server_state,
	Server_info_v1,
	Client_info,
	Client_state,
	Client_friend,
	Client_file,
	Console,
	Network_info, // 20
	User_info,
	Room_info,
	Room_message,
	Room_add_user,
	Client_stats_v1,
	Server_info,
	MessageFromClient,
	ConnectedServers,
	DownloadFiles_v1,
	DownloadFiles_v2, // 30
	Room_info_v2,
	Room_remove_user,
	Shared_file_info_v1,
	Shared_file_upload,
	Shared_file_unshared,
	Add_section_option,
	Client_stats_v2,
	Add_plugin_option,
	Client_stats_v3,
	File_info_v2, // 40
	DownloadFiles_v3,
	DownloadedFiles_v1,
	File_info_v3,
	DownloadFiles_v4,
	DownloadedFiles_v2,
	File_downloaded,
	BadPassword,
	Shared_file_info,
	Client_stats,
	File_remove_source, // 50
	CleanTables,
	File_info,
	DownloadFiles,
	DownloadedFiles
    };

    //! Opcodes for GUI messages.
    enum ToCore {
	GuiProtocol = 0,
	ConnectMore_query,
	CleanOldServers,
	KillServer,
	ExtendedSearch,
	Password_v1,
	Search_query_v1,
	Download_query_v1,
	Url,
	RemoveServer_query,
	SaveOptions_query, // 10
	RemoveDownload_query,
	ServerUsers_query,
	SaveFile,
	AddClientFriend,
	AddUserFriend,
	RemoveFriend,
	RemoveAllFriends,
	FindFriend,
	ViewUsers,
	ConnectAll, // 20
	ConnectServer,
	DisconnectServer,
	SwitchDownload,
	VerifyAllChunks,
	QueryFormat,
	ModifyMp3Tags,
	ForgetSearch,
	SetOption,
	Command,
	Preview, // 30
	ConnectFriend,
	GetServer_users,
	GetClient_files,
	GetFile_locations,
	GetServer_info,
	GetClient_info,
	GetFile_info,
	GetUser_info,
	SendMessage,
	EnableNetwork, // 40
	BrowseUser,
	Search_query,
	MessageToClient,
	GetConnectedServers,
	GetDownloadFiles,
	GetDownloadedFiles,
	GuiExtensions,
	SetRoomState,
	RefreshUploadStats,
	Download_query, // 50
	SetFilePriority,
	Password,
	CloseSearch,
	AddServer_query
    };

    //! Search types.
    enum SearchType {
	LocalSearch = 0,
	RemoteSearch,
	SubscribeSearch
    };

    //! Error values of the DonkeyProtocol.
    enum Error {
	//! No error, disconnect was intentional.
	NoError = 0,
	//! Client is unauthenticated or sent a bad password.
	AuthenticationError,
	//! Connection to the provided address was refused.
	ConnectionRefusedError,
	//! Unable to resolve the provided host name.
	HostNotFoundError,
	//! The QSocket signalled a read error.
	SocketReadError,
	//! The core is using an unsupported GUI protocol.
	ObsoleteProtocolError
    };

    //! Constructor.
    /*!
     * \param poll If true, the object will enable the poll extension when
     *   connecting to the core. This will cause the core to send you regular
     *   Client_stats messages and very little else, unless you ask for it.
     *   Useful for anything that isn't meant to be a full-fledged GUI.
     * \param parent The parent object.
     * \param name The object name.
     */
    DonkeyProtocol(bool poll = false, QObject* parent = 0, const char* name = NULL );
    //! Destructor.
    ~DonkeyProtocol();

    //! Set the authentication credentials.
    /*!
     * Set the username and password the object will send to the core when connecting.
     * You should do this _before_ initiating a connection.
     *
     * Please note: The usename only has meaning for protocol 14 or higher
     * (mldonkey 2.03).
     *
     * \param username The username.
     * \param pwd The password. If an empty string, no password will be sent.
     * \sa setPassword(QString&), username(), password(), connectDonkey()
     */
    void setPassword(const QString& username, const QString& pwd);

    //! Set the authentication password.
    /*!
     * Set the password the object will send to the core when connecting.
     * You should do this _before_ initiating a connection.
     *
     * Please note: Using this method sets the username to "admin", which
     * is the default username when connecting to a protocol 14 core
     * (mldonkey 2.03 or higher). The username will not be sent if the
     * core protocol is below 14, as these protocols use only the password
     * for authentication.
     *
     * \param pwd The password. If an empty string, no password will be sent.
     * \sa setPassword(QString&, QString&), password(), connectDonkey()
     */
    void setPassword(const QString& pwd);

    //! Get the authentication password.
    /*!
     * \return The authentication password set with setPassword().
     * \sa setPassword()
     */
    const QString& password();

    //! Get the authentication username.
    /*!
     * \return The username set with setPassword().
     * \sa setPassword(QString&, QString&)
     */
    const QString& username();

    //! Get if the socket is connected
    /*!
     * \return socket is connected (true) or disconnected (false)
     */
    bool isConnected();

    //! Reconnect to the core.
    /*!
     * Initiate a connection to the core, using the address previously set.
     * \sa connectDonkey(const QString&, Q_UINT16)
     */
    void connectDonkey();

    //! Connect to the core.
    /*!
     * Initiate a connection to the core at the provided address.
     * \param host The host address.
     * \param port The number of the core's GUI port.
     */
    void connectDonkey(const QString& host, Q_UINT16 port);

    //! Connect to the core.
    /*!
     * Initiate a connection to the core described by the provided DonkeyHost.
     * This also sets up the authentication credentials, so you don't need to
     * call setPassword() if you use this method to connect.
     * \param host A DonkeyHost reference describing the core to connect to.
     */
    void connectDonkey(const DonkeyHost& host);

    //! Disconnect from the core.
    /*!
     * Initiate a disconnection from the core. The signal donkeyDisconnected(int)
     * will be emitted when the disconnect is complete.
     * \param force If true, discard any still unwritten messages. If false, any
     *   messages still in the output buffer will be dispatched to the core before
     *   disconnecting.
     * \sa connectDonkey()
     */
    void disconnectDonkey(int force = 0);

    //! Get the protocol version.
    /*!
     * When connecting to the core, the DonkeyProtocol object and the core agree
     * on which version of the protocol to use. This will be the highest version
     * of the protocol supported by both parties. This method is used to discover
     * the protocol version currently in use.
     * \return The current protocol version.
     */
    const int protocolVersion();

    //! Find the state of the communications socket.
    /*!
     * \return The state of the DonkeySocket object being used to communicate
     *   with the core. The return value is equal to QSocket::state() except
     *   that the state will be Connecting until communication with
     *   the core has been established, even if the socket is actually
     *   connected.
     * \sa QSocket::state()
     */
    const QSocket::State donkeyStatus();

    //! Get the currently downloading files.
    /*!
     * \return A dictionary containing FileInfo objects representing the
     *   currently downloading files, indexed on their fileNo() values.
     */
    const QIntDict<FileInfo>& downloadFiles();

    //! Get the completed files.
    /*!
     * \return A dictionary containing FileInfo objects representing the
     *   completed files, indexed on their fileNo() values.
     */
    const QIntDict<FileInfo>& downloadedFiles();

    //! Get the connected servers.
    /*!
     * \return A dictionary containing ServerInfo objects representing the
     *   currently connected servers, indexed on their serverNo() values.
     */
    const QIntDict<ServerInfo>& connectedServers();

    //! Get the network list.
    /*!
     * \return A dictionary containing Network objects representing the
     *   available networks, indexed on their networkNo() values.
     */
    const QIntDict<Network>& availableNetworks();

    //! Get the client list.
    /*!
     * \return A dictionary containing ClientInfo objects representing
     *   the clients, indexed on their clientNo() values.
     */
    const QIntDict<ClientInfo>& clientList();

    //! Get the shared file list.
    /*!
     * \return A dictionary containing ShareInfo objects representing
     *   the shares, indexed on their shareNo() values.
     */
    const QIntDict<ShareInfo>& sharedFiles();

    //! Get the list of active searches.
    /*!
     * \return A dictionary containing SearchInfo objects representing
     *   the active searches, indexed on their searchNo() values.
     */
    const QIntDict<SearchInfo>& activeSearches();

    //! Find a downloading file by its file number.
    /*!
     * \param fileno The file number to search for.
     * \return A pointer to the FileInfo object describing the requested
     *   file number, or NULL if there was no such file number.
     * \sa downloadFiles(), findDownloadFileByHash()
     */
    FileInfo* findDownloadFileNo(int fileno);

    //! Find a downloading file by its MD4 hash.
    /*!
     * \param hash The hash to search for.
     * \return A pointer to the FileInfo object describing the requested
     *   hash, or NULL if there is no file being downloaded with the given
     *   hash.
     * \sa downloadFiles(), findDownloadFileNo(), findDownloadFileByHash(QString&)
     */
    FileInfo* findDownloadFileByHash(const QByteArray& hash);

    //! Find a downloading file by its MD4 hash string representation.
    /*!
     * \param hash String representation of the hash to search for.
     * \return A pointer to the FileInfo object describing the requested
     *   hash, or NULL if there is no file being downloaded with the given
     *   hash.
     * \sa downloadFiles(), findDownloadFileNo(), findDownloadFileByHash(QByteArray&)
     */
    FileInfo* findDownloadFileByHash(const QString& hash);

    //! Find a downloaded file by its file number.
    /*!
     * \param fileno The file number to search for.
     * \return A pointer to the FileInfo object describing the requested
     *   file number, or NULL if there was no such file number.
     * \sa downloadedFiles()
     */
    FileInfo* findDownloadedFileNo(int fileno);

    //! Find a connected server by its server number.
    /*!
     * \param serverno The server number to search for.
     * \return A pointer to the ServerInfo object describing the requested
     *   server, or NULL if there was no such server.
     * \sa connectedServers()
     */
    ServerInfo* findServerNo(int serverno);

    //! Find a network by its network number.
    /*!
     * \param nwno The network number to search for.
     * \return A pointer to the Network object describing the requested
     *   network, or NULL if there was no such network.
     * \sa availableNetworks()
     */
    Network* findNetworkNo(int nwno);

    //! Find a client by its client number.
    /*!
     * \param clno The client number to search for.
     * \return A pointer to the ClientInfo object describing the requested
     *   client, or NULL if there was no such client.
     * \sa clientList()
     */
    ClientInfo* findClientNo(int clno);

    //! Find a shared file by its share number.
    /*!
     * \param shno The share number to search for.
     * \return A pointer to the ShareInfo object describing the requested
     *   share, or NULL if there was no such share.
     * \sa sharedFiles()
     */
    ShareInfo* findShareNo(int shno);

    //! Find a search by its search number.
    /*!
     * \param sno The search number to search for.
     * \return A pointer to the SearchInfo object describing the requested
     *   search, or NULL if there was no such search.
     * \sa activeSearches()
     */
    SearchInfo* findSearchNo(int sno);

    //! Find a client share by its result number.
    /*! This effectively searches the search results that haven't been assigned to a
     *  search. The core reports client shares as search results, but doesn't link
     *  them to any search.
     * \param fileno The result number to search for.
     * \return A pointer to the ResultInfo object describing the requested file,
     *   or NULL if there was no such file.
     * \sa clientFileListing()
     */
    const ResultInfo* findClientFile(int fileno);


    //! Save a file.
    /*!
     * Ask the core to save a file under the given name.
     * \param fileno The file number of the file to save.
     * \param name The name under which the file will be saved.
     */
    void saveFile(int fileno, const QString& name);

    //! Pause or resume a downloading file.
    /*!
     * Ask the core to pause or resume a file.
     * \param fileno The file number to operate on.
     * \param pause True to pause the file, false to resume it.
     */
    void pauseFile(int fileno, bool pause);

    //! Cancel a downloading file.
    /*!
     * Ask the core to abort downloading the given file.
     * \param fileno The file number of the file to cancel.
     */
    void cancelFile(int fileno);

    //! Set the download priority of a file.
    /*!
     * Tell the core to adjust a file's download priority.
     * \param fileno The file number to be adjusted.
     * \param pri The new priority value.
     */
    void setFilePriority(int fileno, int pri);

    //! Retry connecting to a file's sources.
    /*!
     * Ask the core to attempt to reconnect to all known sources
     * for a file.
     * \param fileno The file number to retry.
     */
    void retryFile(int fileno);

    //! Verify a file's chunks.
    /*!
     * Ask the core to verify a file's downloaded chunks.
     * \param fileno The file number to verify.
     */
    void verifyFileChunks(int fileno);

    //! Request the format of a file.
    /*!
     * Ask the core to attempt to determine a file's format.
     * \param fileno The file number to examine.
     */
    void getFileFormat(int fileno);

    //! Preview a file.
    /*!
     * Ask the core to launch the previewer for the given file.
     * \param fileno The file number to preview.
     */
    void previewFile(int fileno);

    //! Submit an ed2k URL for downloading.
    /*!
     * Submit an ed2k URL to the core for downloading.
     * \param url The ed2k URL to download.
     */
    void submitURL(const QString& url);

    //! Send a DonkeyMessage to the core.
    /*!
     * Dispatch a DonkeyMessage to the core. This is a low level
     * method and should be used with caution.
     * \param msg The message to send.
     */
    void sendMessage(const DonkeyMessage& msg);

    //! Ask the core to connect to more servers.
    void connectMoreServers();

    //! Ask the core to remove expired servers.
    void cleanOldServers();

    //! Remove a server from the server list.
    /*! \param serverno The number of the server to remove.
     */
    void removeServer(int serverno);

    //! Ask the core to send updated server info for a given server.
    /*! \param serverno The number of the server to request info for.
     */
    void getServerInfo(int serverno);

    //! Ask the core to send a list of users connected to a server.
    /*! \param serverno The number of the server to request an user list for.
     */
    void getServerUsers(int serverno);

    //! Ask the core to connect to a given server.
    /*! \param serverno The server to connect to.
     */
    void connectServer(int serverno);

    //! Ask the core to disconnect from a given server.
    /*! \param serverno The server to disconnect from.
     */
    void disconnectServer(int serverno);

    //! Add a server to the server list.
    /*! \param network The network this server belongs to.
     *  \param ip The IP address of the server.
     *  \param port The port number of the server.
     */
    void addServer(int network, const QString& ip, int16 port);

    //! Start a search.
    /*! \param searchNum    Each search should have its own identifier.
     *  \param query        The search query to submit.
     *  \param maxHits      Maximum number of results.
     *  \param searchType   The type of search to perform.
     *  \param network      The network to search, or 0 to search all networks.
     */
    void startSearch(int searchNum,
		     SearchQuery* query,
                     int maxHits,
                     SearchType searchType,
                     int network = 0);

    //! Stop a search and discard search results.
    /*! \param searchNum The search ID to stop.
     */
    void stopSearch(int searchNum);

    //! Start to download a file.
    /*! \param names  List of filenames
     *  \param num    Download ID
     *  \param force  Force downloading the files
     */
    void startDownload(const QStringList& names, int num, bool force);

    //! Refresh file information.
    /*! Ask the core to send the information for the given file again,
     *  to allow the protocol object to update it in case it has changed.
     *  \param fileno The number of the file to refresh
     */
    void refreshFileInfo(int fileno);

    //! Refresh client information.
    /*! Ask the core to resend the information for the given client,
     *  to allow the protocol object to update it in case it has changed.
     *  \param clientno The number of the client to refresh
     */
    void refreshClientInfo(int clientno);

    //! Get the number of servers the core is currently connected to.
    /*! \return The number of connected servers.
     */
    uint connectedServerCount();

    //! Get the number of servers in the core's server list.
    /*! \return The total number of servers in the server list.
     */
    uint totalServerCount();

    //! Get a list of friend clients.
    /*! \return A list of ints corresponding to the client IDs of the current friends.
     */
    const QValueList<int>& friendList();
    //! Search for a friend.
    /*! \param name The name of the friend to search for.
     */
    void searchForFriend(const QString& name);
    //! Add a client to the friend list.
    /*! \param client The client number to make a friend.
     */
    void addClientFriend(int client);
    //! Remove a friend from the friend list.
    /*! \param client The client number of the friend to remove.
     */
    void removeFriend(int client);
    //! Attempt to connect to a friend.
    /*! \param client The client number of the friend to connect to.
     */
    void connectFriend(int client);

     //! Returns the MLDonkey-options
     /*! \return a QMap<QString,QString> list of key=value pairs
      */
     QMap<QString, QString> OptionsList();

    //! Set an option.
    /*! Tell the core to set the given option to the given value.
     *  \param option The name of the option to set
     *  \param value The value to assign to the named option
     */
    void setOption(const QString& option, const QString& value);

     //! Set and save MLDonkey-options
     /* \param opts key=value option-pairs
      */
     void setOptions(QMap<QString, QString> opts);

     //! Enable or disable an avaible network
     /* \param nwno QIntDict<Network> networks-number of the network
      * \param enable Enable or disable the network
      */
     void enableNetwork(int nwno, bool enable);

public slots:

    //! Request a new download file list.
    /*!
     * Ask the core to send an updated download file list.
     * \sa updatedDownloadFiles()
     */
    void updateDownloadFiles();

    //! Request a new downloaded file list.
    /*!
     * Ask the core to send an updated list of completed files.
     * \sa updatedDownloadedFiles()
     */
    void updateDownloadedFiles();

    //! Request a new list of connected servers.
    /*!
     * Ask the core to send an updated list of connected servers.
     * \sa updatedConnectedServers()
     */
    void updateConnectedServers();

    //! Request an updated list of shared files.
    /*!
     * Ask the core to send an updated list of shared files.
     */
    void refreshShared();

    //! Send a console message to the core.
    /*!
     * Dispatch a console message to the core. The result will be sent
     * as a consoleMessage() signal when it arrives.
     * \sa consoleMessage()
     */
    void sendConsoleMessage(const QString& msg);

protected slots:

    //! Process any messages in the DonkeySocket queue.
    void processMessage();

private slots:
    void socketDisconnected();
    void socketError(int err);

private:
    void pruneClientRecord(int clientno);
    void flushState();

    DonkeySocket sock;
    QString uname, passwd;
    QIntDict<FileInfo> download;
    QIntDict<FileInfo> downloaded;
    QIntDict<ServerInfo> servers;
    QIntDict<Network> networks;
    QIntDict<ClientInfo> clients;
    QIntDict<ShareInfo> shares;
    QIntDict<SearchInfo> searches;
    QIntDict<ResultInfo> unmappedResults;
    QMap<QString,QString> options;
    int donkeyError, cstate, proto;
    uint connectedservers;
    bool wantpoll;
    QValueList<int> friends;

signals:
    //! Communication with the core has been established.
    void donkeyConnected();
    //! The core has been disconnected for some reason.
    /*!
     * \param status The Error value that caused the disconnect.
     */
    void donkeyDisconnected(int status);
    //! A message has been received from the core.
    /*!
     * This is a low level function and should be used with caution.
     * The signal is emitted before the DonkeyProtocol object attempts
     * to parse the message.
     * \param message The message that was received.
     */
    void messageReceived(const DonkeyMessage* message);
    //! Core has sent updated client stats.
    /*!
     * \param ul Bytes uploaded.
     * \param dl Bytes downloaded.
     * \param sh Bytes being shared.
     * \param nsh Number of files being shared.
     * \param tul TCP upload rate.
     * \param tdl TCP download rate.
     * \param uul UDP upload rate.
     * \param udl UDP download rate.
     * \param ndl Number of files being downloaded.
     * \param ncp Number of completed files waiting to be committed.
     */
    void clientStats(int64 ul, int64 dl, int64 sh, int nsh, int tul, int tdl, int uul, int udl, int ndl, int ncp);
    //! Updated list of downloading files has been received.
    /*!
     * \sa updateDownloadFiles()
     */
    void updatedDownloadFiles();
    //! Updated list of completed files has been received.
    /*!
     * \sa updateDownloadedFiles()
     */
    void updatedDownloadedFiles();
    //! Updated list of connected servers has been received.
    /*!
     * \sa updateConnectedServers()
     */
    void updatedConnectedServers();
    //! A console message has arrived.
    /*!
     * \param msg The text of the received console message.
     * \sa sendConsoleMessage()
     */
    void consoleMessage(const QString& msg);
    //! A file source has been updated.
    /*! \param file The file number.
     *  \param source The source number.
     */
    void fileSourceUpdated(int file, int source);
    //! A file source has been removed.
    /*! \param file The file number.
     *  \param source The source number.
     */
    void fileSourceRemoved(int file, int source);
    //! A file has been updated.
    /*! \param file The file number. */
    void fileUpdated(int file);
    //! A file has been removed.
    /*! \param file The file number. */
    void fileRemoved(int file);
    //! A client has updated information (or has just been added).
    /*! \param client The client number. */
    void clientUpdated(int client);
    ///! A client has been removed.
    /*! \param client The client number. */
    void clientRemoved(int client);
    //! A server has updated information.
    /*! \param server The server number. */
    void serverUpdated(int server);
    //! A server has been removed.
    /*! \param server The server number. */
    void serverRemoved(int server);
    //! A shared file has updated information.
    /*! \param share The share number. */
    void shareUpdated(int share);
    //! A shared file has been unshared.
    /*! \param share The share number. */
    void shareRemoved(int share);
    //! Network information has been updated.
    /*!
     * \param network The number of the network that was updated.
     * \sa findNetworkNo()
     */
    void networkUpdated(int network);
    //! A friend has been updated.
    /*! \param client The client number of the updated friend. */
    void friendUpdated(int client);
    //! A friend has been removed.
    /*! \param client The client number of the removed friend. */
    void friendRemoved(int client);
    //! A directory listing has been received from a client.
    /*! This will be followed by a number of searchUpdated() signals keyed to
     *  the result ID in this signal, constituting the contents of the given
     *  directory.
     *  \param client The number of the client involved.
     *  \param directory The name of the directory we are receiving.
     *  \param result The search result ID the contents of this directory will be sent as.
     */
    void clientFileListing(int client, const QString& directory, int result);

    //! A message was received from a client.
    /*! \param client The number of the client who sent the message.
     *  \param message The message that was received.
     */
    void messageFromClient(int client, const QString& message);

    //! A Search-Result arrived.
    /*! \param searchnum The search number.
     *  \param searchinfo The SearchInfo-Object
     */
    void searchUpdated(int searchnum, const ResultInfo* searchinfo);
    //! An unhandled message has been received.
    /*!
     * This signal is emitted when the DonkeyProtocol object receives a
     * message it doesn't know how to handle.
     * \param message The unhandled message.
     */
    void unhandledMessage(const DonkeyMessage* message);

};

#endif
