/* -*- c++ -*-
 *
 * $Id: donkeymessage.h,v 1.10 2003/06/30 14:53:42 gibreel Exp $
 *
 * Copyright (C) 2003 Petter E. Stokke <gibreel@gibreel.net>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * $Log: donkeymessage.h,v $
 * Revision 1.10  2003/06/30 14:53:42  gibreel
 * Still hunting bugs in source management: implemented a lot of missing
 * message handlers pertaining to cleanup of client records. Quite a few bug
 * fixes. Protocol object now emits *Removed() signals corresponding to
 * *Updated() signals instead of sending update signals for removed records and
 * leaving the user to determine whether to remove the records.
 *
 * Revision 1.9  2003/06/28 22:28:44  gibreel
 * All DonkeyMessage typedefs are now unsigned - having some of them signed
 * caused problems.
 *
 * Revision 1.8  2003/06/27 11:48:42  gibreel
 * Added a static method to DonkeyMessage for overriding the default string
 * codec.
 *
 * Revision 1.7  2003/06/26 23:31:00  gibreel
 * readString() and writeString() now use QTextCodec to ensure proper character
 * conversion. It assumes the core uses ISO-8859-1; in the future, one might
 * want this to be configurable.
 *
 * Revision 1.6  2003/06/13 18:20:01  gibreel
 * Libkmldonkey now uses references instead of pointers everywhere except where
 * it would cause an obvious performance impact, which should lead to less
 * chance of memory leaks and cleaner code in general. Almost everything that
 * should be const is now also const.
 *
 * Revision 1.5  2003/03/23 23:34:59  gibreel
 * A lot of API additions, especially lists keeping track of shared files and
 * clients.
 *
 * Revision 1.4  2003/03/08 22:37:58  gibreel
 * Written some Doxygen documentation for the library API.
 *
 * Revision 1.3  2003/03/08 20:13:19  gibreel
 * Switched readString/writeString in DonkeyMessage from using pointers to
 * QStrings to using references, which leads to far less hassle, shorter code,
 * and less chance of memory leaks.
 *
 * Revision 1.2  2003/03/08 17:08:12  gibreel
 * Lots of minor API changes.
 *
 * Revision 1.1.1.1  2003/03/07 11:50:20  gibreel
 * Initial import.
 *
 */

#ifndef __libkmldonkey_donkeymessage_h__
#define __libkmldonkey_donkeymessage_h__

#include <qstring.h>
#include <qmemarray.h>
#include <qmap.h>
#include <qvariant.h>

typedef Q_UINT8 int8;
typedef Q_UINT16 int16;
typedef Q_UINT32 int32;
typedef Q_UINT64 int64;


class QTextCodec;


//! A core protocol message.
/*!
 * This class represents a message of the kind the core and client
 * use to communicate with.
 */

class DonkeyMessage : public QMemArray<int8>

{
public:
    //! Construct an empty message with the given opcode.
    /*! \param opcode The message's opcode. */
    DonkeyMessage(int opcode);
    //! Construct a message from a memory buffer.
    /*! The buffer must include the opcode as the first two bytes.
     *  It should not include the message length. */
    DonkeyMessage(const char* data, int len);
    //! Construct a message with the given opcode and length.
    /*! \param opcode The message's opcode.
     *  \param len The message size. */
    DonkeyMessage(int opcode, int len);

    //! Set the opcode of the message.
    /*! \param opcode The new opcode. */
    void setOpcode(int opcode);
    //! Get the opcode of the message.
    /*! \return The opcode. */
    int opcode() const;

    //! Append a byte to the message.
    /*! \param v A byte. */
    void writeInt8(int8 v);
    //! Append a 16-bit word to the message.
    /*! \param v A 16-bit word. */
    void writeInt16(int16 v);
    //! Append a 32-bit word to the message.
    /*! \param v A 32-bit word. */
    void writeInt32(int32 v);
    //! Append a 64-bit word to the message.
    /*! \param v A 64-bit word. */
    void writeInt64(int64 v);
    //! Append a string to the message.
    /*! \param v A string. */
    void writeString(const QString& v);
    //! Append a string to the message.
    /*! \param v A null terminated string buffer. */
    void writeString(const char* v);
    //! Append a floating point value to the message.
    /*! \param v A floating point value. */
    void writeFloat(double v);

    //! Append a copy of a memory buffer to the message.
    /*! \param buf A pointer to the start of the buffer.
     *  \param sz The size in bytes of the buffer. */
    void feedBuffer(const char* buf, int sz);
    //! Reset the read position to the start of the message.
    void resetPosition();

    //! Read a byte from the message.
    /*! \return A byte. */
    int8 readInt8();
    //! Read a 16-bit word from the message.
    /*! \return A 16-bit word. */
    int16 readInt16();
    //! Read a 32-bit word from the message.
    /*! \return A 32-bit word. */
    int32 readInt32();
    //! Read a 64-bit word from the message.
    /*! \return A 64-bit word. */
    int64 readInt64();
    //! Read a string from the message.
    /*! \return A string. */
    QString readString();
    //! Read a floating point value from the message.
    /*! \return A floating point value. */
    double readFloat();
    //! Read an IP address from the message.
    /*! \return A string describing an IP address. */
    QString readIPAddress();
    //! Read an encoded IP address from the message.
    /*! \return A string describing an IP address. */
    QString readAddress();
    //! Read an encoded tag and insert it into the given dictionary.
    /*! \param dict A reference to the dictionary to operate on.
     *  \return True on success, false if there was a syntax error.
     */
    bool readTag(QMap<QString,QVariant>& dict);

    //! Construct a string containing a hex dump of the message.
    QString dumpArray() const;

    //! Explicitly specify a QTextCodec to use in decoding strings from messages.
    /*! DonkeyMessage defaults to using an ISO-8859-1 codec when translating
     *  strings from the core into internal Unicode representations. If you are
     *  expecting a different charset, you can call this static method to specify
     *  the codec to be used.
     *  \param codec The QTextCodec object to be used in string decoding.
     */
    static void setStringCodec(QTextCodec* codec);

protected:
    //! The message's opcode.
    int op;

private:
    void initCodec();
    void writeInt(int64 v, int sz);
    int64 readInt(int sz);
    int pos;
    static QTextCodec* codec;
};

#endif
