/*
 * $Id: donkeymessage.cpp,v 1.13 2003/07/30 19:11:22 gibreel Exp $
 *
 * Copyright (C) 2003 Petter E. Stokke <gibreel@gibreel.net>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * $Log: donkeymessage.cpp,v $
 * Revision 1.13  2003/07/30 19:11:22  gibreel
 * Fixed a bug in the previous bug fix.
 *
 * Revision 1.12  2003/07/30 17:53:37  gibreel
 * Fixed a crash bug when the active QTextCodec fails to convert a Unicode
 * string into the user's selected charset. Note that this only keeps the
 * application from crashing - the string that ends up being sent when this
 * happens is an empty string, hence the problem still exists; it's just no
 * longer a fatal problem.
 *
 * Revision 1.11  2003/07/29 23:14:50  gibreel
 * Trying to correct a possible endian issue when decoding IP addresses from
 * the core protocol stream. Either this closes #4526, or it makes things
 * worse.
 *
 * Revision 1.10  2003/06/30 14:53:42  gibreel
 * Still hunting bugs in source management: implemented a lot of missing
 * message handlers pertaining to cleanup of client records. Quite a few bug
 * fixes. Protocol object now emits *Removed() signals corresponding to
 * *Updated() signals instead of sending update signals for removed records and
 * leaving the user to determine whether to remove the records.
 *
 * Revision 1.9  2003/06/27 11:48:42  gibreel
 * Added a static method to DonkeyMessage for overriding the default string
 * codec.
 *
 * Revision 1.8  2003/06/26 23:31:00  gibreel
 * readString() and writeString() now use QTextCodec to ensure proper character
 * conversion. It assumes the core uses ISO-8859-1; in the future, one might
 * want this to be configurable.
 *
 * Revision 1.7  2003/06/13 18:20:01  gibreel
 * Libkmldonkey now uses references instead of pointers everywhere except where
 * it would cause an obvious performance impact, which should lead to less
 * chance of memory leaks and cleaner code in general. Almost everything that
 * should be const is now also const.
 *
 * Revision 1.6  2003/06/09 23:55:06  gibreel
 * Added missing include required to build on BSD platforms.
 *
 * Revision 1.5  2003/04/13 01:44:43  gibreel
 * Corrected readString() implementation; conversion from QByteArray to QString
 * seems to null terminate, so we construct the QString manually instead.
 *
 * Revision 1.4  2003/03/23 23:34:59  gibreel
 * A lot of API additions, especially lists keeping track of shared files and
 * clients.
 *
 * Revision 1.3  2003/03/08 20:13:19  gibreel
 * Switched readString/writeString in DonkeyMessage from using pointers to
 * QStrings to using references, which leads to far less hassle, shorter code,
 * and less chance of memory leaks.
 *
 * Revision 1.2  2003/03/08 17:08:12  gibreel
 * Lots of minor API changes.
 *
 * Revision 1.1.1.1  2003/03/07 11:50:20  gibreel
 * Initial import.
 *
 */

#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <sys/types.h>
#include <kdebug.h>
#include <qtextcodec.h>

#include "donkeymessage.h"

QTextCodec* DonkeyMessage::codec = 0;

void DonkeyMessage::initCodec()
{
    if (!codec) {
	codec = QTextCodec::codecForName("ISO-8859-1");
	if (!codec) codec = QTextCodec::codecForLocale();
    }
}

void DonkeyMessage::setStringCodec(QTextCodec* newCodec)
{
    codec = newCodec;
}

DonkeyMessage::DonkeyMessage( int opcode ) : QMemArray<int8>()
{
    initCodec();
    op = opcode;
    pos = 0;
}

DonkeyMessage::DonkeyMessage( const char* data, int len ) : QMemArray<int8>()
{
    initCodec();
    resize(len - 2);
    op = data[0] | (data[1] << 8);
    char *p = (char*)data + 2;
    int i;
    for (i=0; i<(len-2); i++)
	(*this)[i] = p[i];
    pos = 0;
}

DonkeyMessage::DonkeyMessage(int opcode, int len) : QMemArray<int8>(len)
{
    initCodec();
    op = opcode;
    pos = 0;
}

int DonkeyMessage::opcode() const
{
    return op;
}

void DonkeyMessage::setOpcode(int opcode)
{
    op = opcode;
}

inline void DonkeyMessage::writeInt(int64 v, int sz)
{
    int j;
    pos = size();
    resize(pos + sz);
    for(j = 0; j < sz; j++)
        (*this)[pos + j] = (v & (-1)) >> (8 * j);
    pos += sz;
}

void DonkeyMessage::writeInt8(int8 v)
{
    writeInt((int64)v, 1);
}

void DonkeyMessage::writeInt16(int16 v)
{
    writeInt((int64)v, 2);
}

void DonkeyMessage::writeInt32(int32 v)
{
    writeInt((int64)v, 4);
}

void DonkeyMessage::writeInt64(int64 v)
{
    writeInt((int64)v, 8);
}

void DonkeyMessage::writeString(const QString& v)
{
    QCString s = codec->fromUnicode(v);
    if (s.isNull()) {
	kdDebug() << "Unable to convert string into charset " << codec->name() << ": \"" << v << "\"" << endl;
	writeString("");
    } else
	writeString((const char*)s);
}

void DonkeyMessage::writeString(const char* v)
{
    int i,l = strlen(v);
    pos = size();
    writeInt16(l);
    resize(pos + l);
    for (i=0; i<l; i++) {
	(*this)[pos++] = v[i];
    }
}

void DonkeyMessage::writeFloat(double v)
{
    QString foo = QString();
    foo.sprintf("%.4f", v);
    writeString(foo);
}

void DonkeyMessage::feedBuffer(const char* buf, int sz)
{
    memcpy(data() + pos, buf, sz);
    pos += sz;
}

void DonkeyMessage::resetPosition()
{
    pos = 0;
}

inline int64 DonkeyMessage::readInt(int sz)
{
    int i;
    int64 res = 0;
    for(i = 0; i < sz; i++)
        res += ((*this)[pos + i] & 0xFF) << (8 * i);
    pos += sz;
    return res;
}

int8 DonkeyMessage::readInt8()
{
    return (int8)readInt(1);
}

int16 DonkeyMessage::readInt16()
{
    return (int16)readInt(2);
}

int32 DonkeyMessage::readInt32()
{
    return (int32)readInt(4);
}

int64 DonkeyMessage::readInt64()
{
    return (int64)readInt(8);
}

QString DonkeyMessage::readString() {
    int sz = (int)readInt16();
    QByteArray buf(sz);
    memcpy(buf.data(), *this + pos, sz);
    pos+= sz;
    // kdDebug() << "String sz " << sz << " data \"" << buf.data() << "\"" << endl;
    return codec->toUnicode(buf);
}

double DonkeyMessage::readFloat()
{
    return readString().toDouble();
}

QString DonkeyMessage::readIPAddress()
{
    QString address = QString::number(readInt8()) + "."
	+ QString::number(readInt8()) + "."
	+ QString::number(readInt8()) + "."
	+ QString::number(readInt8());
    /*
      struct in_addr in;
      in.s_addr = readInt32();
      address = inet_ntoa(in);
    */
    return address;
}

QString DonkeyMessage::readAddress()
{
    if (readInt8())
        return readString();
    else
        return readIPAddress();
}

bool DonkeyMessage::readTag(QMap<QString,QVariant>& dict)
{
    QString name = readString();
    QVariant value;
    switch (readInt8()) {
    case 0:
        value = QVariant(readInt32());
        break;
    case 1:
        value = QVariant(readInt32());
        break;
    case 2:
        value = QVariant(readString());
        break;
    case 3:
        value = QVariant(readIPAddress());
        break;
    default:
        return false;
    }
    dict.replace(name, value);
    return true;
}

QString DonkeyMessage::dumpArray() const
{
    QString out = "Opcode " + QString::number(op) + ", size " + QString::number(size()) + "\n";

    int i;
    QString hex = "", asc = "", tmp;
    for (i=0; i<(int)size(); i++) {
	if (at(i) >= 32 && at(i) <= 127)
	    asc += QChar(at(i));
	else
	    asc += ".";
	tmp.sprintf("%02x", at(i));
	hex += tmp + " ";
	if (i % 16 == 15) {
	    tmp.sprintf("%8d: ", i - 15);
	    out += tmp + hex + "  " + asc + "\n";
	    hex = "";
	    asc = "";
	}
    }
    tmp.sprintf("%8d: ", i - (i % 16));
    for (i %= 16; i < 16; i++)
	hex += "   ";
    out += tmp + hex + "  " + asc + "\n";
    return out;
}

