/* -*- c++ -*-
 *
 * $Id: webbrowser.cpp,v 1.2 2003/06/30 23:30:35 gibreel Exp $
 *
 * Copyright (C) 2003 Petter E. Stokke <gibreel@gibreel.net>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * $Log: webbrowser.cpp,v $
 * Revision 1.2  2003/06/30 23:30:35  gibreel
 * Preliminary friend list support. A ton of updates to the libkmldonkey API to
 * accommodate this, most notably improvements to the search handling
 * necessitated by mldonkey's somewhat awkward reporting of friend shares.
 *
 * Revision 1.1  2003/06/30 00:34:03  gibreel
 * Added an embedded web browser class, and through it added Jigle and Bitzi
 * lookups to the file info dialog.
 *
 */

#include <kdebug.h>
#include <klocale.h>
#include <kguiitem.h>
#include <kprogress.h>
#include <kpushbutton.h>
#include <ksqueezedtextlabel.h>
#include <klineedit.h>
#include <qlayout.h>
#include <qregexp.h>
#include <qdatastream.h>

#include <fileinfo.h>
#include <serverinfo.h>
#include <shareinfo.h>
#include <searchinfo.h>

#include "kmldonkey.h"
#include "webbrowser.h"
#include "webbrowser.moc"



ED2KURL::ED2KURL(const KURL& u)
{
    QString url = u.prettyURL();
    QRegExp fre("^ed2k://\\|file\\|([^|]+)\\|(\\d+)\\|([0-9a-fA-F]+)\\|/?$");
    QRegExp sre("^ed2k://\\|server\\|(\\d+\\.\\d+\\.\\d+\\.\\d+)\\|(\\d+)\\|/?$");

    if (fre.search(url) >= 0) {
	type = "file";
	name = fre.cap(1).replace('+', ' ');
	size = fre.cap(2).toULong();
	hash = FileInfo::stringToMd4(fre.cap(3));
    }
    else if (sre.search(url) >= 0) {
	type = "server";
	address = sre.cap(1);
	port = sre.cap(2).toUShort();
    }
    else {
	type = "invalid";
    }
}

ED2KURL::ED2KURL(FileInfo* fi)
{
    type = "file";
    name = fi->fileName();
    size = fi->fileSize();
    hash = fi->fileMD4();
}

ED2KURL::ED2KURL(ShareInfo* fi)
{
    type = "file";
    name = fi->shareName();
    size = fi->shareSize();
    hash = fi->shareMD4();
}

ED2KURL::ED2KURL(ResultInfo* fi)
{
    type = "file";
    name = fi->resultName();
    size = fi->resultSize();
    hash = fi->resultMD4();
}

ED2KURL::ED2KURL(ServerInfo* si)
{
    type = "server";
    address = si->serverAddress();
    port = si->serverPort();
}

QString ED2KURL::ed2kType() const { return type; }
QString ED2KURL::serverAddress() const { return address; }
int16 ED2KURL::serverPort() const { return port; }
QString ED2KURL::fileName() const { return name; }
int64 ED2KURL::fileSize() const { return size; }
QByteArray ED2KURL::fileHash() const { return hash; }
bool ED2KURL::isInvalid() const { return type == "invalid"; }

bool ED2KURL::isSameFile(const ED2KURL& u) const
{
    if (type != "file" || u.ed2kType() != "file") return false;
    if (size != u.fileSize()) return false;
    if (hash != u.fileHash()) return false;
    return true;
}

QString ED2KURL::toString() const
{
    if (type == "file")
	return QString("ed2k://|file|") + name + "|" + QString::number(size) + "|" + FileInfo::md4ToString(hash) + "|/";
    else if (type == "server")
	return QString("ed2k://|server|") + address + "|" + QString::number(port) + "|/";
    else
	return QString::null;
}



WebBrowser::WebBrowser(QWidget* parent, const char* name)
    : QWidget(parent, name)
{
    hpos = 0;
    initialState = true;

    QBoxLayout *layout = new QVBoxLayout(this);

    browser = new KHTMLPart(this);
    connect(browser->browserExtension(), SIGNAL(openURLRequestDelayed(const KURL&, const KParts::URLArgs&)),
	    this, SLOT(slotUrlSelected(const KURL&, const KParts::URLArgs&)));
    connect(browser, SIGNAL(setStatusBarText(const QString&)), this, SLOT(setStatusBarText(const QString&)));
    connect(browser->browserExtension(), SIGNAL(loadingProgress(int)), this, SLOT(loadingProgress(int)));
    layout->addWidget(browser->widget());

    QBoxLayout* bottom = new QHBoxLayout();

    backButton = new KPushButton(KGuiItem(QString::null, "back", i18n("Move backward through the page history")), this);
    backButton->setEnabled(false);
    bottom->addWidget(backButton);
    forwardButton = new KPushButton(KGuiItem(QString::null, "forward", i18n("Move forward through the page history")), this);
    forwardButton->setEnabled(false);
    bottom->addWidget(forwardButton);
    homeButton = new KPushButton(KGuiItem(QString::null, "gohome", i18n("Go to the original page")), this);
    homeButton->setEnabled(false);
    bottom->addWidget(homeButton);

    connect(backButton, SIGNAL(clicked()), this, SLOT(backState()));
    connect(forwardButton, SIGNAL(clicked()), this, SLOT(forwardState()));
    connect(homeButton, SIGNAL(clicked()), this, SLOT(goHome()));

    statusbar = new KSqueezedTextLabel(this);
    statusbar->setFrameStyle(QFrame::StyledPanel | QFrame::Sunken);
    bottom->addWidget(statusbar, 90);

    progressbar = new KProgress(100, this);
    bottom->addWidget(progressbar, 10);

    layout->addLayout(bottom);
}

void WebBrowser::slotUrlSelected(const KURL& url, const KParts::URLArgs& args)
{
    browser->browserExtension()->setURLArgs(args);
    openURL(url);
}

void WebBrowser::restoreState(const QByteArray& state)
{
    QDataStream s(state, IO_ReadOnly);
    browser->browserExtension()->restoreState(s);
}

QByteArray WebBrowser::saveState()
{
    QByteArray state;
    QDataStream s(state, IO_WriteOnly);
    browser->browserExtension()->saveState(s);
    return state;
}

void WebBrowser::pushState()
{
    while (hpos < history.count()) history.pop_back();
    history.append(saveState());
    hpos = history.count();
    forwardButton->setEnabled(false);
    backButton->setEnabled(true);
}

void WebBrowser::backState()
{
    if (hpos <= 0) {
	backButton->setEnabled(false);
	return;
    }
    if (hpos >= history.count()) history.append(saveState());
    hpos--;
    restoreState(history[hpos]);
    backButton->setEnabled(hpos > 0);
    forwardButton->setEnabled(hpos < history.count() - 1);
}

void WebBrowser::forwardState()
{
    if (hpos >= history.count() - 1) {
	forwardButton->setEnabled(false);
	return;
    }
    hpos++;
    restoreState(history[hpos]);
    backButton->setEnabled(hpos > 0);
    forwardButton->setEnabled(hpos < history.count() - 1);
}

void WebBrowser::openURL(const QString& url)
{
    openURL(KURL(url));
}

void WebBrowser::openURL(const KURL& url)
{
    if (url.protocol() == "ed2k")
	handleED2K(ED2KURL(url));
    else {
	if (!initialState)
	    pushState();
	else {
	    initialState = false;
	    initialURL = url;
	    homeButton->setEnabled(true);
	}
	setStatusBarText(url.url());
	browser->openURL(url);
    }
}

void WebBrowser::goHome()
{
    if (!initialState)
	openURL(initialURL);
}

void WebBrowser::setStatusBarText(const QString& text)
{
    statusbar->setText(text);
}

void WebBrowser::loadingProgress(int percent)
{
    progressbar->setProgress(percent);
}

void WebBrowser::handleED2K(const ED2KURL& url)
{
    if (url.isInvalid()) return;
    KMLDonkey::App->donkey->submitURL(url.toString());
}

