/* -*- c++ -*-
 *
 * $Id: serverpage.cpp,v 1.7 2003/07/27 20:24:26 gibreel Exp $
 *
 * Copyright (C) 2003 Petter E. Stokke <gibreel@gibreel.net>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * $Log: serverpage.cpp,v $
 * Revision 1.7  2003/07/27 20:24:26  gibreel
 * Added KActions for switching between pages.
 *
 * Revision 1.6  2003/07/17 20:45:49  gibreel
 * Added copy icons to all copy actions.
 *
 * Revision 1.5  2003/06/30 14:59:22  gibreel
 * Updated lists to support libkmldonkey's new removed signals. Moved pages'
 * connect statements out of KMLDonkey's constructor into their own
 * constructors. Added a debug console displaying dumps of unhandled messages.
 *
 * Revision 1.4  2003/06/28 10:24:28  dipesh
 * Corrected a small typo.
 *
 * Revision 1.3  2003/06/19 22:21:09  gibreel
 * A saner fix for the disconnect/reconnect bug, which in turn solves a bug
 * introduced by the previous fix.
 *
 * Revision 1.2  2003/06/19 22:05:52  gibreel
 * Merged the General and Donkey prefs pages back together.
 *
 * Revision 1.1  2003/06/19 21:50:45  gibreel
 * Split the various pages off into individual objects, and made a ton of code
 * cleanups, API changes, and not a few bugfixes in the process. The
 * disconnect/reconnect bug, especially, now seems to be gone.
 *
 */

#include <klocale.h>
#include <kconfig.h>
#include <kaction.h>
#include <klineeditdlg.h>
#include <qintdict.h>
#include <qcheckbox.h>
#include <qpopupmenu.h>
#include <qclipboard.h>

#include "infolist.h"
#include "prefs.h"
#include "kmldonkey.h"
#include "serverpage.h"
#include "serverpage.moc"

ServerPage::ServerPage(QWidget* parent)
    : QVBox(parent, "serverPage")
    , KMLDonkeyPage()
    , ClipboardHelper()
{
    serverItemList.setAutoDelete(true);

    serverView = new InfoList(this, "serverView");
    serverView->addColumn( i18n( "Server name" ) );
    serverView->addColumn( i18n( "Network" ) );
    serverView->addColumn( i18n( "Status" ) );
    serverView->addColumn( i18n( "Users" ) );
    serverView->addColumn( i18n( "Files" ) );
    serverView->addColumn( i18n( "Description" ) );
    serverView->addColumn( i18n( "Address" ) );

    connect(serverView, SIGNAL(contextMenu(KListView*, QListViewItem*, const QPoint&)),
	    this, SLOT(contextServer(KListView*, QListViewItem*, const QPoint&)));

    connect(KMLDonkey::App->donkey, SIGNAL(serverUpdated(int)), this, SLOT(serverUpdated(int)));
    connect(KMLDonkey::App->donkey, SIGNAL(serverRemoved(int)), this, SLOT(serverRemoved(int)));
}

void ServerPage::setupActions(KActionCollection* actionCollection)
{
    (void)new KAction(i18n("&Connect to server"), "connect_creating", 0, this, SLOT(actionConnectServer()),
                      actionCollection, "connect_server");
    (void)new KAction(i18n("Connect &more servers"), "connect_established", 0, this, SLOT(actionConnectMoreServers()),
                      actionCollection, "connect_more_servers");
    (void)new KAction(i18n("&Disconnect from server"), "connect_no", 0, this, SLOT(actionDisconnectServer()),
                      actionCollection, "disconnect_server");

    (void)new KAction(i18n("&Add server"), "edit", 0, this, SLOT(actionAddServer()),
                      actionCollection, "add_server");
    (void)new KAction(i18n("&Remove server"), "editdelete", 0, this, SLOT(actionRemoveServer()),
                      actionCollection, "remove_server");
    (void)new KAction(i18n("Remove old servers"), "editshred", 0, this, SLOT(actionRemoveOldServers()),
                      actionCollection, "remove_old_servers");

    (void)new KAction(i18n("Copy server &URL to clipboard"), "editcopy", 0, this, SLOT(actionServerCopyURL()),
                      actionCollection, "server_copy_url");
    (void)new KAction(i18n("Copy server HTML to clipboard"), "editcopy", 0, this, SLOT(actionServerCopyHTML()),
                      actionCollection, "server_copy_html");
    showAllServersAction = new KToggleAction(i18n("Show all servers"), "filter", 0, actionCollection, "show_all_servers");

    connect(showAllServersAction, SIGNAL(toggled(bool)), this, SLOT(actionShowAllServers(bool)));

    (void)new KAction(i18n("Activate servers page"), 0, 0, this, SLOT(actionActivatePage()),
		      actionCollection, "activate_page_servers");
}

void ServerPage::clear()
{
    serverItemList.clear();
    serverView->clear();
}

void ServerPage::configurePrefsDialog(KMLDonkeyPreferences* prefs)
{
    prefs->generalPage->ShowAllServersCheckbox->setChecked(showAllServersAction->isChecked());
}

void ServerPage::applyPreferences(KMLDonkeyPreferences* prefs)
{
    if (prefs)
	if (prefs->generalPage->ShowAllServersCheckbox->isChecked() != showAllServersAction->isChecked())
	    actionShowAllServers( prefs->generalPage->ShowAllServersCheckbox->isChecked() );

    if (KMLDonkey::App->listFont !=  serverView->font())
        serverView->setFont(KMLDonkey::App->listFont);
}

void ServerPage::restoreState(KConfig* conf)
{
    conf->setGroup("Options");
    showAllServersAction->setChecked(conf->readBoolEntry("ShowAllServers", false));

    serverView->restoreLayout(conf, "ServerView");

    applyPreferences();
}

void ServerPage::saveState(KConfig* conf)
{
    serverView->saveLayout(conf, "ServerView");

    conf->setGroup("Options");
    conf->writeEntry("ShowAllServers", showAllServersAction->isChecked());
}

void ServerPage::contextServer(KListView*,QListViewItem*,const QPoint& pt)
{
    QPopupMenu *pop = (QPopupMenu*)(KMLDonkey::App->factory())->container("server_actions", KMLDonkey::App);
    pop->popup(pt);
}

void ServerPage::actionConnectServer()
{
    QPtrList<QListViewItem> list = serverView->selectedItems();
    ServerInfoItem* it;
    for (it = (ServerInfoItem*)list.first(); it; it = (ServerInfoItem*)list.next()) {
	ServerInfo* si = KMLDonkey::App->donkey->findServerNo(it->fileNo());
	if (!si) continue;
	KMLDonkey::App->donkey->connectServer(si->serverNo());
    }
}

void ServerPage::actionConnectMoreServers()
{
    KMLDonkey::App->donkey->connectMoreServers();
}

void ServerPage::actionDisconnectServer()
{
    QPtrList<QListViewItem> list = serverView->selectedItems();
    ServerInfoItem* it;
    for (it = (ServerInfoItem*)list.first(); it; it = (ServerInfoItem*)list.next()) {
	ServerInfo* si = KMLDonkey::App->donkey->findServerNo(it->fileNo());
	if (!si) continue;
	KMLDonkey::App->donkey->disconnectServer(si->serverNo());
    }
}

void ServerPage::actionAddServer()
{
    bool ok;
    QString server = KLineEditDlg::getText( i18n("Add server"),
                                            i18n("Enter the new server (fe. 'ed2k://|server|192.168.0.1|4662|/')."),
                                            QString::null, &ok, this );
    if (ok) KMLDonkey::App->donkey->submitURL(server);
}

void ServerPage::actionRemoveServer()
{
    QPtrList<QListViewItem> list = serverView->selectedItems();
    ServerInfoItem* it;
    for (it = (ServerInfoItem*)list.first(); it; it = (ServerInfoItem*)list.next()) {
        ServerInfo* si = KMLDonkey::App->donkey->findServerNo(it->fileNo());
        if (!si) continue;
        KMLDonkey::App->donkey->removeServer(si->serverNo());
    }
}

void ServerPage::actionRemoveOldServers()
{
    KMLDonkey::App->donkey->cleanOldServers();
}

void ServerPage::copyServerToClipboard(QPtrList<ServerInfo>& servers, ClipFormat format)
{
    QClipboard* clip = KApplication::clipboard();
    QStringList out;
    QPtrListIterator<ServerInfo> it(servers);
    ServerInfo* server;
    while ((server = it.current())) {
        ++it;
        switch (format) {
        case URL:
            out.append("ed2k://|server|" + server->serverAddress() + "|" + QString::number(server->serverPort()) + "|/");
            break;
        case HTML:
            out.append("<a href=\"ed2k://|server|" + server->serverAddress() + "|" + QString::number(server->serverPort()) +
                       "|/\">" + server->serverAddress() + ":" + QString::number(server->serverPort()) + "</a>");
            break;
        default:
            break;
        }
    }
    clip->setText(out.join("\n"), QClipboard::Selection);
}

void ServerPage::actionServerCopyURL()
{
    QPtrList<ServerInfo> servers;
    QPtrList<QListViewItem> list = serverView->selectedItems();
    ServerInfoItem* it;
    for (it = (ServerInfoItem*)list.first(); it; it = (ServerInfoItem*)list.next()) {
	ServerInfo* si = KMLDonkey::App->donkey->findServerNo(it->fileNo());
        if (si)
            servers.append(si);
    }
    copyServerToClipboard(servers, URL);
}

void ServerPage::actionServerCopyHTML()
{
    QPtrList<ServerInfo> servers;
    QPtrList<QListViewItem> list = serverView->selectedItems();
    ServerInfoItem* it;
    for (it = (ServerInfoItem*)list.first(); it; it = (ServerInfoItem*)list.next()) {
	ServerInfo* si = KMLDonkey::App->donkey->findServerNo(it->fileNo());
        if (si)
            servers.append(si);
    }
    copyServerToClipboard(servers, HTML);
}

void ServerPage::actionShowAllServers(bool show)
{
    showAllServersAction->setChecked(show);
    QIntDictIterator<ServerInfo> it( KMLDonkey::App->donkey->connectedServers() );
    for ( ; it.current(); ++it) serverUpdated( it.current()->serverNo() );
}

void ServerPage::serverUpdated(int server)
{
    ServerInfo* si = KMLDonkey::App->donkey->findServerNo(server);
    ServerInfoItem* df = serverItemList[server];

    if (! si || ( si->serverState() == ServerInfo::Removed ||
                  ! (showAllServersAction->isChecked() ||
                     si->serverState() == ServerInfo::Connected ||
                     si->serverState() == ServerInfo::Connected2 ||
                     si->serverState() == ServerInfo::Connecting ||
                     si->serverState() == ServerInfo::Initiating) )) {
        if (df) serverItemList.remove(server);
        return;
    }

    if (df) {
        df->repaint();
    }
    else {
        df = new ServerInfoItem(serverView, server);
        serverItemList.insert(server, df);
    }
}

void ServerPage::serverRemoved(int server)
{
    serverItemList.remove(server);
}

void ServerPage::actionActivatePage()
{
    KMLDonkey::App->activatePage(this);
}

