/*
 * $Id: infolist.cpp,v 1.14 2003/07/29 00:14:24 gibreel Exp $
 *
 * Copyright (C) 2003 Petter E. Stokke <gibreel@gibreel.net>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * $Log: infolist.cpp,v $
 * Revision 1.14  2003/07/29 00:14:24  gibreel
 * Sorts files by the correct value for the Downloaded column. Closes #4511.
 *
 * Revision 1.13  2003/07/23 18:04:38  gibreel
 * Support for multiple host definitions using HostManager. Configurable
 * charset encoding.
 *
 * Revision 1.12  2003/06/30 23:30:35  gibreel
 * Preliminary friend list support. A ton of updates to the libkmldonkey API to
 * accommodate this, most notably improvements to the search handling
 * necessitated by mldonkey's somewhat awkward reporting of friend shares.
 *
 * Revision 1.11  2003/06/28 21:17:00  gibreel
 * The file info dialog now displays file sources. Libkmldonkey received a
 * number of updates and bugfixes to facilitate this.
 *
 * Revision 1.10  2003/06/19 21:50:45  gibreel
 * Split the various pages off into individual objects, and made a ton of code
 * cleanups, API changes, and not a few bugfixes in the process. The
 * disconnect/reconnect bug, especially, now seems to be gone.
 *
 * Revision 1.9  2003/06/13 18:20:01  gibreel
 * Libkmldonkey now uses references instead of pointers everywhere except where
 * it would cause an obvious performance impact, which should lead to less
 * chance of memory leaks and cleaner code in general. Almost everything that
 * should be const is now also const.
 *
 * Revision 1.8  2003/06/13 09:50:53  dipesh
 * optional coloured server- and search lists
 *
 * Revision 1.7  2003/06/09 18:10:23  gibreel
 * Configurable availability display colours. Extended the DCOP interface and
 * improved the interaction between the GUI and the applet. Added colour for
 * the download list's queued state. Cleanups, bugfixes all round.
 *
 * Revision 1.6  2003/05/30 11:53:00  dipesh
 * Optional colored Downloadlist
 *
 * Revision 1.5  2003/05/27 18:55:15  dipesh
 * QSplitter-Bug workaround
 *
 * Revision 1.4  2003/05/17 15:00:04  dipesh
 * fixed sorting of column "Size" in SearchResultItem and made the sorting
 * case-insensitive in InfoItem
 *
 * Revision 1.3  2003/04/13 17:59:46  gibreel
 * File info dialog and coloured availability bars.
 *
 * Revision 1.2  2003/04/06 21:29:45  gibreel
 * Rudimentary server actions, and all lists now display network type.
 *
 * Revision 1.1  2003/03/23 23:40:01  gibreel
 * Preliminary version of standalone GUI.
 *
 * Revision 1.6  2003/03/10 14:52:02  gibreel
 * Support for GUI protocol 14. Specifically: Authentication with username, new
 * download file state "Queued", and support for the new message types in the
 * DonkeyProtocol class.
 *
 * Revision 1.5  2003/03/10 11:20:38  gibreel
 * Corrected sorting on percentage in the download list. Reported by R.
 * Rodriguez.
 *
 * Revision 1.4  2003/03/08 18:45:02  gibreel
 * ETA display is more helpful when downloaded size exceeds file size.
 *
 * Revision 1.3  2003/03/07 21:51:18  gibreel
 * Changed the MD4 storage in the FileInfo object from an array of int to a
 * QByteArray, added conversion functions between that and QString, and methods
 * to search downloaded files by hash. The applet now uses the file hash
 * instead of the fileno when storing and restoring priorities, as the fileno
 * changes between core restarts.
 *
 * Revision 1.2  2003/03/07 15:11:48  gibreel
 * Added an address column to the server list, and general code cleanup.
 *
 * Revision 1.1.1.1  2003/03/07 11:50:11  gibreel
 * Initial import.
 *
 */

#include <kglobal.h>
#include <klocale.h>
#include <kdebug.h>
#include <klistview.h>
#include <qvariant.h>
#include <qpainter.h>
#include <sys/types.h>
#include <time.h>

#include "infolist.h"
#include "kmldonkey.h"

// InfoList

InfoList::InfoList(QWidget* parent, const char* name)
    : KListView(parent, name)
{
    setSelectionModeExt( KListView::Extended );
    setAllColumnsShowFocus( TRUE );
    setShowSortIndicator( TRUE );
    setResizeMode( KListView::NoColumn );
    setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Expanding);
    setMinimumSize(1,1);
}

InfoList::~InfoList()
{
}

int InfoList::addColumn(const QString& label, int width)
{
    int r = KListView::addColumn(label, width);
    setColumnWidthMode(r, QListView::Manual);
    return r;
}


// InfoItem (base class)

InfoItem::InfoItem( KListView *parent, int file )
    : KListViewItem( parent )
{
    fileno = file;
}

InfoItem::~InfoItem()
{
}

void InfoItem::setFileNo( int file )
{
    fileno = file;
    repaint();
}

int InfoItem::fileNo()
{
    return fileno;
}

double InfoItem::numeric(int) const
{
    return 0.0;
}

bool InfoItem::isNumeric(int) const
{
    return false;
}

int InfoItem::compare( QListViewItem* i, int col, bool ascending ) const
{
    if (isNumeric(col)) {
        double v1 = numeric(col), v2 = ((InfoItem*)i)->numeric(col);
	return (v1 < v2) ? -1 : (v1 > v2) ? 1 : 0;
    } else
	return key( col, ascending ).lower().localeAwareCompare( i->key( col, ascending).lower() );
}


// Utility functions

QString humanReadableSize(int64 rsz)
{
    QString foo;
    double sz = (double)rsz;
    if (sz >= (double)(1024 * 1024 * 1024)) {
	sz = sz / (1024 * 1024 * 1024);
	foo.sprintf(i18n("%.1fG"), sz);
    } else if (sz >= (double)(1024 * 1024)) {
	sz = sz / (1024 * 1024);
	foo.sprintf(i18n("%.1fM"), sz);
    } else if (sz >= (double)1024) {
	sz = sz / 1024;
	foo.sprintf(i18n("%.1fK"), sz);
    } else foo.sprintf("%.0f", sz);
    return foo;
}

QString humanReadableSpeed(double sp)
{
    QString foo;
    if (!sp) foo.sprintf("-");
    else foo.sprintf("%.1f", sp / 1024.0);
    return foo;
}

QString humanReadableTime(time_t t, bool shortFormat)
{
    if (!t) return i18n("0s");
    QString foo;
    int f = 0;
    if (t > 86400) {
	foo += QString::number(t / 86400);
	foo += i18n("d ");
	t %= 86400;
	f = 1;
	if (shortFormat) return foo.simplifyWhiteSpace();
    }
    if (t > 3600) {
	foo += QString::number(t / 3600);
	foo += i18n("h ");
	t %= 3600;
	if (shortFormat) return foo.simplifyWhiteSpace();
    }
    if (t > 60) {
	foo += QString::number(t / 60);
	foo += i18n("m ");
	t %= 60;
	if (shortFormat) return foo.simplifyWhiteSpace();
    }
    if (t && !f) {
	foo += QString::number(t);
	foo += i18n("s");
    }
    return foo.simplifyWhiteSpace();
}

QString calculateETA(FileInfo* fi)
{
    if (fi->fileSize() < fi->fileDownloaded()) {
	return i18n("Overdue");
    }
    if (fi->fileSize() == fi->fileDownloaded()) {
	return i18n("Imminent");
    }
    if (!fi->fileSpeed()) {
	return i18n("-");
    }
    return humanReadableTime((time_t)((double)(fi->fileSize() - fi->fileDownloaded()) / fi->fileSpeed()), false);
}

QString humanReadablePriority(int pri)
{
    if (pri > 0)
	return pri > 10 ? i18n("Very high") : i18n("High");
    if (pri < 0)
	return pri < -10 ? i18n("Very low") : i18n("Low");
    return i18n("Normal");
}


// DownloadFile

QString DownloadFile::text( int column ) const
{
    QString foo;
    FileInfo* it = KMLDonkey::App->donkey->findDownloadFileNo(fileno);
    if (!it) {
	if (!column) return i18n("Unknown file");
	return i18n("-");
    }
    switch (column) {
    case 0: // name
	return it->fileName();
    case 1: // Priority
	return humanReadablePriority(it->filePriority());
    case 4: // size
	return humanReadableSize(it->fileSize());
    case 5: // downloaded
	return humanReadableSize(it->fileDownloaded());
    case 6: // remaining
	foo.sprintf(i18n("%.1f%%"), it->fileSize() ? (it->fileDownloaded() * 100.0) / it->fileSize() : 0.0);
	return foo;
    case 2: // status
	switch (it->fileState()) {
	case FileInfo::Downloading:
	    if (!it->fileSpeed()) return i18n("Looking");
	    return i18n("Downloading");
	case FileInfo::Paused: return i18n("Paused");
	case FileInfo::Complete: return i18n("Complete");
	case FileInfo::Shared: return i18n("Shared");
	case FileInfo::Cancelled: return i18n("Cancelled");
	case FileInfo::New: return i18n("New");
	case FileInfo::Aborted: return QString(i18n("Aborted: ") + it->fileAbortedMsg());
	case FileInfo::Queued: return i18n("Queued");
	}
	return i18n("Unknown");
    case 3: // speed
	return humanReadableSpeed(it->fileSpeed());
    case 7: // eta
	return calculateETA(it);
    case 8: // network
        return KMLDonkey::App->donkey->findNetworkNo(it->fileNetwork())->networkName();
    case 9: // availability
        return QString::null;
    case 10: // hash
	return FileInfo::md4ToString(it->fileMD4());
    case 11: // last seen
	return humanReadableTime((time_t)it->fileLastSeen(), true);
    case 12: // age
	return humanReadableTime(time(0) - (time_t)it->fileAge(), true);
    default:
	return "ERROR!";
    }
}

double DownloadFile::numeric( int col ) const
{
    FileInfo* it = KMLDonkey::App->donkey->findDownloadFileNo(fileno);
    if (!it) return 0.0;
    switch (col) {
    case 1:
	return (double)it->filePriority();
    case 4:
	return (double)it->fileSize();
    case 5:
	return (double)it->fileDownloaded();
    case 6:
	return it->fileSize() ? ((double)it->fileDownloaded() * 100.0) / (double)it->fileSize() : 0.0;
    case 3:
	return it->fileSpeed();
    default:
	return 0.0;
    }
}

bool DownloadFile::isNumeric(int col) const
{
    switch (col) {
    case 1:
    case 3:
    case 4:
    case 5:
    case 6:
	return true;
    default:
	return false;
    }
}

void DownloadFile::paintCell(QPainter* p, const QColorGroup& cg, int col, int w, int align)
{
    if (col == 9) {
        QRect foo(0, 0, w, height());
        p->setBrush(Qt::red);
        p->setPen(Qt::green);
        p->drawRect(foo);
        paintAvailability(*p, foo);
        return;
    }

    QColorGroup colgrp(cg);
    if (KMLDonkey::App->colorDownloadView) {
        FileInfo* it = KMLDonkey::App->donkey->findDownloadFileNo(fileno);
	if (it) {
	    switch (it->fileState()) {
	    case FileInfo::Downloading:
		if (it->fileSpeed() > 0)
		    colgrp.setColor(QColorGroup::Text, KMLDonkey::App->colorDownloadDownloading);
		else
		    colgrp.setColor(QColorGroup::Text, KMLDonkey::App->colorDownloadLooking);
		break;
	    case FileInfo::Paused:
		colgrp.setColor(QColorGroup::Text, KMLDonkey::App->colorDownloadPaused);
		break;
	    case FileInfo::Queued:
		colgrp.setColor(QColorGroup::Text, KMLDonkey::App->colorDownloadQueued);
		break;
	    default:
		break;
	    }
	}
    }

    p->save();
    InfoItem::paintCell(p, colgrp, col, w, align);
    p->restore();
}

int DownloadFile::width(const QFontMetrics& fm, const QListView* lv, int c) const
{
    if (c == 9) return (chunkNo > 100) ? 100 : chunkNo;
    return InfoItem::width(fm, lv, c);
}


// DownloadedFile

QString DownloadedFile::text( int column ) const
{
    FileInfo* it = KMLDonkey::App->donkey->findDownloadedFileNo(fileno);
    if (!it) it = KMLDonkey::App->donkey->findDownloadFileNo(fileno);
    if (!it) {
	if (!column) return i18n("Unknown file");
	return i18n("-");
    }
    switch (column) {
    case 0: // name
	return it->fileName();
    case 1: // size
	return humanReadableSize(it->fileSize());
    case 2: // format
	return it->fileFormatInfo();
    case 3: // network
        return KMLDonkey::App->donkey->findNetworkNo(it->fileNetwork())->networkName();
    case 4: // hash
	return FileInfo::md4ToString(it->fileMD4());
    default:
	return "ERROR!";
    }
}

double DownloadedFile::numeric( int col ) const
{
    FileInfo* it = KMLDonkey::App->donkey->findDownloadedFileNo(fileno);
    if (!it) it = KMLDonkey::App->donkey->findDownloadFileNo(fileno);
    if (!it) return 0.0;
    switch (col) {
    case 1:
	return (double)it->fileSize();
    default:
	return 0.0;
    }
}

bool DownloadedFile::isNumeric(int col) const
{
    switch (col) {
    case 1:
	return true;
    default:
	return false;
    }
}


// ServerInfoItem

QString ServerInfoItem::text( int column ) const
{
    ServerInfo* it = KMLDonkey::App->donkey->findServerNo(fileno);
    if (!it) {
	if (!column) return i18n("Unknown server");
	return i18n("-");
    }
    switch (column) {
    case 0: // name
	return it->serverName();
    case 1: // network
        return KMLDonkey::App->donkey->findNetworkNo(it->serverNetwork())->networkName();
    case 2: // status
	switch (it->serverState()) {
	case ServerInfo::NotConnected:
	case ServerInfo::NotConnected2: return i18n("Not connected");
	case ServerInfo::Connecting: return i18n("Connecting");
	case ServerInfo::Initiating: return i18n("Initiating");
	case ServerInfo::Downloading: return i18n("Downloading");
	case ServerInfo::Connected:
	case ServerInfo::Connected2: return i18n("Connected");
	case ServerInfo::NewHost: return i18n("New host");
	case ServerInfo::Removed: return i18n("Removed");
	case ServerInfo::Blacklisted: return i18n("Blacklisted");
	default: return i18n("Unknown");
	}
    case 3: // users
	return QString::number(it->serverNUsers());
    case 4: // files
	return QString::number(it->serverNFiles());
    case 5: // description
	return it->serverDescription();
    case 6: // address
	return QString(it->serverAddress() + ":" + QString::number(it->serverPort()));
    default:
	return "ERROR!";
    }
}

double ServerInfoItem::numeric( int col ) const
{
    ServerInfo* it = KMLDonkey::App->donkey->findServerNo(fileno);
    if (!it) return 0.0;
    switch (col) {
    case 3:
	return (double)it->serverNUsers();
    case 4:
	return (double)it->serverNFiles();
    default:
	return 0.0;
    }
}

bool ServerInfoItem::isNumeric(int col) const
{
    switch (col) {
    case 3:
    case 4:
	return true;
    default:
	return false;
    }
}

void ServerInfoItem::paintCell(QPainter* p, const QColorGroup& cg, int col, int w, int align)
{
    QColorGroup colgrp(cg);
    if (KMLDonkey::App->colorServerView) {
        ServerInfo* it = KMLDonkey::App->donkey->findServerNo(fileno);
	if (it) {
	    switch (it->serverState()) {
	    case ServerInfo::NotConnected:
		colgrp.setColor(QColorGroup::Text, KMLDonkey::App->colorServerNotConnected);
		break;
	    case ServerInfo::Blacklisted:
		colgrp.setColor(QColorGroup::Text, KMLDonkey::App->colorServerBlacklisted);
		break;
	    case ServerInfo::Connecting:
	    case ServerInfo::Initiating:
		colgrp.setColor(QColorGroup::Text, KMLDonkey::App->colorServerConnecting);
		break;
	    case ServerInfo::Connected:
	    case ServerInfo::Connected2:
		colgrp.setColor(QColorGroup::Text, KMLDonkey::App->colorServerConnected);
		break;
	    default:
		break;
	    }
	}
    }

    p->save();
    InfoItem::paintCell(p, colgrp, col, w, align);
    p->restore();
}

// SharedFile

QString SharedFile::text( int column ) const
{
    ShareInfo* it = KMLDonkey::App->donkey->findShareNo(fileno);
    if (!it) {
	if (!column) return i18n("Unknown share");
	return i18n("-");
    }
    switch (column) {
    case 0: // name
	return it->shareName();
    case 1: // network
        return KMLDonkey::App->donkey->findNetworkNo(it->shareNetwork())->networkName();
    case 2: // requests
        return QString::number(it->shareRequests());
    case 3: // bytes
	return humanReadableSize(it->shareUploaded());
    default:
	return "ERROR!";
    }
}

double SharedFile::numeric( int col ) const
{
    ShareInfo* it = KMLDonkey::App->donkey->findShareNo(fileno);
    if (!it) return 0.0;
    switch (col) {
    case 2:
	return (double)it->shareRequests();
    case 3:
	return (double)it->shareUploaded();
    default:
	return 0.0;
    }
}

bool SharedFile::isNumeric(int col) const
{
    switch (col) {
    case 2:
    case 3:
	return true;
    default:
	return false;
    }
}

// ClientItem

void ClientItem::setFilesListed(bool listed)
{
    filesListed = listed; repaint();
}

QString ClientItem::text( int column ) const
{
    ClientInfo* it = KMLDonkey::App->donkey->findClientNo(fileno);
    if (!it) {
	if (!column) return i18n("Unknown client");
	return i18n("-");
    }
    switch (column) {
    case 0: // name
	return it->clientName();
    case 1: // network
        return KMLDonkey::App->donkey->findNetworkNo(it->clientNetwork())->networkName();
    case 2: // type
	switch (it->clientType()) {
	case ClientInfo::NormalClient:
	    return i18n("Normal");
	case ClientInfo::FriendClient:
	    return i18n("Friend");
	case ClientInfo::ContactClient:
	    return i18n("Contact");
	default:
	    return i18n("Unknown");
	}
    case 3: // kind
	return it->clientKind();
    case 4: // state
	switch (it->clientState()) {
	case ClientInfo::NotConnected:
	case ClientInfo::NotConnected2: return i18n("Not connected");
	case ClientInfo::Connecting: return i18n("Connecting");
	case ClientInfo::Initiating: return i18n("Initiating");
	case ClientInfo::Downloading: return filesListed ? i18n("Files listed") : i18n("Downloading");
	case ClientInfo::Connected: return filesListed ? i18n("Files listed") : i18n("Connected");
	case ClientInfo::Connected2: return filesListed ? i18n("Files listed") : (i18n("Queued: ") + QString::number(it->clientQueuePosition()));
	case ClientInfo::NewHost: return i18n("New host");
	case ClientInfo::Removed: return i18n("Removed");
	case ClientInfo::Blacklisted: return i18n("Blacklisted");
	default: return i18n("Unknown: ") + QString::number(it->clientState());
	}
    default:
	return "ERROR!";
    }
}

void ClientItem::paintCell(QPainter* p, const QColorGroup& cg, int col, int w, int align)
{
    QColorGroup colgrp(cg);
    if (KMLDonkey::App->colorSourceView) {
	ClientInfo* cl = KMLDonkey::App->donkey->findClientNo(fileno);
	if (cl) {
	    switch (cl->clientState()) {
	    case ClientInfo::NotConnected:
	    case ClientInfo::NotConnected2:
		colgrp.setColor(QColorGroup::Text, KMLDonkey::App->colorSourceNotConnected);
		break;
	    case ClientInfo::Connecting:
	    case ClientInfo::Initiating:
		colgrp.setColor(QColorGroup::Text, KMLDonkey::App->colorSourceConnecting);
		break;
	    case ClientInfo::Downloading:
		colgrp.setColor(QColorGroup::Text, KMLDonkey::App->colorSourceDownloading);
		break;
	    case ClientInfo::Connected:
	    case ClientInfo::Connected2:
		colgrp.setColor(QColorGroup::Text, KMLDonkey::App->colorSourceQueued);
		break;
	    case ClientInfo::Blacklisted:
		colgrp.setColor(QColorGroup::Text, KMLDonkey::App->colorSourceBlacklisted);
		break;
	    default:
		break;
	    }
	}
    }

    p->save();
    InfoItem::paintCell(p, colgrp, col, w, align);
    p->restore();
}

// ClientFile

QString ClientFile::text( int column ) const
{
    const ResultInfo* it = KMLDonkey::App->donkey->findClientFile(fileno);
    if (!it) {
	if (!column) return i18n("Unknown file");
	return i18n("-");
    }
    switch (column) {
    case 0: // name
	return it->resultName();
    case 1: // network
        return KMLDonkey::App->donkey->findNetworkNo(it->resultNetwork())->networkName();
    case 2: // size
	return humanReadableSize(it->resultSize());
    case 3: // format
	return it->resultFormat();
    case 4: // comment
	return it->resultComment();
    case 5: // hash
	return FileInfo::md4ToString(it->resultMD4());
    default:
	return "ERROR!";
    }
}

double ClientFile::numeric( int col ) const
{
    const ResultInfo* it = KMLDonkey::App->donkey->findClientFile(fileno);
    if (!it) return 0.0;
    switch (col) {
    case 2:
	return (double)it->resultSize();
    default:
	return 0.0;
    }
}

bool ClientFile::isNumeric(int col) const
{
    switch (col) {
    case 2:
	return true;
    default:
	return false;
    }
}
