/* -*- c++ -*-
 *
 * $Id: friendpage.cpp,v 1.8 2003/07/27 20:24:26 gibreel Exp $
 *
 * Copyright (C) 2003 Petter E. Stokke <gibreel@gibreel.net>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * $Log: friendpage.cpp,v $
 * Revision 1.8  2003/07/27 20:24:26  gibreel
 * Added KActions for switching between pages.
 *
 * Revision 1.7  2003/07/19 20:20:32  gibreel
 * Uses autoconf to detect the presence of KTabWidget, rather than guessing its
 * presence using KDE_IS_VERSION.
 *
 * Revision 1.6  2003/07/19 14:18:22  gibreel
 * Uses KTabWidget instead of QTabWidget/ClosableTabWidget if available (KDE
 * 3.2 or CVS). ClosableTabWidget API changed to match KTabWidget's.
 *
 * Revision 1.5  2003/07/17 20:45:49  gibreel
 * Added copy icons to all copy actions.
 *
 * Revision 1.4  2003/07/17 20:39:50  gibreel
 * Added actions to the friend file lists, so they're actually good for
 * something besides spying for the RIAA.
 *
 * Revision 1.3  2003/07/17 19:55:07  gibreel
 * Implemented multiple friend file view tabs.
 *
 * Revision 1.2  2003/07/16 22:41:32  dipesh
 * Clear fileView on removing showing friend
 *
 * Revision 1.1  2003/06/30 23:30:35  gibreel
 * Preliminary friend list support. A ton of updates to the libkmldonkey API to
 * accommodate this, most notably improvements to the search handling
 * necessitated by mldonkey's somewhat awkward reporting of friend shares.
 *
 */

#include <kdebug.h>
#include <klocale.h>
#include <kconfig.h>
#include <kaction.h>
#include <klineeditdlg.h>
#include <kiconloader.h>
#include <qintdict.h>
#include <qpopupmenu.h>
#include <qclipboard.h>
#include <qtimer.h>

#include <kdeversion.h>

#include <donkeyprotocol.h>

#include "infolist.h"
#include "prefs.h"
#include "kmldonkey.h"
#include "friendpage.h"
#include "friendpage.moc"

#ifdef HAVE_KTABWIDGET
#include <ktabwidget.h>
#else
#include "closabletab.h"
#endif

FriendPage::FriendPage(QWidget* parent)
    : QVBox(parent, "friendPage")
    , KMLDonkeyPage()
    , ClipboardHelper()
{
    friendList.setAutoDelete(true);
    shares.setAutoDelete(true);
    fileViews.setAutoDelete(true);

    friendHSplitter = new QSplitter(this, "friendHSplitter");
    friendHSplitter->setOrientation(QSplitter::Horizontal);

    friendView = new InfoList(friendHSplitter, "friendView");
    friendView->addColumn( i18n( "Name" ) );
    friendView->addColumn( i18n( "Network" ) );
    friendView->addColumn( i18n( "Type" ) );
    friendView->addColumn( i18n( "Location" ) );
    friendView->addColumn( i18n( "State" ) );

    friendVSplitter = new QSplitter(friendHSplitter, "friendVSplitter");
    friendVSplitter->setOrientation(QSplitter::Vertical);

#ifdef HAVE_KTABWIDGET
    fileTab = new KTabWidget(friendVSplitter, "fileTab");
    fileTab->setHoverCloseButton(true);
    fileTab->setTabReorderingEnabled(true);
#else
    fileTab = new ClosableTabWidget(friendVSplitter, "fileTab");
#endif

    new QWidget(friendVSplitter);

    QTimer* timer = new QTimer(this);
    connect(timer, SIGNAL(timeout()), this, SLOT(mightUpdateShareList()));
    timer->start(100);

    connect(friendView, SIGNAL(contextMenu(KListView*, QListViewItem*, const QPoint&)),
	    this, SLOT(contextFriend(KListView*, QListViewItem*, const QPoint&)));
    connect(friendView, SIGNAL(executed(QListViewItem*)), this, SLOT(openFriendFiles(QListViewItem*)));

    connect(fileTab, SIGNAL(closeRequest(QWidget*)), this, SLOT(closeFileList(QWidget*)));

    connect(KMLDonkey::App->donkey, SIGNAL(friendUpdated(int)), this, SLOT(friendUpdated(int)));
    connect(KMLDonkey::App->donkey, SIGNAL(friendRemoved(int)), this, SLOT(friendRemoved(int)));
    connect(KMLDonkey::App->donkey, SIGNAL(clientFileListing(int,const QString&,int)), this, SLOT(receiveShare(int,const QString&,int)));
}

void FriendPage::setupActions(KActionCollection* actionCollection)
{
    (void)new KAction(i18n("&Search for friend"), "find", 0, this, SLOT(actionAddFriend()),
                      actionCollection, "add_friend");
    (void)new KAction(i18n("&Connect to friend"), "connecting", 0, this, SLOT(actionConnectFriend()),
                      actionCollection, "connect_friend");
    (void)new KAction(i18n("&Remove friend"), "edit_remove", 0, this, SLOT(actionRemoveFriend()),
                      actionCollection, "remove_friend");


    (void)new KAction(i18n("&Download"), "down", 0, this, SLOT(actionDownload()),
                      actionCollection, "friend_share_download");
    (void)new KAction(i18n("&Force download"), "down", 0, this, SLOT(actionForceDownload()),
                      actionCollection, "friend_share_force_download");
    (void)new KAction(i18n("Download &as..."), "down", 0, this, SLOT(actionDownloadAs()),
                      actionCollection, "friend_share_download_as");

    (void)new KAction(i18n("Copy search &URL to clipboard"), "editcopy", 0, this, SLOT(actionCopyURL()),
                      actionCollection, "friend_share_copy_url");
    (void)new KAction(i18n("Copy search HTML to clipboard"), "editcopy", 0, this, SLOT(actionCopyHTML()),
                      actionCollection, "friend_share_copy_html");
    (void)new KAction(i18n("Copy search hash to clipboard"), "editcopy", 0, this, SLOT(actionCopyHash()),
                      actionCollection, "friend_share_copy_hash");

    (void)new KAction(i18n("Activate friends page"), 0, 0, this, SLOT(actionActivatePage()),
		      actionCollection, "activate_page_friends");
}

void FriendPage::applyPreferences(KMLDonkeyPreferences*)
{
    if (KMLDonkey::App->listFont !=  friendView->font())
        friendView->setFont(KMLDonkey::App->listFont);

    QIntDictIterator<InfoList> it(fileViews);
    for (; it.current(); ++it)
	if (KMLDonkey::App->listFont !=  it.current()->font())
	    it.current()->setFont(KMLDonkey::App->listFont);
}

void FriendPage::clear()
{
    closeAllFileLists();
    friendList.clear();
    shares.clear();
    friendView->clear();
}

void FriendPage::restoreState(KConfig* conf)
{
    conf->setGroup("Splitters");
    friendHSplitter->setSizes(conf->readIntListEntry("FriendHSplitter"));
    friendVSplitter->setSizes(conf->readIntListEntry("FriendVSplitter"));

    friendView->restoreLayout(conf, "FriendView");

    applyPreferences();
}

void FriendPage::saveState(KConfig* conf)
{
    conf->setGroup("Splitters");
    conf->writeEntry("FriendHSplitter", friendHSplitter->sizes());
    conf->writeEntry("FriendVSplitter", friendVSplitter->sizes());

    friendView->saveLayout(conf, "FriendView");

    QIntDictIterator<InfoList> it(fileViews);
    if (it.current())
	it.current()->saveLayout(KGlobal::config(), "FriendFileView");
}

void FriendPage::closeAllFileLists()
{
    fileViews.clear();
    pleaseUpdateShareList.clear();
}

void FriendPage::openFileList(int client)
{
    InfoList* fileView = fileViews.find(client);
    if (!fileView) {
	ClientInfo* cl = KMLDonkey::App->donkey->findClientNo(client);
	fileView = new InfoList(fileTab, "fileView");
	fileView->setFont(KMLDonkey::App->listFont);
	fileView->addColumn( i18n( "Filename" ) );
	fileView->addColumn( i18n( "Network" ) );
	fileView->addColumn( i18n( "Size" ) );
	fileView->addColumn( i18n( "Format" ) );
	fileView->addColumn( i18n( "Comment" ) );
	fileView->addColumn( i18n( "Hash" ) );
	fileView->restoreLayout(KGlobal::config(), "FriendFileView");
	fileTab->addTab(fileView, KGlobal::iconLoader()->loadIconSet("personal", KIcon::Small), cl->clientName());
	fileViews.insert(client, fileView);
	constructFileList(client);
	connect(fileView, SIGNAL(contextMenu(KListView*,QListViewItem*,const QPoint&)),
		this, SLOT(contextFile(KListView*,QListViewItem*,const QPoint&)));
    }
    fileTab->showPage(fileView);
}

void FriendPage::closeFileList(QWidget* widget)
{
    InfoList* fileView = dynamic_cast<InfoList*>(widget);
    QIntDictIterator<InfoList> it(fileViews);
    for (; it.current(); ++it)
	if (it.current() == fileView) {
	    it.current()->saveLayout(KGlobal::config(), "FriendFileView");
	    fileViews.remove(it.currentKey());
	    return;
	}
}

void FriendPage::friendUpdated(int client)
{
    ClientInfo* cl = KMLDonkey::App->donkey->findClientNo(client);
    if (!cl) return;

    ClientItem* it = friendList[client];
    if (it) {
        it->repaint();
        return;
    }
    it = new ClientItem(friendView, client);
    friendList.insert(client, it);
}

void FriendPage::friendRemoved(int client)
{
    friendList.remove(client);
    shares.remove(client);
    fileViews.remove(client);
}

void FriendPage::contextFriend(KListView*, QListViewItem*, const QPoint& pt)
{
    QPopupMenu *pop = (QPopupMenu*)(KMLDonkey::App->factory())->container("friend_actions", KMLDonkey::App);
    pop->popup(pt);
}

void FriendPage::contextFile(KListView*, QListViewItem*, const QPoint& pt)
{
    QPopupMenu *pop = (QPopupMenu*)(KMLDonkey::App->factory())->container("friend_share_actions", KMLDonkey::App);
    pop->popup(pt);
}

void FriendPage::actionAddFriend()
{
    bool ok;
    QString name = KLineEditDlg::getText( i18n("Search for friend"),
					  i18n("Enter the name of the friend you are searching for."),
					  QString::null, &ok, this );
    if (ok)
	KMLDonkey::App->donkey->searchForFriend(name);
}

void FriendPage::actionRemoveFriend()
{
    QPtrList<QListViewItem> list = friendView->selectedItems();
    ClientItem* it;
    for (it = (ClientItem*)list.first(); it; it = (ClientItem*)list.next())
	KMLDonkey::App->donkey->removeFriend(it->fileNo());
}

void FriendPage::actionConnectFriend()
{
    QPtrList<QListViewItem> list = friendView->selectedItems();
    ClientItem* it;
    for (it = (ClientItem*)list.first(); it; it = (ClientItem*)list.next())
	KMLDonkey::App->donkey->connectFriend(it->fileNo());
}

void FriendPage::receiveShare(int clno, const QString& dir, int result)
{
    FriendShares* sh = shares[clno];
    if (!sh) {
	shares.insert(clno, new FriendShares());
	sh = shares[clno];
	QPtrList<QListViewItem> list = friendView->selectedItems();
	ClientItem* it;
	for (it = (ClientItem*)list.first(); it; it = (ClientItem*)list.next())
	    if (it->fileNo() == clno) it->setFilesListed(true);
    }

    const ResultInfo* si = KMLDonkey::App->donkey->findClientFile(result);
    if (!si) { kdDebug() << "No result #" << result << " available!" << endl; return; }

    QString dirname = dir;
    if (dirname.isEmpty()) dirname = ".";
    QValueList<int>& files = sh->dirs[dirname];
    if (!files.contains(si->resultNo()))
	files.append(si->resultNo());
    if (fileViews.find(clno))
	pleaseUpdateShareList.insert(clno, true);
}

void FriendPage::constructFileList(int clno)
{
    InfoList* fileView = fileViews.find(clno);
    if (!fileView) return;

    fileView->clear();
    FriendShares* sh = shares[clno];
    if (!sh) {
	DonkeyMessage out(DonkeyProtocol::GetClient_files);
	out.writeInt32(clno);
	KMLDonkey::App->donkey->sendMessage(out);
	return;
    }

    QMap<QString,QValueList<int> >::iterator it;
    for (it = sh->dirs.begin(); it != sh->dirs.end(); ++it) {
	QValueList<int>::iterator lit;
	for (lit = it.data().begin(); lit != it.data().end(); ++lit) {
	    new ClientFile(fileView, *lit);
	}
    }
}

void FriendPage::openFriendFiles(QListViewItem* item)
{
    ClientItem* cl = (ClientItem*)item;
    openFileList(cl->fileNo());
}

void FriendPage::mightUpdateShareList()
{
    QMap<int,bool>::Iterator it;
    for (it = pleaseUpdateShareList.begin(); it != pleaseUpdateShareList.end(); ++it)
	if (it.data())
	    constructFileList(it.key());
    pleaseUpdateShareList.clear();
}

QPtrList<QListViewItem> FriendPage::currentFileListSelection()
{
    QPtrList<QListViewItem> list;
    InfoList* fileView = dynamic_cast<InfoList*>(fileTab->currentPage());
    if (!fileView) return list;
    list = fileView->selectedItems();
    return list;
}

void FriendPage::downloadSelected(bool force, bool ask)
{
    QPtrList<QListViewItem> list = currentFileListSelection();
    ClientFile* it;
    for (it = (ClientFile*)list.first(); it; it = (ClientFile*)list.next()) {
	const ResultInfo* fi = KMLDonkey::App->donkey->findClientFile(it->fileNo());
	if (!fi) continue;
	QStringList names;
	if (ask || !fi->resultName().length()) {
	    bool ok;
            QString filename = KLineEditDlg::getText( i18n("Download as..."),
						      i18n("Choose a filename for the new download"),
						      fi->resultName(), &ok, this );
            if (!ok) continue;
	    names.append(filename);
	    if (filename != fi->resultName())
		names.append(fi->resultName());
	} else {
	    names.append(fi->resultName());
	}
	KMLDonkey::App->donkey->startDownload(names, fi->resultNo(), force);
    }
}

void FriendPage::actionDownload()
{
    downloadSelected(false, false);
}

void FriendPage::actionForceDownload()
{
    downloadSelected(true, false);
}

void FriendPage::actionDownloadAs()
{
    downloadSelected(false, true);
}


void FriendPage::copyFileToClipboard(ClipFormat format)
{
    QStringList l;
    QPtrList<QListViewItem> list = currentFileListSelection();
    ClientFile* it;
    for (it = (ClientFile*)list.first(); it; it = (ClientFile*)list.next()) {
	const ResultInfo* fi = KMLDonkey::App->donkey->findClientFile(it->fileNo());
	if (!fi) continue;
        l.append(fi->resultName());
        l.append(FileInfo::md4ToString(fi->resultMD4()));
        l.append(QString::number((long int)fi->resultSize()));
    }
    copyToClipboard(l, format);
}

void FriendPage::actionCopyURL() { copyFileToClipboard(URL); }
void FriendPage::actionCopyHTML() { copyFileToClipboard(HTML); }
void FriendPage::actionCopyHash() { copyFileToClipboard(Hash); }

void FriendPage::actionActivatePage()
{
    KMLDonkey::App->activatePage(this);
}

