/* -*- c++ -*-
 *
 * $Id: downloadpage.cpp,v 1.6 2003/07/27 20:24:26 gibreel Exp $
 *
 * Copyright (C) 2003 Petter E. Stokke <gibreel@gibreel.net>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * $Log: downloadpage.cpp,v $
 * Revision 1.6  2003/07/27 20:24:26  gibreel
 * Added KActions for switching between pages.
 *
 * Revision 1.5  2003/07/23 18:04:38  gibreel
 * Support for multiple host definitions using HostManager. Configurable
 * charset encoding.
 *
 * Revision 1.4  2003/07/17 20:45:49  gibreel
 * Added copy icons to all copy actions.
 *
 * Revision 1.3  2003/06/30 14:59:22  gibreel
 * Updated lists to support libkmldonkey's new removed signals. Moved pages'
 * connect statements out of KMLDonkey's constructor into their own
 * constructors. Added a debug console displaying dumps of unhandled messages.
 *
 * Revision 1.2  2003/06/19 22:21:09  gibreel
 * A saner fix for the disconnect/reconnect bug, which in turn solves a bug
 * introduced by the previous fix.
 *
 * Revision 1.1  2003/06/19 21:50:45  gibreel
 * Split the various pages off into individual objects, and made a ton of code
 * cleanups, API changes, and not a few bugfixes in the process. The
 * disconnect/reconnect bug, especially, now seems to be gone.
 *
 */

#include <kdebug.h>
#include <klocale.h>
#include <kconfig.h>
#include <klistview.h>
#include <kmessagebox.h>
#include <kaction.h>
#include <qsplitter.h>
#include <qpopupmenu.h>

#include "kmldonkey.h"
#include "infolist.h"
#include "infodialog.h"
#include "renamedialog.h"
#include "downloadpage.h"
#include "downloadpage.moc"

DownloadPage::DownloadPage(QWidget* parent)
    : QVBox(parent, "downloadPage")
    , KMLDonkeyPage()
    , ClipboardHelper()
{
    downloadFileList.setAutoDelete(true);
    downloadedFileList.setAutoDelete(true);

    downloadSplitter = new QSplitter((QWidget*)this, "downloadSplitter");
    downloadSplitter->setOrientation(QSplitter::Vertical);

    downloadView = new InfoList(downloadSplitter, "downloadView");
    downloadView->addColumn( i18n( "File name" ) );
    downloadView->addColumn( i18n( "Priority" ) );
    downloadView->addColumn( i18n( "Status" ) );
    downloadView->addColumn( i18n( "Speed" ) );
    downloadView->addColumn( i18n( "Size" ) );
    downloadView->addColumn( i18n( "Downloaded" ) );
    downloadView->addColumn( i18n( "%" ) );
    downloadView->addColumn( i18n( "ETA" ) );
    downloadView->addColumn( i18n( "Network" ) );
    downloadView->addColumn( i18n( "Availability" ) );
    downloadView->addColumn( i18n( "Hash" ) );
    downloadView->addColumn( i18n( "Last seen" ) );
    downloadView->addColumn( i18n( "Age" ) );

    downloadedView = new InfoList( downloadSplitter, "downloadedView" );
    downloadedView->addColumn( i18n( "File name" ) );
    downloadedView->addColumn( i18n( "Size" ) );
    downloadedView->addColumn( i18n( "Format" ) );
    downloadedView->addColumn( i18n( "Network" ) );
    downloadedView->addColumn( i18n( "Hash" ) );

    connect(downloadView, SIGNAL(contextMenu(KListView*, QListViewItem*, const QPoint&)),
	    this, SLOT(contextDownload(KListView*, QListViewItem*, const QPoint&)));
    connect(downloadView, SIGNAL(doubleClicked(QListViewItem*, const QPoint&, int)),
            this, SLOT(downloadDoubleClick(QListViewItem*, const QPoint&, int)));
    connect(downloadedView, SIGNAL(contextMenu(KListView*, QListViewItem*, const QPoint&)),
	    this, SLOT(contextDownloaded(KListView*, QListViewItem*, const QPoint&)));

    connect(KMLDonkey::App->donkey, SIGNAL(fileUpdated(int)), this, SLOT(fileUpdated(int)));
    connect(KMLDonkey::App->donkey, SIGNAL(fileRemoved(int)), this, SLOT(fileRemoved(int)));
}

void DownloadPage::setupActions(KActionCollection* actionCollection)
{
    (void)new KAction(i18n("&Pause download"), "player_pause", 0, this, SLOT(actionPauseFile()),
		      actionCollection, "pause_selected");
    (void)new KAction(i18n("&Resume download"), "player_play", 0, this, SLOT(actionResumeFile()),
		      actionCollection, "resume_selected");
    (void)new KAction(i18n("&Cancel download"), "cancel", 0, this, SLOT(actionCancelFile()),
		      actionCollection, "cancel_selected");
    (void)new KAction(i18n("Re&name download"), "filesaveas", 0, this, SLOT(actionRenameFile()),
		      actionCollection, "rename_selected");
    (void)new KAction(i18n("Re&try connect"), "reload", 0, this, SLOT(actionRetryConnect()),
                      actionCollection, "retry_selected");
    (void)new KAction(i18n("Pre&view file"), "thumbnail", 0, this, SLOT(actionPreviewFile()),
                      actionCollection, "preview_selected");
    (void)new KAction(i18n("Verify chunks"), NULL, 0, this, SLOT(actionVerifyFile()),
                      actionCollection, "verify_selected");
    (void)new KAction(i18n("Get &format info"), NULL, 0, this, SLOT(actionGetFileFormat()),
                      actionCollection, "get_format_selected");
    (void)new KAction(i18n("File &information"), "viewmag", 0, this, SLOT(actionFileInformation()),
                      actionCollection, "file_information");
    (void)new KAction(i18n("Copy download &URL to clipboard"), "editcopy", 0, this, SLOT(actionFileCopyURL()),
                      actionCollection, "file_copy_url");
    (void)new KAction(i18n("Copy download HTML to clipboard"), "editcopy", 0, this, SLOT(actionFileCopyHTML()),
                      actionCollection, "file_copy_html");
    (void)new KAction(i18n("Copy download hash to clipboard"), "editcopy", 0, this, SLOT(actionFileCopyHash()),
                      actionCollection, "file_copy_hash");

    KActionMenu* priMenu = new KActionMenu(i18n("Priorit&y"), "flag", actionCollection, "priority_menu");
    priMenu->setDelayed(false);

    priMenu->insert(new KAction(i18n("Very l&ow priority"), "2leftarrow", 0, this, SLOT(actionFilePriorityVeryLow()),
                      actionCollection, "priority_very_low"));
    priMenu->insert(new KAction(i18n("&Low priority"), "1leftarrow", 0, this, SLOT(actionFilePriorityLow()),
                      actionCollection, "priority_low"));
    priMenu->insert(new KAction(i18n("&Normal priority"), "1uparrow", 0, this, SLOT(actionFilePriorityNormal()),
                      actionCollection, "priority_normal"));
    priMenu->insert(new KAction(i18n("&High priority"), "1rightarrow", 0, this, SLOT(actionFilePriorityHigh()),
                      actionCollection, "priority_high"));
    priMenu->insert(new KAction(i18n("&Very high priority"), "2rightarrow", 0, this, SLOT(actionFilePriorityVeryHigh()),
                      actionCollection, "priority_very_high"));

    (void)new KAction(i18n("&Commit"), "filesave", 0, this, SLOT(actionCommitFile()),
                      actionCollection, "commit_selected");
    (void)new KAction(i18n("Commit &as..."), "filesaveas", 0, this, SLOT(actionCommitFileAs()),
                      actionCollection, "commit_as_selected");
    (void)new KAction(i18n("Copy file &URL to clipboard"), "editcopy", 0, this, SLOT(actionCompleteCopyURL()),
                      actionCollection, "complete_copy_url");
    (void)new KAction(i18n("Copy file HTML to clipboard"), "editcopy", 0, this, SLOT(actionCompleteCopyHTML()),
                      actionCollection, "complete_copy_html");
    (void)new KAction(i18n("Copy file hash to clipboard"), "editcopy", 0, this, SLOT(actionCompleteCopyHash()),
                      actionCollection, "complete_copy_hash");

    (void)new KAction(i18n("Activate download page"), 0, 0, this, SLOT(actionActivatePage()),
		      actionCollection, "activate_page_download");
}

void DownloadPage::clear()
{
    downloadedFileList.clear();
    downloadFileList.clear();
    downloadView->clear();
    downloadedView->clear();
}

void DownloadPage::applyPreferences(KMLDonkeyPreferences*)
{
    const QFont& font = KMLDonkey::App->listFont;

    if (downloadView->font() != font) downloadView->setFont(font);
    if (downloadedView->font() != font) downloadedView->setFont(font);
}

void DownloadPage::saveState(KConfig* conf)
{
    conf->setGroup("Splitters");
    conf->writeEntry("DownloadSplitter", downloadSplitter->sizes());

    downloadView->saveLayout(conf, "DownloadView");
    downloadedView->saveLayout(conf, "DownloadedView");
}

void DownloadPage::restoreState(KConfig* conf)
{
    conf->setGroup("Splitters");
    downloadSplitter->setSizes(conf->readIntListEntry("DownloadSplitter"));

    downloadView->restoreLayout(conf, "DownloadView");
    downloadedView->restoreLayout(conf, "DownloadedView");

    applyPreferences();
}

void DownloadPage::contextDownload(KListView*,QListViewItem*,const QPoint& pt)
{
    QPopupMenu *pop = (QPopupMenu*)(KMLDonkey::App->factory())->container("download_actions", KMLDonkey::App);
    pop->popup(pt);
}

void DownloadPage::contextDownloaded(KListView*,QListViewItem*,const QPoint& pt)
{
    QPopupMenu *pop = (QPopupMenu*)(KMLDonkey::App->factory())->container("downloaded_actions", KMLDonkey::App);
    pop->popup(pt);
}

void DownloadPage::downloadDoubleClick(QListViewItem*, const QPoint&, int)
{
    actionFileInformation();
}

void DownloadPage::actionActivatePage()
{
    KMLDonkey::App->activatePage(this);
}

void DownloadPage::actionPauseFile()
{
    QPtrList<QListViewItem> list = downloadView->selectedItems();
    DownloadFile* it;
    for (it = (DownloadFile*)list.first(); it; it = (DownloadFile*)list.next()) {
	FileInfo* fi = KMLDonkey::App->donkey->findDownloadFileNo(it->fileNo());
	if (!fi) continue;
	KMLDonkey::App->donkey->pauseFile(fi->fileNo(), 1);
    }
}

void DownloadPage::actionResumeFile()
{
    QPtrList<QListViewItem> list = downloadView->selectedItems();
    DownloadFile* it;
    for (it = (DownloadFile*)list.first(); it; it = (DownloadFile*)list.next()) {
	FileInfo* fi = KMLDonkey::App->donkey->findDownloadFileNo(it->fileNo());
	if (!fi) continue;
	KMLDonkey::App->donkey->pauseFile(fi->fileNo(), 0);
    }
}

void DownloadPage::actionCancelFile()
{
    QPtrList<QListViewItem> list = downloadView->selectedItems();
    DownloadFile* it;
    QString msg = i18n("Are you sure you want to cancel these files?\n\n");
    for (it = (DownloadFile*)list.first(); it; it = (DownloadFile*)list.next()) {
	FileInfo* fi = KMLDonkey::App->donkey->findDownloadFileNo(it->fileNo());
	if (!fi) continue;
	msg.append(fi->fileName());
	msg.append("\n");
    }
    if (KMessageBox::warningYesNo(this, msg, i18n("Cancel downloads")) == KMessageBox::Yes) {
	for (it = (DownloadFile*)list.first(); it; it = (DownloadFile*)list.next()) {
	    FileInfo* fi = KMLDonkey::App->donkey->findDownloadFileNo(it->fileNo());
	    if (!fi) continue;
	    KMLDonkey::App->donkey->cancelFile(fi->fileNo());
	}
    }
}

void DownloadPage::setFilePriority(int pri)
{
    QPtrList<QListViewItem> list = downloadView->selectedItems();
    DownloadFile* it;
    for (it = (DownloadFile*)list.first(); it; it = (DownloadFile*)list.next()) {
	FileInfo* fi = KMLDonkey::App->donkey->findDownloadFileNo(it->fileNo());
	if (!fi) continue;
	KMLDonkey::App->donkey->setFilePriority(fi->fileNo(), pri);
    }
}

void DownloadPage::actionFilePriorityVeryLow() { setFilePriority(-20); }
void DownloadPage::actionFilePriorityLow() { setFilePriority(-10); }
void DownloadPage::actionFilePriorityNormal() { setFilePriority(0); }
void DownloadPage::actionFilePriorityHigh() { setFilePriority(10); }
void DownloadPage::actionFilePriorityVeryHigh() { setFilePriority(20); }

void DownloadPage::actionRetryConnect()
{
    QPtrList<QListViewItem> list = downloadView->selectedItems();
    DownloadFile* it;
    for (it = (DownloadFile*)list.first(); it; it = (DownloadFile*)list.next()) {
	FileInfo* fi = KMLDonkey::App->donkey->findDownloadFileNo(it->fileNo());
	if (!fi) continue;
	KMLDonkey::App->donkey->retryFile(fi->fileNo());
    }
}

void DownloadPage::actionPreviewFile()
{
    QPtrList<QListViewItem> list = downloadView->selectedItems();
    DownloadFile* it;
    for (it = (DownloadFile*)list.first(); it; it = (DownloadFile*)list.next()) {
	FileInfo* fi = KMLDonkey::App->donkey->findDownloadFileNo(it->fileNo());
	if (!fi) continue;
	KMLDonkey::App->donkey->previewFile(fi->fileNo());
    }
}

void DownloadPage::actionVerifyFile()
{
    QPtrList<QListViewItem> list = downloadView->selectedItems();
    DownloadFile* it;
    for (it = (DownloadFile*)list.first(); it; it = (DownloadFile*)list.next()) {
	FileInfo* fi = KMLDonkey::App->donkey->findDownloadFileNo(it->fileNo());
	if (!fi) continue;
	KMLDonkey::App->donkey->verifyFileChunks(fi->fileNo());
    }
}

void DownloadPage::actionGetFileFormat()
{
    QPtrList<QListViewItem> list = downloadView->selectedItems();
    DownloadFile* it;
    for (it = (DownloadFile*)list.first(); it; it = (DownloadFile*)list.next()) {
	FileInfo* fi = KMLDonkey::App->donkey->findDownloadFileNo(it->fileNo());
	if (!fi) continue;
	KMLDonkey::App->donkey->getFileFormat(fi->fileNo());
    }
}

void DownloadPage::actionFileInformation()
{
    QPtrList<QListViewItem> list = downloadView->selectedItems();
    DownloadFile* it;
    for (it = (DownloadFile*)list.first(); it; it = (DownloadFile*)list.next()) {
	FileInfo* fi = KMLDonkey::App->donkey->findDownloadFileNo(it->fileNo());
	if (!fi) continue;
        FileInfoDialog* d = new FileInfoDialog(fi->fileNo());
        d->show();
    }
}

void DownloadPage::actionCommitFile()
{
    QPtrList<QListViewItem> list = downloadedView->selectedItems();
    DownloadedFile* it;
    for (it = (DownloadedFile*)list.first(); it; it = (DownloadedFile*)list.next()) {
	FileInfo* fi = KMLDonkey::App->donkey->findDownloadFileNo(it->fileNo());
	if (!fi) continue;
        KMLDonkey::App->donkey->saveFile(fi->fileNo(), fi->fileName());
    }
}

void DownloadPage::actionCommitFileAs()
{
    QPtrList<QListViewItem> list = downloadedView->selectedItems();
    DownloadedFile* it;
    for (it = (DownloadedFile*)list.first(); it; it = (DownloadedFile*)list.next()) {
	FileInfo* fi = KMLDonkey::App->donkey->findDownloadFileNo(it->fileNo());
	if (!fi) continue;
        RenameDialog* d = new RenameDialog(KMLDonkey::App->donkey, fi->fileNo());
        connect(d, SIGNAL(renameFile(int, QString&)), this, SLOT(commitFileAs(int, QString&)));
        d->show();
    }
}

void DownloadPage::commitFileAs(int fileno, QString& name)
{
    KMLDonkey::App->donkey->saveFile(fileno, name);
}

void DownloadPage::actionRenameFile()
{
    QPtrList<QListViewItem> list = downloadView->selectedItems();
    DownloadFile* it;
    for (it = (DownloadFile*)list.first(); it; it = (DownloadFile*)list.next()) {
	FileInfo* fi = KMLDonkey::App->donkey->findDownloadFileNo(it->fileNo());
	if (!fi) continue;
        RenameDialog* d = new RenameDialog(KMLDonkey::App->donkey, fi->fileNo());
        connect(d, SIGNAL(renameFile(int, QString&)), this, SLOT(renameFile(int, QString&)));
        d->show();
    }
}

void DownloadPage::renameFile(int fileno, QString& name)
{
    QString foo = "rename " + QString::number(fileno) + " \"" + name + "\"";
    KMLDonkey::App->sendConsoleMessage(foo);
    KMLDonkey::App->donkey->refreshFileInfo(fileno);
}

void DownloadPage::copyFileToClipboard(ClipFormat format)
{
    QStringList l;
    QPtrList<QListViewItem> list = downloadView->selectedItems();
    DownloadFile* it;
    for (it = (DownloadFile*)list.first(); it; it = (DownloadFile*)list.next()) {
	FileInfo* fi = KMLDonkey::App->donkey->findDownloadFileNo(it->fileNo());
	if (!fi) continue;
        l.append(fi->fileName());
        l.append(FileInfo::md4ToString(fi->fileMD4()));
        l.append(QString::number((long int)fi->fileSize()));
    }
    copyToClipboard(l, format);
}

void DownloadPage::copyCompleteToClipboard(ClipFormat format)
{
    QStringList l;
    QPtrList<QListViewItem> list = downloadedView->selectedItems();
    DownloadedFile* it;
    for (it = (DownloadedFile*)list.first(); it; it = (DownloadedFile*)list.next()) {
	FileInfo* fi = KMLDonkey::App->donkey->findDownloadFileNo(it->fileNo());
	if (!fi) continue;
        l.append(fi->fileName());
        l.append(FileInfo::md4ToString(fi->fileMD4()));
        l.append(QString::number((long int)fi->fileSize()));
    }
    copyToClipboard(l, format);
}

void DownloadPage::actionFileCopyURL() { copyFileToClipboard(URL); }
void DownloadPage::actionFileCopyHash() { copyFileToClipboard(Hash); }
void DownloadPage::actionFileCopyHTML() { copyFileToClipboard(HTML); }
void DownloadPage::actionCompleteCopyURL() { copyCompleteToClipboard(URL); }
void DownloadPage::actionCompleteCopyHash() { copyCompleteToClipboard(Hash); }
void DownloadPage::actionCompleteCopyHTML() { copyCompleteToClipboard(HTML); }

void DownloadPage::fileUpdated(int fn)
{
    FileInfo* fi = KMLDonkey::App->donkey->findDownloadFileNo(fn);
    if (!fi) return;

    int del;
    switch (fi->fileState()) {
    case FileInfo::Complete:
    case FileInfo::Cancelled:
    case FileInfo::Aborted:
    case FileInfo::Shared:
        del = 1;
        break;
    default:
        del = 0;
        break;
    }

    DownloadFile* df = downloadFileList[fn];
    if (df) {
        if (del) downloadFileList.remove(fn);
        else {
            df->repaint();
            del = 1;
        }
    }
    if (!del) {
        df = new DownloadFile(downloadView, fn);
        downloadFileList.insert(fn, df);
        return;
    }

    switch (fi->fileState()) {
    case FileInfo::Complete:
        del = 0;
        break;
    default:
        del = 1;
        break;
    }

    DownloadedFile* foo = downloadedFileList[fn];
    if (foo) {
        if (del) downloadedFileList.remove(fn);
        else {
            df->repaint();
            return;
        }
    }
    if (!del) {
        foo = new DownloadedFile(downloadedView, fn);
        downloadedFileList.insert(fn, foo);
    }
}

void DownloadPage::fileRemoved(int fn)
{
    downloadFileList.remove(fn);
    downloadedFileList.remove(fn);
}
