/*
 * $Id: availability.cpp,v 1.6 2003/06/28 21:17:00 gibreel Exp $
 *
 * Copyright (C) 2003 Petter E. Stokke <gibreel@gibreel.net>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * $Log: availability.cpp,v $
 * Revision 1.6  2003/06/28 21:17:00  gibreel
 * The file info dialog now displays file sources. Libkmldonkey received a
 * number of updates and bugfixes to facilitate this.
 *
 * Revision 1.5  2003/06/19 21:50:45  gibreel
 * Split the various pages off into individual objects, and made a ton of code
 * cleanups, API changes, and not a few bugfixes in the process. The
 * disconnect/reconnect bug, especially, now seems to be gone.
 *
 * Revision 1.4  2003/06/09 18:10:23  gibreel
 * Configurable availability display colours. Extended the DCOP interface and
 * improved the interaction between the GUI and the applet. Added colour for
 * the download list's queued state. Cleanups, bugfixes all round.
 *
 * Revision 1.3  2003/04/21 18:17:06  gibreel
 * Tiny bugfix in availability renderer.
 *
 * Revision 1.2  2003/04/15 14:54:30  gibreel
 * Increased colour threshold a little.
 *
 * Revision 1.1  2003/04/13 17:59:46  gibreel
 * File info dialog and coloured availability bars.
 *
 */

#include "availability.h"
#include "kmldonkey.h"

#include <kglobal.h>
#include <kconfig.h>
#include <klocale.h>
#include <kdebug.h>
#include <qpainter.h>

AvailabilityRenderer::AvailabilityRenderer(int file)
{
    rfileno = file;
    updateFileInfo();
    isClientRenderer = false;
}

AvailabilityRenderer::AvailabilityRenderer(int file, int source)
{
    rfileno = file;
    clno = source;
    updateFileInfo();
    isClientRenderer = true;
}

void AvailabilityRenderer::updateFileInfo()
{
    FileInfo* fi = fileInfo();
    chunkNo = 0;
    if (fi) chunkNo = fi->fileChunks().length();
}

FileInfo* AvailabilityRenderer::fileInfo()
{
    return KMLDonkey::App->donkey->findDownloadFileNo(rfileno);
}

int AvailabilityRenderer::chunks() const
{
    return chunkNo;
}

void AvailabilityRenderer::paintChunk(QPainter& p, QRect& r, int chunk)
{
    FileInfo* fi = fileInfo();
    if (!fi) return;

    QColorGroup& cg = KMLDonkey::App->availabilityColours;
    int threshold = KMLDonkey::App->availabilityThreshold;

    if (KMLDonkey::App->donkey->findNetworkNo(fi->fileNetwork())->networkName() == "Donkey") {
	// Only the Donkey network currently supports chunk availability.

	int avail = (int)fi->fileAvailability()[chunk].cell();

	if (!avail)
	    p.setBrush(cg.background());
	else {
	    avail--;
	    if (avail > threshold) avail = threshold;
	    int rdiff = cg.light().red() - cg.dark().red(),
		gdiff = cg.light().green() - cg.dark().green(),
		bdiff = cg.light().blue() - cg.dark().blue();
	    p.setBrush(QColor(cg.dark().red() + ((rdiff * avail) / threshold),
			      cg.dark().green() + ((gdiff * avail) / threshold),
			      cg.dark().blue() + ((bdiff * avail) / threshold)));
	}
    } else {
	// Other networks seem to report ascii '0' as availability,
	// we just render it as a single source.
	p.setBrush(cg.dark());
    }
    p.setPen(Qt::NoPen);
    p.drawRect(r);

    QRect d = r;
    QChar dv;
    if (isClientRenderer) {
	const QMap<int,QString>& cc = fi->fileSources();
	if (cc.contains(clno))
	    dv = cc[clno][chunk];
	else
	    dv = '0';
    } else {
	dv = fi->fileChunks()[chunk];
    }
    if (dv.digitValue() <= 0 || dv == '0')
	return;
    if (dv == '1' && !isClientRenderer) d.setTop((d.top() + d.bottom()) / 2);
    p.setBrush(cg.foreground());
    p.drawRect(d);
}

void AvailabilityRenderer::paintAvailability(QPainter& p, QRect& g)
{
    updateFileInfo();
    int c = chunkNo, s = -1, l, r, i;
    for (i = 0; i < c; i++) {
        l = (i * g.width()) / c;
        if (l > s) {
            r = ((i+1) * g.width()) / c;
            QRect d = g;
            d.setLeft(l);
            d.setRight(r);
            paintChunk(p, d, i);
            s = l;
        }
    }
}



AvailabilityWidget::AvailabilityWidget(int file, QWidget* parent, const char* name, bool frame)
    : QFrame(parent, name)
    , AvailabilityRenderer(file)
{
    if (frame)
        setFrameStyle(QFrame::StyledPanel | QFrame::Sunken);
    setMinimumSize(minimumSizeHint());
}

void AvailabilityWidget::paintEvent(QPaintEvent* e)
{
    QFrame::paintEvent(e);
    QPainter p(this);
    QRect g = contentsRect();
    p.setClipRect(g);
    paintAvailability(p, g);
    FileInfo* fi = fileInfo();
    QString t;
    if (fi)
        t.sprintf(i18n("%.02f%%"), ((float)fi->fileDownloaded() * 100.0) / (float)fi->fileSize());
    else
        t = i18n("Unknown file!");
    QFont f = font();
    f.setBold(true);
    p.setFont(f);
    p.setPen(Qt::black);
    for (int x = -1; x <= 1; x++) {
        for (int y = -1; y <= 1; y++) {
            QRect foo = rect();
            foo.moveBy(x, y);
            p.drawText(foo, AlignCenter, t);
        }
    }
    p.setPen(Qt::white);
    p.drawText(rect(), AlignCenter, t);
}

QSizePolicy AvailabilityWidget::sizePolicy() const
{
    return QSizePolicy(QSizePolicy::Expanding, QSizePolicy::Minimum);
}

QSize AvailabilityWidget::minimumSizeHint() const
{
    return QSize((chunkNo > 300) ? 300 : chunkNo, QFontMetrics(font()).height() + 4 + (frameWidth() * 2));
}

#include "availability.moc"
