/*
 * $Id: mldonkeyapplet.cpp,v 1.16 2003/07/23 18:03:47 gibreel Exp $
 *
 * Copyright (C) 2002, 2003 Petter E. Stokke <gibreel@gibreel.net>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * $Log: mldonkeyapplet.cpp,v $
 * Revision 1.16  2003/07/23 18:03:47  gibreel
 * Support for multiple host definitions using HostManager.
 *
 * Revision 1.15  2003/07/01 23:08:23  gibreel
 * The applet and GUI now accept drag and drop events containing URLs, which
 * are passed on to the core.
 *
 * Revision 1.14  2003/06/09 23:06:01  gibreel
 * Improved applet configurability, fixed a few layout issues, and finally
 * implemented the mute button.
 *
 * Revision 1.13  2003/06/09 18:10:23  gibreel
 * Configurable availability display colours. Extended the DCOP interface and
 * improved the interaction between the GUI and the applet. Added colour for
 * the download list's queued state. Cleanups, bugfixes all round.
 *
 * Revision 1.12  2003/05/01 16:59:08  gibreel
 * Removed the GUI functionality from the applet, and reimplemented the GUI
 * button as a launcher and hide button (over DCOP) for the standalone GUI.
 *
 * Revision 1.11  2003/03/17 11:44:14  gibreel
 * Discarded the .ui files for mldonkeyappletgui in favour of own code. Now
 * uses KIconLoader to load icons rather than embedding them. Also layouts
 * itself properly when resized, and includes a different layout for small
 * panel sizes.
 *
 * Revision 1.10  2003/03/10 14:52:02  gibreel
 * Support for GUI protocol 14. Specifically: Authentication with username, new
 * download file state "Queued", and support for the new message types in the
 * DonkeyProtocol class.
 *
 * Revision 1.9  2003/03/10 00:23:27  gibreel
 * Added copy to clipboard features to file context menus.
 *
 * Revision 1.8  2003/03/08 20:13:18  gibreel
 * Switched readString/writeString in DonkeyMessage from using pointers to
 * QStrings to using references, which leads to far less hassle, shorter code,
 * and less chance of memory leaks.
 *
 * Revision 1.7  2003/03/08 17:08:11  gibreel
 * Lots of minor API changes.
 *
 * Revision 1.6  2003/03/08 13:55:14  gibreel
 * Static version string replaced by the VERSION macro provided by autoconf.
 *
 * Revision 1.5  2003/03/07 22:02:48  gibreel
 * Updated home page URL.
 *
 * Revision 1.4  2003/03/07 21:51:18  gibreel
 * Changed the MD4 storage in the FileInfo object from an array of int to a
 * QByteArray, added conversion functions between that and QString, and methods
 * to search downloaded files by hash. The applet now uses the file hash
 * instead of the fileno when storing and restoring priorities, as the fileno
 * changes between core restarts.
 *
 * Revision 1.3  2003/03/07 15:09:09  gibreel
 * Repaired about dialog and added some more information to it.
 *
 * Revision 1.2  2003/03/07 14:28:15  gibreel
 * Finished implementing priority save/restore.
 *
 * Revision 1.1.1.1  2003/03/07 11:50:11  gibreel
 * Initial import.
 *
 */

#include <klocale.h>
#include <kglobal.h>
#include <kaboutdata.h>
#include <kaboutapplication.h>
#include <kconfig.h>
#include <kapplication.h>
#include <qmessagebox.h>
#include <qtimer.h>
#include <qlayout.h>
#include <qcheckbox.h>
#include <knuminput.h>

#include "config.h"
#include "appletconfig.h"
#include "mldonkeyapplet.h"
#include "mldonkeyapplet.moc"


extern "C"
{
    KPanelApplet* init( QWidget *parent, const QString& configFile )
    {
        KGlobal::locale()->insertCatalogue( "mldonkeyapplet" );
        return new MLDonkeyApplet( configFile, KPanelApplet::Normal,
				   KPanelApplet::Preferences |
				   KPanelApplet::About,
				   parent, "mldonkeyapplet" );
    }
}

MLDonkeyApplet::MLDonkeyApplet( const QString& configFile,
				Type type, int actions,
				QWidget *parent, const char *name )
    : DCOPObject("MLDonkeyAppletIface")
    , KPanelApplet( configFile, type, actions, parent, name )
{
    aboutData = NULL;

    setSizePolicy( QSizePolicy( QSizePolicy::Fixed, QSizePolicy::Fixed, 0, 0, sizePolicy().hasHeightForWidth() ) );

    client = new DCOPClient();
    client->registerAs("mldonkeyapplet");
    client->setNotifications(true);
    connect(client, SIGNAL(applicationRemoved(const QCString&)), this, SLOT(applicationRemoved(const QCString&)));
    connectDCOPSignal("kmldonkey", "KMLDonkeyIface", "kmldonkeyAppeared(bool)", "kmldonkeyAppeared(bool)", false);

    restoreConfiguration();

    configDialog = new AppletConfig(this);
    connect(configDialog, SIGNAL(applyClicked()), this, SLOT(applyConfig()));
    connect(configDialog, SIGNAL(okClicked()), this, SLOT(applyConfig()));

    setFrameStyle( StyledPanel | Sunken );
    gui = new MLDonkeyAppletGUI(this);
    QHBoxLayout *layout = new QHBoxLayout(this);
    layout->setAlignment(Qt::AlignRight);
    layout->add(gui);
    gui->setLaunchState(isGUIVisible());
    updateLabels();

    connect(gui, SIGNAL(toggledLaunch(bool)), this, SLOT(showGUI(bool)));
    connect(gui, SIGNAL(toggledMute(bool)), this, SLOT(muteDonkey(bool)));

    timer = new QTimer(this);
    connect( timer, SIGNAL( timeout() ), SLOT( refreshDisplay() ));
    timer->start(5000);

    hostManager = new HostManager(this);
    donkey = new DonkeyProtocol(true, this);
    connect(donkey, SIGNAL(clientStats(int64, int64, int64, int, int, int, int, int, int, int)),
	    this, SLOT(updateStatus(int64, int64, int64, int, int, int, int, int, int, int)));
    connect(donkey, SIGNAL(donkeyDisconnected(int)), this, SLOT(donkeyDisconnected(int)));
    connect(donkey, SIGNAL(donkeyConnected()), this, SLOT(donkeyConnected()));
    connect(donkey, SIGNAL(updatedDownloadFiles()), this, SLOT(updateDownloadFiles()));
    connect(donkey, SIGNAL(updatedDownloadedFiles()), this, SLOT(updateDownloadedFiles()));
    connect(donkey, SIGNAL(updatedConnectedServers()), this, SLOT(updateServers()));
    connect(donkey, SIGNAL(consoleMessage(QString&)), this, SLOT(consoleMessage(QString&)));
    connect(hostManager, SIGNAL(hostListUpdated()), this, SLOT(connectDonkey()));

    setAcceptDrops(true);

    connectDonkey();
    reconnect = 0;
}

MLDonkeyApplet::~MLDonkeyApplet()
{
}

void MLDonkeyApplet::updateLabels()
{
    QString first, second;
    if (activeDisplays.count() >= 1)
	first = configDialog->itemTitles[activeDisplays[0]];
    if (activeDisplays.count() >= 2)
	second = configDialog->itemTitles[activeDisplays[1]];
    gui->updateLabels(first, second);
    emit updateLayout();
}

const QString humanReadableSize(int64 rsz)
{
    QString foo;
    double sz = (double)rsz;
    if (sz >= (double)(1024 * 1024 * 1024)) {
	sz = sz / (1024 * 1024 * 1024);
	foo.sprintf("%.1fG", sz);
    } else if (sz >= (double)(1024 * 1024)) {
	sz = sz / (1024 * 1024);
	foo.sprintf("%.1fM", sz);
    } else if (sz >= (double)1024) {
	sz = sz / 1024;
	foo.sprintf("%.1fK", sz);
    } else foo.sprintf("%.0f", sz);
    return foo;
}

QString MLDonkeyApplet::produceStatus(const QString& key, int64 ul, int64 dl, int64 sh, int nsh,
				   int tul, int tdl, int uul, int udl, int ndl, int ncp)
{
    QString txt;
    QTextOStream s(&txt);
    if (key == "speed")
	s << QString::number((double)(tul+uul) / 1024.0, 'f', 1) << "/" << QString::number((double)(tdl+udl) / 1024.0, 'f', 1);
    else if (key == "files")
	s << QString::number(ncp) << "/" << QString::number(ndl);
    else if (key == "transfer")
	s << humanReadableSize(ul) << "/" << humanReadableSize(dl);
    else if (key == "shared")
	s << QString::number(nsh) << "/" << humanReadableSize(sh);
    return txt;
}

void MLDonkeyApplet::updateStatus(int64 ul, int64 dl, int64 sh, int nsh, int tul, int tdl, int uul, int udl, int ndl, int ncp)
{
    QString first, second;

    if (activeDisplays.count() >= 1)
	first = produceStatus(activeDisplays[0], ul, dl, sh, nsh, tul, tdl, uul, udl, ndl, ncp);
    if (activeDisplays.count() >= 2)
	second = produceStatus(activeDisplays[1], ul, dl, sh, nsh, tul, tdl, uul, udl, ndl, ncp);

    gui->updateStatus(first, second);
    emit updateLayout();
}

void MLDonkeyApplet::notifyUnhandled(int op)
{
    QString foo;
    foo.sprintf(i18n("Unhandled message from core: %d"), op);
    QMessageBox::information(this, "KMLDonkey", foo);
}

void MLDonkeyApplet::donkeyDisconnected(int err)
{
    switch (err) {
    case DonkeyProtocol::AuthenticationError:
	QMessageBox::critical(this, i18n("KMLDonkey Applet"),
			      i18n("Incorrect password."));
	break;
    case DonkeyProtocol::ConnectionRefusedError:
	if (!reconnect) QMessageBox::critical(this, i18n("KMLDonkey Applet"),
					      i18n("Connection refused."));
	reconnect = 1;
	break;
    case DonkeyProtocol::HostNotFoundError:
	QMessageBox::critical(this, i18n("KMLDonkey Applet"),
			      i18n("Host not found."));
	break;
    case DonkeyProtocol::SocketReadError:
	if (!reconnect) QMessageBox::critical(this, i18n("KMLDonkey Applet"),
					      i18n("Read error. Applet disconnected."));
	reconnect = 1;
	break;
    case DonkeyProtocol::ObsoleteProtocolError:
	QMessageBox::critical(this, i18n("KMLDonkey Applet"),
			      i18n("Your mldonkey core uses an obsolete communications protocol. "
				   "Please upgrade it to a more recent version!"));
	break;
    default:
	reconnect = 1;
	break;
    }
    gui->donkeyDisconnected();
    emit updateLayout();
}

void MLDonkeyApplet::donkeyConnected()
{
    reconnect = 0;
}

void MLDonkeyApplet::refreshDisplay()
{
    if (reconnect && (!donkey || donkey->donkeyStatus() == QSocket::Idle)) {
	if (!donkey) donkey = new DonkeyProtocol(this);
	connectDonkey();
    }
}

void MLDonkeyApplet::connectDonkey()
{
    donkey->connectDonkey(hostManager->defaultHost());
}

void MLDonkeyApplet::restoreConfiguration()
{
    KConfig *conf = config();

    conf->setGroup("Layout");
    showLabels = conf->readBoolEntry("ShowLabels", false);
    showDouble = conf->readBoolEntry("ShowLabelsOnlyInDouble", false);
    showMute = conf->readBoolEntry("ShowMuteInSingle", true);
    activeDisplays = conf->readListEntry("ActiveDisplays");
    if (activeDisplays.empty()) {
	activeDisplays.push_back("files");
	activeDisplays.push_back("speed");
    }

    conf->setGroup("Mute");
    muteUploadRate = conf->readUnsignedNumEntry("MuteUploadRate",4);
    muteDownloadRate = conf->readUnsignedNumEntry("MuteDownloadRate",4);
    normalUploadRate = conf->readUnsignedNumEntry("NormalUploadRate",0);
    normalDownloadRate = conf->readUnsignedNumEntry("NormalDownloadRate",0);

    KConfig *global = new KConfig("mldonkeyrc", false, false);
    global->setGroup("MLDonkey");
    donkeyHost = global->readEntry("DonkeyHost", "localhost");
    donkeyPort = global->readNumEntry("DonkeyGuiPort", 4001);
    donkeyUname = global->readEntry("DonkeyUsername","admin");
    donkeyPasswd = global->readEntry("DonkeyPassword","");
    delete global;

}

void MLDonkeyApplet::writeConfiguration()
{
    KConfig *conf = config();

    conf->setGroup("Layout");
    conf->writeEntry("ShowLabels", showLabels);
    conf->writeEntry("ShowLabelsOnlyInDouble", showDouble);
    conf->writeEntry("ShowMuteInSingle", showMute);
    conf->writeEntry("ActiveDisplays", activeDisplays);

    conf->setGroup("Mute");
    conf->writeEntry("MuteUploadRate", muteUploadRate);
    conf->writeEntry("MuteDownloadRate", muteDownloadRate);
    conf->writeEntry("NormalUploadRate", normalUploadRate);
    conf->writeEntry("NormalDownloadRate", normalDownloadRate);

    conf->sync();
}

void MLDonkeyApplet::applyConfig()
{
    showLabels = configDialog->showLabelsSelect->isChecked();
    showDouble = configDialog->showDoubleSelect->isChecked();
    showMute = configDialog->showMuteSelect->isChecked();
    activeDisplays = configDialog->active();

    normalDownloadRate = configDialog->normalDownloadEntry->value();
    normalUploadRate = configDialog->normalUploadEntry->value();
    muteDownloadRate = configDialog->muteDownloadEntry->value();
    muteUploadRate = configDialog->muteUploadEntry->value();

    writeConfiguration();
    updateLabels();
    gui->updateLayout();
    emit updateLayout();
}

void MLDonkeyApplet::preferences()
{
    configDialog->showLabelsSelect->setChecked(showLabels);
    configDialog->showDoubleSelect->setChecked(showDouble);
    configDialog->showMuteSelect->setChecked(showMute);
    configDialog->setActive(activeDisplays);

    configDialog->normalDownloadEntry->setValue(normalDownloadRate);
    configDialog->normalUploadEntry->setValue(normalUploadRate);
    configDialog->muteDownloadEntry->setValue(muteDownloadRate);
    configDialog->muteUploadEntry->setValue(muteUploadRate);

    configDialog->show();
}

int MLDonkeyApplet::widthForHeight(int height) const
{
    if (height < 32) gui->relayoutWidgets(1);
    else gui->relayoutWidgets(0);
    return gui->width();
}

int MLDonkeyApplet::heightForWidth(int) const
{
    return gui->height();
}

void MLDonkeyApplet::resizeEvent(QResizeEvent* e)
{
    KPanelApplet::resizeEvent(e);
    emit updateLayout();
}

void MLDonkeyApplet::about()
{
    if(!aboutData) {

	aboutData = new KAboutData("mldonkeyapplet", I18N_NOOP("MLDonkey Applet"),
				   VERSION, I18N_NOOP("<h2>MLDonkey Applet</h2><p>An applet for interacting with MLDonkey.</p>"),
				   KAboutData::License_GPL_V2, I18N_NOOP("Copyright &copy; 2002, 2003 Petter E. Stokke"),
				   I18N_NOOP("<p>Part of the KMLDonkey package.</p>"),
				   "http://www.gibreel.net/projects/kmldonkey/", "gibreel@gibreel.net");

	aboutData->addAuthor("Petter E. Stokke", 0, "gibreel@gibreel.net");
    }

    KAboutApplication dialog(aboutData);
    dialog.exec();
}

void MLDonkeyApplet::applicationRemoved(const QCString& appId)
{
    if (appId == QCString("kmldonkey"))
        gui->setLaunchState(false);
}

void MLDonkeyApplet::showGUI(bool state)
{
    if (state && !isGUIRunning()) {
        // launch gui
        if (KApplication::startServiceByDesktopPath(QString("Internet/kmldonkey.desktop"), QString()) > 0)
            gui->setLaunchState(false);
        return;
    }
    QByteArray data;
    QDataStream arg(data, IO_WriteOnly);
    if (state)
        arg << true;
    else
        arg << false;
    client->send("kmldonkey", "KMLDonkey", "setShown(bool)", data);
}

bool MLDonkeyApplet::isGUIRunning()
{
    QCString name = "kmldonkey";
    QCStringList apps = client->registeredApplications();
    QCStringList::iterator it;
    for (it = apps.begin(); it != apps.end(); ++it)
        if (*it == name) return true;
    return false;
}

bool MLDonkeyApplet::isGUIVisible()
{
    QCString replyType;
    QByteArray data, replyData;
    QDataStream reply(replyData, IO_ReadOnly);
    if (!client->call("kmldonkey", "KMLDonkey", "isVisible()", data, replyType, replyData, true))
        return false;
    if (replyType != "bool") return false;
    bool state;
    reply >> state;
    return state;
}

void MLDonkeyApplet::kmldonkeyAppeared(bool a)
{
    gui->setLaunchState(a);
}

void MLDonkeyApplet::muteDonkey(bool mute)
{
    donkey->setOption("max_hard_upload_rate", QString::number(mute ? muteUploadRate : normalUploadRate));
    donkey->setOption("max_hard_download_rate", QString::number(mute ? muteDownloadRate : normalDownloadRate));
}

void MLDonkeyApplet::dragEnterEvent(QDragEnterEvent* event)
{
    event->accept(QUriDrag::canDecode(event));
}

void MLDonkeyApplet::dropEvent(QDropEvent* event)
{
    QStringList uri;

    if (QUriDrag::decodeToUnicodeUris(event, uri))
    {
	QStringList::Iterator it;
	for (it = uri.begin(); it != uri.end(); ++it)
	    donkey->submitURL(*it);
    }
}
