/* -*- c++ -*-
 *
 * fileinfo.cpp
 *
 * Copyright (C) 2003 Petter E. Stokke <gibreel@gibreel.net>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 */


#include <kdebug.h>
#include <klocale.h>

#include "fileinfo.h"
#include "donkeymessage.h"



FileInfo::FileInfo(DonkeyMessage* msg, int proto)
{
    updateFileInfo(msg, proto);
}

FileInfo::FileInfo(const FileInfo& fi)
{
    num = fi.fileNo();
    updateFileInfo(&fi);
}

FileInfo::~FileInfo()
{
}

void FileInfo::updateFileInfo(const FileInfo* fi)
{
    network = fi->fileNetwork();
    names = fi->fileNames();
    md4 = fi->fileMD4();
    size = fi->fileSize();
    downloaded = fi->fileDownloaded();
    nlocations = fi->fileNLocations();
    nclients = fi->fileNClients();
    state = fi->fileState();
    abortedmsg = fi->fileAbortedMsg();
    chunks = fi->fileChunks();
    availability = fi->fileAvailability();
    speed = fi->fileSpeed();
    chunks_age = fi->fileChunksAge();
    age = fi->fileAge();
    format = fi->fileFormat();
    formatinfo = fi->fileFormatInfo();
    name = fi->fileName();
    lastseen = fi->fileLastSeen();
    priority = fi->filePriority();
    comment = fi->fileComment();
}

void FileInfo::updateFileInfo(DonkeyMessage* msg, int proto)
{
    int i,j;
    num = msg->readInt32();
    network = msg->readInt32();
    j = msg->readInt16();
    names.clear();
    for (i=0; i<j; i++)
	names.append(msg->readString());
    md4 = QByteArray(16);
    for (i=0; i<16; i++)
	md4[i] = msg->readInt8();
    size = msg->readInt32();
    downloaded = msg->readInt32();
    nlocations = msg->readInt32();
    nclients = msg->readInt32();
    state = (State)msg->readInt8();
    if (state == 6)
	abortedmsg = msg->readString();
    chunks = msg->readByteArray();
    availability.clear();
    if (proto > 17) {
	j = msg->readInt16();
	for (i=0; i<j; i++) {
	    int av_net = msg->readInt32();
	    availability.replace(av_net, msg->readByteArray());
	}
    } else {
	availability.replace(network, msg->readByteArray());
    }
    speed = msg->readFloat();
    j = msg->readInt16();
    chunks_age.clear();
    for (i=0; i<j; i++) {
	time_t a = (int)msg->readFloat();
	if (a < 0) a = 0x7fffffff + a;
	chunks_age.append(a);
    }
    age = (time_t)msg->readFloat();
	if (age < 0) age = 0x7fffffff + age;
    format = msg->readInt8();
    switch (format) {
    case 1:
    {
	QString foo = msg->readString();
	QString bar = msg->readString();
	formatinfo = foo + " " + bar;
    } break;
    case 2:
    {
	QString foo = msg->readString();
	int i = msg->readInt32();
	int j = msg->readInt32();
	int k = msg->readInt32();
	int l = msg->readInt32();
	formatinfo = i18n("AVI codec resolution framerate bitrate", "AVI %1 %2x%3 %4fps rate %5")
	    .arg(foo).arg(i).arg(j).arg((double)k / 1000.0, 0, 'f', 2).arg(l);
    } break;
    case 3:
    {
	QString foo = msg->readString();
	QString bar = msg->readString();
	QString baz = msg->readString();
	QString quux = msg->readString();
	msg->readString();
	int i = msg->readInt32();
	msg->readInt32();
	formatinfo = i18n("MP3 title artist album year track", "MP3 Ti:%1 Ar:%2 Al:%3 Yr:%4 Tr:%5")
	    .arg(foo).arg(bar).arg(baz).arg(quux).arg(i);
    } break;
    default:
	formatinfo = i18n("Unknown format");
	break;
    }

    if (proto >= 8)
	name = msg->readString();
    else if(names.count() > 0)
	name = names[0];
    else // should never be the case, but just to be sure...
        name = QString::null;

    if (proto >= 9)
	lastseen = msg->readInt32();
    else
	lastseen = -1;

    if (proto >= 12)
	priority = msg->readInt32();
    else
	priority = 0;
    if (priority > 0x40000000) priority -= 0x80000000;
    
    if (proto >= 22)
	comment = msg->readString();
}

const int FileInfo::fileNo() const
{
    return num;
}

const int FileInfo::fileNetwork() const
{
    return network;
}

const QString& FileInfo::fileName() const
{
    return name;
}

const QStringList& FileInfo::fileNames() const
{
    return names;
}

const QByteArray& FileInfo::fileMD4() const
{
    return md4;
}

const int64 FileInfo::fileSize() const
{
    return size;
}

const int64 FileInfo::fileDownloaded() const
{
    return downloaded;
}

const int FileInfo::fileNLocations() const
{
    return nlocations;
}

const int FileInfo::fileNClients() const
{
    return nclients;
}

const FileInfo::State FileInfo::fileState() const
{
    return state;
}

const QString& FileInfo::fileAbortedMsg() const
{
    return abortedmsg;
}

const QByteArray& FileInfo::fileChunks() const
{
    return chunks;
}

const QMap<int,QByteArray>& FileInfo::fileAvailability() const
{
    return availability;
}

const double FileInfo::fileSpeed() const
{
    return speed;
}

const QValueList<time_t>& FileInfo::fileChunksAge() const
{
    return chunks_age;
}

const time_t FileInfo::fileAge() const
{
    return age;
}

const int FileInfo::fileFormat() const
{
    return format;
}

const QString& FileInfo::fileFormatInfo() const
{
    return formatinfo;
}

const int FileInfo::fileLastSeen() const
{
    return lastseen;
}

const int FileInfo::filePriority() const
{
    return priority;
}

const QMap<int,QString>& FileInfo::fileSources() const
{
    return sources;
}

const QString& FileInfo::fileComment() const
{
    return comment;
}


void FileInfo::setFileName(const QString& newname)
{
    name = newname;
}

void FileInfo::addSource(int source)
{
    if (!sources.contains(source))
	sources.insert(source, QString::null);
}

void FileInfo::removeSource(int source)
{
    sources.remove(source);
}

void FileInfo::updateAvailability(int source, const QString& avail)
{
    sources.insert(source, avail);
}

void FileInfo::updateDownloadStatus(DonkeyMessage* msg, int)
{
    downloaded = msg->readInt32();
    speed = msg->readFloat();
    if (msg->opcode() >= 46)
        lastseen = msg->readInt32();
}


QString FileInfo::md4ToString(const QByteArray& hash)
{
    char foo[33], bar[16];
    int i;
    foo[0] = 0;
    for (i=0; i<16; i++) {
	sprintf(bar, "%02x", (unsigned char)hash[i]);
	strcat(foo, bar);
    }
    return QString(foo).upper();
}

static const QString hexmap = "0123456789ABCDEF";

QByteArray FileInfo::stringToMd4(const QString& hash)
{
    QByteArray out(16);
    if (hash.length() != 32) return out;
    QString in = hash.upper();
    int b;
    for (int i=0; i<(int)in.length(); i+=2) {
	b = hexmap.find(in[i]) << 4;
	b |= hexmap.find(in[i+1]);
	out[i>>1] = b;
    }
    return out;
}

