/* -*- c++ -*-
 *
 * hostmanager.cpp
 *
 * Copyright (C) 2003 Petter E. Stokke <gibreel@gibreel.net>
 * Copyright (C) 2003 Sebastian Sauer <mail@dipe.org> 
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 */

#include <kconfig.h>
#include <kstandarddirs.h>
#include <kdirwatch.h>
#include <qsignalmapper.h>

#include "donkeyhost.h"
#include "gifthost.h"

#include "hostmanager.h"
#include "hostmanager.moc"

HostManager::HostManager(QObject* parent, const char* name)
    : QObject(parent, name)
{
    refreshHostList();

    configWatcher = new KDirWatch(this);
    configWatcher->addFile(locateLocal("config", "mldonkeyrc"));

    connect(configWatcher, SIGNAL(dirty(const QString&)), SLOT(fileChanged(const QString&)));
    connect(configWatcher, SIGNAL(created(const QString&)), SLOT(fileChanged(const QString&)));
    connect(configWatcher, SIGNAL(deleted(const QString&)), SLOT(fileChanged(const QString&)));
}

void HostManager::refreshHostList()
{
    m_hosts.clear();
    m_default = QString::null;

    KConfig *config = new KConfig("mldonkeyrc", false, false);

    QStringList list = config->groupList();
    QStringList::iterator it;
    for (it = list.begin(); it != list.end(); ++it) {
        config->setGroup(*it);
        if(config->hasKey("DonkeyHost")) {
            DonkeyHost* entry = new DonkeyHost(*it,
                config->readEntry("DonkeyHost", "localhost"),
                config->readNumEntry("DonkeyPort", 4000),
                config->readNumEntry("DonkeyGuiPort", 4001),
                config->readEntry("DonkeyUsername", "admin"),
                config->readEntry("DonkeyPassword"),
                config->readBoolEntry("LocalHost", false),
                config->readPathEntry("TempPath"),
                config->readPathEntry("IncomingPath") );
            if (config->readBoolEntry("Default") && m_default.isNull())
                m_default = entry->name();
            m_hosts.replace(entry->name(), entry);
        }
        else if (config->hasKey("GiftHost")) {
            GiftHost* entry = new GiftHost(*it,
                config->readEntry("GiftHost", "localhost"),
                config->readNumEntry("GiftPort", 4000),
                config->readEntry("GiftUsername", "giFTuser") );
            if (config->readBoolEntry("Default") && m_default.isNull())
                m_default = entry->name();
            m_hosts.replace(entry->name(), entry);
        }
    }
    if (m_hosts.empty()) { // create default
	DonkeyHost* entry = new DonkeyHost("MLDonkey", "localhost", 4000, 4001, "admin", "");
	m_default = entry->name();
	m_hosts.replace(entry->name(), entry);
    }
    if (m_default.isNull()) {
	QMap<QString,HostInterface*>::Iterator it = m_hosts.begin();
        m_default = it.key();
    }
}

void HostManager::fileChanged(const QString&)
{
    refreshHostList();
    emit hostListUpdated();
}

QStringList HostManager::hostList()
{
    return m_hosts.keys();
}

const QString& HostManager::defaultHostName()
{
    return m_default;
}

HostInterface* HostManager::defaultHost()
{
    return m_hosts[m_default];
}

HostInterface* HostManager::hostProperties(const QString& hostName)
{
    return m_hosts[hostName];
}

bool HostManager::validHostName(const QString& hostName)
{
    return m_hosts.contains(hostName);
}

bool HostManager::isLocalHost(const QString& hostName)
{
    if (!validHostName(hostName)) return false;
    return m_hosts[hostName]->localHost();
}



HostSelectAction::HostSelectAction(const QString& text, const QString& icon,
				   HostManager* hostManager, QObject* parent, const char* name)
    : KActionMenu(text, icon, parent, name)
{
    m_connectActions.setAutoDelete(true);
    if (hostManager)
	m_hostManager = hostManager;
    else
	m_hostManager = new HostManager(this);
    setDelayed(true);
    m_connectMapper = new QSignalMapper(this);
    connect(m_connectMapper, SIGNAL(mapped(const QString&)), SLOT(slotItemSelected(const QString&)));
    populateMenu();
    connect(m_hostManager, SIGNAL(hostListUpdated()), SLOT(populateMenu()));
}

void HostSelectAction::populateMenu()
{
    QPtrListIterator<KAction> ait(m_connectActions);
    for (; ait.current(); ++ait)
	remove(ait.current());
    m_connectActions.clear();

    QStringList list(m_hostManager->hostList());
    QStringList::iterator it;
    for (it = list.begin(); it != list.end(); ++it) {
	KAction* action = new KAction(*it, 0, this);
	connect(action, SIGNAL(activated()), m_connectMapper, SLOT(map()));
	m_connectMapper->setMapping(action, *it);
	insert(action);
	m_connectActions.append(action);
    }
}

void HostSelectAction::slotItemSelected(const QString& item)
{
    if (m_hostManager->validHostName(item)) {
	emit hostSelected(item);
	emit hostSelected(m_hostManager->hostProperties(item));
    }
}
