/* -*- c++ -*-
 *
 * infolist.h
 *
 * Copyright (C) 2003 Petter E. Stokke <gibreel@gibreel.net>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 */

#ifndef __infolist_h__
#define __infolist_h__

#include <qptrlist.h>
#include <klistview.h>
#include "availability.h"


class InfoList : public KListView
{
    Q_OBJECT
public:
    InfoList(QWidget* parent = 0, const char* name = 0);
    ~InfoList();

    void initialise(KConfig* config, const QString& group);
    void finalise(KConfig* config, const QString& group);

    int addColumn(const QString& label, int width = -1, QListView::WidthMode widthMode = QListView::Manual);
    int realColumnCount() const;
    QValueList<int> visibleColumns() const;
    int visibleToReal(int c) const;
    int realToVisible(int c) const;

signals:
    void gotFocus();
    void lostFocus();
    void focusChanged();

protected:
    virtual void focusInEvent(QFocusEvent*);
    virtual void focusOutEvent(QFocusEvent*);

private:
    int buildColumn(int c);

    QMap<int,int> m_visibleReal;
    QMap<int,int> m_realVisible;
    QMap<int,QString> m_columns;
    QMap<int,int> m_columnWidth;
    QMap<int,QListView::WidthMode> m_columnWidthMode;
    int m_columnCount;
};


class InfoItemCacheEntry
{
public:
    InfoItemCacheEntry() {}

    void update(const QString& str);
    void update(const QString& str, double num);
    int compare(const InfoItemCacheEntry*) const;

    const QString& text() const { return m_str; }
    double numeric() const { return m_isNum ? m_num : 0.0; }
    bool isNumeric() const { return m_isNum; }

private:
    QString m_str;
    double m_num;
    bool m_isNum;
};


class InfoItem : public KListViewItem
{
public:
    InfoItem( KListView *parent, int file );
    ~InfoItem();

    QString text( int column ) const;
    virtual QString xtext( int column ) const = 0;
    void setFileNo( int file );
    int fileNo();
    int compare(QListViewItem*, int, bool) const;
    virtual double numeric(int) const;
    virtual bool isNumeric(int) const;

    void refresh(bool alsoRepaint = true);
    const InfoItemCacheEntry* cacheEntry(int column) const;

protected:
    int fileno;
    QPtrList<InfoItemCacheEntry> cache;
};



class DownloadFile : public InfoItem, public AvailabilityRenderer
{
public:
    DownloadFile( KListView *parent, int file );
    ~DownloadFile() {}

    void setText(int, const QString&);
    QString xtext( int column ) const;
    double numeric( int col ) const;
    bool isNumeric(int col) const;

    virtual void paintCell(QPainter* p, const QColorGroup& cg, int col, int w, int align);
    virtual int width(const QFontMetrics& fm, const QListView* lv, int c) const;
};

class DownloadedFile : public InfoItem
{
public:
    DownloadedFile( KListView *parent, int file ) : InfoItem(parent,file) { refresh(); }
    ~DownloadedFile() {}

    QString xtext( int column ) const;
    double numeric( int col ) const;
    bool isNumeric(int col) const;

};


class ServerInfoItem : public InfoItem
{
public:
    ServerInfoItem( KListView *parent, int server ) : InfoItem(parent,server) { refresh(); }
    ~ServerInfoItem() {}

    QString xtext( int column ) const;
    double numeric( int col ) const;
    bool isNumeric(int col) const;

    virtual void paintCell(QPainter* p, const QColorGroup& cg, int col, int w, int align);
};

class SharedFile : public InfoItem
{
public:
    SharedFile( KListView *parent, int share ) : InfoItem(parent,share) { refresh(); }
    ~SharedFile() {}

    QString xtext( int column ) const;
    double numeric( int col ) const;
    bool isNumeric(int col) const;

};

class UploadersFile : public KListViewItem
{
public:
    UploadersFile(KListView *parent) : KListViewItem(parent) {}
    ~UploadersFile() {}

    int compare(QListViewItem*, int, bool) const;
};

class ClientItem : public InfoItem
{
public:
    ClientItem( KListView *parent, int client ) : InfoItem(parent,client) { filesListed = false; refresh(); }
    ~ClientItem() {}

    QString xtext( int column ) const;

    virtual void paintCell(QPainter* p, const QColorGroup& cg, int col, int w, int align);

    void setFilesListed(bool listed);

private:
    bool filesListed;
};

class ResultInfo;

class ClientFile : public InfoItem
{
public:
    ClientFile(KListView* parent, int file) : InfoItem(parent, file) { refresh(); }
    ~ClientFile() {}

    QString xtext(int column) const;
    double numeric( int col ) const;
    bool isNumeric(int col) const;
};

QString humanReadableSize(int64 rsz);
QString humanReadableSpeed(double sp);
QString calculateETA(FileInfo* fi);
QString humanReadablePriority(int pri);

#endif
