/* -*- c++ -*-
 *
 * kcmdonkey.cpp
 *
 * Copyright (C) 2002 Petter E. Stokke <gibreel@gibreel.net>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 */

#include <qwidget.h>
#include <qvariant.h>
#include <qlayout.h>
#include <qframe.h>
#include <qgroupbox.h>
#include <qlabel.h>
#include <qtooltip.h>
#include <qwhatsthis.h>
#include <qcheckbox.h>

#include <kdebug.h>
#include <klocale.h>
#include <kglobal.h>
#include <kconfig.h>
#include <kaboutdata.h>
#include <klineedit.h>
#include <klistbox.h>
#include <knuminput.h>
#include <kpassdlg.h>
#include <kpushbutton.h>
#include <kmessagebox.h>
#include <kdirselectdialog.h>
#include <dcopclient.h>

#include "version.h"
#include "kcmdonkey.h"
#include "mobilemule.h"
#include "kcmdonkey.moc"

extern "C"
{
    KCModule *create_kcmdonkey(QWidget* parent, const char*) {
        return new KCMDonkey(parent, "kmldonkey");
    }
}

DonkeyEntry::DonkeyEntry(QListBox* box, const QString& name, const QString& address,
			 int telnetPort, int guiPort, const QString& username, const QString& password,
			 bool isLocal, const QString& temp, const QString& incoming,
			 bool isDefault)
    : QListBoxText(box, name)
    , m_name(name)
    , m_address(address)
    , m_username(username)
    , m_password(password)
    , m_temp(temp)
    , m_incoming(incoming)
    , m_telnetPort(telnetPort)
    , m_guiPort(guiPort)
    , m_default(isDefault)
    , m_local(isLocal)
{
    if (isDefault)
	setText(m_name + i18n(" (Default)"));
}

void DonkeyEntry::setName(const QString& name)
{
    m_name = name;
    if (m_default)
	setText(m_name + i18n(" (Default)"));
    else
	setText(m_name);
}

void DonkeyEntry::setDefault(bool isDefault)
{
    if (isDefault == m_default) return;
    m_default = isDefault;
    setName(m_name);
}



KCMDonkey::KCMDonkey( QWidget *parent, const char *name )
        : KCModule( parent, name )
{
    processSignals = true;
    activeEntry = 0;

    QVBoxLayout* KCMDonkeyLayout = new QVBoxLayout( this, 11, 6, "KCMDonkeyLayout");

    QGroupBox* groupBox1 = new QGroupBox( this, "groupBox1" );
    groupBox1->setColumnLayout(0, Qt::Vertical );
    groupBox1->layout()->setSpacing( 6 );
    groupBox1->layout()->setMargin( 11 );
    groupBox1->setTitle(i18n("MLDonkey Hosts"));
    QHBoxLayout* groupBox1Layout = new QHBoxLayout( groupBox1->layout() );
    groupBox1Layout->setAlignment( Qt::AlignTop );

    hostList = new KListBox( groupBox1, "hostList" );
    hostList->setSizePolicy( QSizePolicy( (QSizePolicy::SizeType)7, (QSizePolicy::SizeType)7, 1, 0, hostList->sizePolicy().hasHeightForWidth() ) );
    QWhatsThis::add(hostList, i18n("This is the list of MLDonkey hosts that have been configured for KMLDonkey to use. Select one to modify it."));
    groupBox1Layout->addWidget( hostList );

    QFrame* frame4 = new QFrame( groupBox1, "frame4" );
    frame4->setSizePolicy( QSizePolicy( (QSizePolicy::SizeType)7, (QSizePolicy::SizeType)7, 2, 0, frame4->sizePolicy().hasHeightForWidth() ) );
    frame4->setFrameShape( QFrame::NoFrame );
    frame4->setFrameShadow( QFrame::Plain );
    frame4->setLineWidth( 0 );
    QVBoxLayout* frame4Layout = new QVBoxLayout( frame4, 0, 0, "frame4Layout");

    QVBoxLayout* layout6 = new QVBoxLayout( 0, 0, 6, "layout6");

    QGridLayout* layout5 = new QGridLayout( 0, 1, 1, 0, 6, "layout5");

    nameEntry = new KLineEdit( frame4, "nameEntry" );
    QWhatsThis::add(nameEntry, i18n("The name of the host is simply a name you use for yourself to identify it. "
				    "You can enter anything you like here, as long as it makes sense to you."));

    layout5->addWidget( nameEntry, 0, 1 );

    addressEntry = new KLineEdit( frame4, "addressEntry" );
    QWhatsThis::add(addressEntry, i18n("The address of the MLDonkey you want to connect to. If this is the machine "
				       "you are currently using, enter \"localhost\". Otherwise, enter the Internet "
				       "address of the machine running your MLDonkey."));

    layout5->addWidget( addressEntry, 1, 1 );

    telnetPortEntry = new KIntNumInput( frame4, "telnetPortEntry" );
    telnetPortEntry->setValue( 4000 );
    telnetPortEntry->setMinValue( 0 );
    telnetPortEntry->setMaxValue( 65535 );
    QWhatsThis::add(telnetPortEntry, i18n("The port on which MLDonkey listens for telnet connections. "
					  "This is normally port 4000."));

    layout5->addWidget( telnetPortEntry, 2, 1 );

    guiPortEntry = new KIntNumInput( frame4, "guiPortEntry" );
    guiPortEntry->setValue( 4001 );
    guiPortEntry->setMinValue( 0 );
    guiPortEntry->setMaxValue( 65535 );
    QWhatsThis::add(guiPortEntry, i18n("The port on which MLDonkey listens for GUI connections. "
				       "This is normally port 4001."));

    layout5->addWidget( guiPortEntry, 3, 1 );

    usernameEntry = new KLineEdit( frame4, "usernameEntry" );
    QWhatsThis::add(usernameEntry, i18n("The user name you want to use to log into MLDonkey with. "
					"You can set up multiple user accounts, but by default MLDonkey "
					"comes with the user \"admin\" already set up."));

    layout5->addWidget( usernameEntry, 4, 1 );

    passwordEntry = new KPasswordEdit( frame4, "passwordEntry" );
    QWhatsThis::add(passwordEntry, i18n("The password associated with the user name you have entered above. "
					"If you are using the default \"admin\" user, the password should be left blank."));

    layout5->addWidget( passwordEntry, 5, 1 );

    addressLabel = new QLabel( frame4, "addressLabel" );
    addressLabel->setText( i18n( "&Address:" ) );
    addressLabel->setAlignment( int( QLabel::AlignVCenter | QLabel::AlignRight ) );
    QWhatsThis::add(addressLabel, QWhatsThis::textFor(addressEntry));

    layout5->addWidget( addressLabel, 1, 0 );

    guiPortLabel = new QLabel( frame4, "guiPortLabel" );
    guiPortLabel->setText( i18n( "&GUI port:" ) );
    guiPortLabel->setAlignment( int( QLabel::AlignVCenter | QLabel::AlignRight ) );
    QWhatsThis::add(guiPortLabel, QWhatsThis::textFor(guiPortEntry));

    layout5->addWidget( guiPortLabel, 3, 0 );

    telnetPortLabel = new QLabel( frame4, "telnetPortLabel" );
    telnetPortLabel->setText( i18n( "&Telnet port:" ) );
    telnetPortLabel->setAlignment( int( QLabel::AlignVCenter | QLabel::AlignRight ) );
    QWhatsThis::add(telnetPortLabel, QWhatsThis::textFor(telnetPortEntry));

    layout5->addWidget( telnetPortLabel, 2, 0 );

    passwordLabel = new QLabel( frame4, "passwordLabel" );
    passwordLabel->setText( i18n( "&Password:" ) );
    passwordLabel->setAlignment( int( QLabel::AlignVCenter | QLabel::AlignRight ) );
    QWhatsThis::add(passwordLabel, QWhatsThis::textFor(passwordEntry));

    layout5->addWidget( passwordLabel, 5, 0 );

    nameLabel = new QLabel( frame4, "nameLabel" );
    nameLabel->setText( i18n( "&Name:" ) );
    nameLabel->setAlignment( int( QLabel::AlignVCenter | QLabel::AlignRight ) );
    QWhatsThis::add(nameLabel, QWhatsThis::textFor(nameEntry));

    layout5->addWidget( nameLabel, 0, 0 );

    usernameLabel = new QLabel( frame4, "usernameLabel" );
    usernameLabel->setText( i18n( "&User name:" ) );
    usernameLabel->setAlignment( int( QLabel::AlignVCenter | QLabel::AlignRight ) );
    QWhatsThis::add(usernameLabel, QWhatsThis::textFor(usernameEntry));

    layout5->addWidget( usernameLabel, 4, 0 );
    layout6->addLayout( layout5 );

    QGroupBox* localGroup = new QGroupBox(i18n("Local Settings"), frame4);
    localGroup->setColumnLayout(0, Qt::Vertical );
    localGroup->layout()->setSpacing( 6 );
    localGroup->layout()->setMargin( 11 );
    QGridLayout* localLayout = new QGridLayout(localGroup->layout());
    localLayout->setAlignment( Qt::AlignTop );

    localMode = new QCheckBox(i18n("M&LDonkey is running locally"), localGroup);
    localLayout->addMultiCellWidget(localMode, 0, 0, 0, 2);

    QLabel* tempLabel = new QLabel(i18n( "Temp directory:" ), localGroup);
    localLayout->addWidget(tempLabel, 1, 0);
    tempPath = new KLineEdit(localGroup);
    localLayout->addWidget(tempPath, 1, 1);
    KPushButton* tempButton = new KPushButton(KGuiItem(QString::null, "fileopen"), localGroup);
    connect(tempButton, SIGNAL(clicked()), SLOT(tempPathDialog()));
    localLayout->addWidget(tempButton, 1, 2);

    QLabel* incomingLabel = new QLabel(i18n( "Incoming directory:" ), localGroup);
    localLayout->addWidget(incomingLabel, 2, 0);
    incomingPath = new KLineEdit(localGroup);
    localLayout->addWidget(incomingPath, 2, 1);
    KPushButton* incomingButton = new KPushButton(KGuiItem(QString::null, "fileopen"), localGroup);
    connect(incomingButton, SIGNAL(clicked()), SLOT(incomingPathDialog()));
    localLayout->addWidget(incomingButton, 2, 2);

    connect(localMode, SIGNAL(toggled(bool)), tempPath, SLOT(setEnabled(bool)));
    connect(localMode, SIGNAL(toggled(bool)), tempButton, SLOT(setEnabled(bool)));
    connect(localMode, SIGNAL(toggled(bool)), incomingPath, SLOT(setEnabled(bool)));
    connect(localMode, SIGNAL(toggled(bool)), incomingButton, SLOT(setEnabled(bool)));
    localMode->setChecked(true);
    localMode->setChecked(false);

    layout6->addWidget(localGroup);

    QSpacerItem* spacer = new QSpacerItem( 210, 50, QSizePolicy::Minimum, QSizePolicy::Expanding );
    layout6->addItem( spacer );

    QHBoxLayout* layout3 = new QHBoxLayout( 0, 0, 6, "layout3");
    QSpacerItem* spacer_2 = new QSpacerItem( 91, 91, QSizePolicy::Expanding, QSizePolicy::Minimum );
    layout3->addItem( spacer_2 );

    QVBoxLayout* layout2 = new QVBoxLayout( 0, 0, 6, "layout2");

    defaultButton = new KPushButton(KGuiItem(i18n("&Set as Default"), "kmldonkey"), frame4, "defaultButton");
    defaultButton->setEnabled(false);
    QWhatsThis::add(defaultButton, i18n("Press this button to make the selected host the default host. "
					"The default host is the host to which ED2K links will be sent, "
					"and to which KMLDonkey will connect by default."));
    layout2->addWidget( defaultButton );

    newButton = new KPushButton(KGuiItem(i18n("New &Host"), "edit_add"), frame4, "newButton");
    QWhatsThis::add(newButton, i18n("Press this button to add a new host."));
    layout2->addWidget( newButton );

    deleteButton = new KPushButton(KGuiItem(i18n("&Delete Host"), "editdelete"), frame4, "deleteButton");
    defaultButton->setEnabled(false);
    QWhatsThis::add(deleteButton, i18n("Press this button to delete the currently selected host."));
    layout2->addWidget( deleteButton );
    layout3->addLayout( layout2 );
    layout6->addLayout( layout3 );
    frame4Layout->addLayout( layout6 );
    groupBox1Layout->addWidget( frame4 );
    KCMDonkeyLayout->addWidget( groupBox1 );
    
    mm = new MobileMuleConfig(this);
    KCMDonkeyLayout->addWidget(mm);
    
    resize( QSize(600, 480).expandedTo(minimumSizeHint()) );

    // buddies
    telnetPortLabel->setBuddy( telnetPortEntry );
    addressLabel->setBuddy( addressEntry );
    guiPortLabel->setBuddy( guiPortEntry );
    passwordLabel->setBuddy( passwordEntry );
    nameLabel->setBuddy( nameEntry );
    usernameLabel->setBuddy( usernameEntry );

    setCaption(i18n("MLDonkey Host Configuration"));

    connect(hostList, SIGNAL(highlighted(QListBoxItem*)), SLOT(entrySelected(QListBoxItem*)));

    connect(nameEntry, SIGNAL(textChanged(const QString&)), SLOT(nameChanged(const QString&)));
    connect(addressEntry, SIGNAL(textChanged(const QString&)), SLOT(addressChanged(const QString&)));
    connect(usernameEntry, SIGNAL(textChanged(const QString&)), SLOT(usernameChanged(const QString&)));
    connect(passwordEntry, SIGNAL(textChanged(const QString&)), SLOT(passwordChanged(const QString&)));
    connect(telnetPortEntry, SIGNAL(valueChanged(int)), SLOT(telnetPortChanged(int)));
    connect(guiPortEntry, SIGNAL(valueChanged(int)), SLOT(guiPortChanged(int)));
    connect(localMode, SIGNAL(toggled(bool)), SLOT(localModeChanged(bool)));
    connect(tempPath, SIGNAL(textChanged(const QString&)), SLOT(tempChanged(const QString&)));
    connect(incomingPath, SIGNAL(textChanged(const QString&)), SLOT(incomingChanged(const QString&)));
    connect(defaultButton, SIGNAL(clicked()), SLOT(defaultButtonClicked()));
    connect(newButton, SIGNAL(clicked()), SLOT(newButtonClicked()));
    connect(deleteButton, SIGNAL(clicked()), SLOT(deleteButtonClicked()));
    connect(mm, SIGNAL(changed()), SLOT(configChanged()));

    load();
}

KCMDonkey::~KCMDonkey()
{}

void KCMDonkey::load()
{
    bool foo = false;

    KConfig *config = new KConfig("mldonkeyrc", false, false);

    hostList->clear();
    defaultEntry = 0;

    QStringList list = config->groupList();
    QStringList::iterator it;
    for (it = list.begin(); it != list.end(); ++it) {
	bool isDefault;
	config->setGroup(*it);
	DonkeyEntry* entry = new DonkeyEntry(hostList, *it,
					     config->readEntry("DonkeyHost", "localhost"),
					     config->readNumEntry("DonkeyPort", 4000),
					     config->readNumEntry("DonkeyGuiPort", 4001),
					     config->readEntry("DonkeyUsername", "admin"),
					     config->readEntry("DonkeyPassword"),
					     config->readBoolEntry("LocalHost", false),
					     config->readPathEntry("TempPath"),
					     config->readPathEntry("IncomingPath"),
					     isDefault = config->readBoolEntry("Default"));
	if (isDefault)
	    if (defaultEntry)
		entry->setDefault(false);
	    else
		defaultEntry = entry;
    }

    if (!hostList->count()) {
	defaultEntry = new DonkeyEntry(hostList, "MLDonkey", "localhost", 4000, 4001, "admin", "", true);
	foo = true;
	KMessageBox::queuedMessageBox(this, KMessageBox::Information, i18n("Your KMLDonkey host list was empty, "
									   "so I have created a default entry for you. "
									   "Please verify that it contains the "
									   "correct information."));
    }

    if (!defaultEntry && hostList->count())
	setDefaultEntry(static_cast<DonkeyEntry*>(hostList->firstItem()));

    hostList->sort();

    if (defaultEntry) {
	hostList->setCurrentItem(defaultEntry);
	hostList->centerCurrentItem();
    }

    delete config;
    
    config = new KConfig("mobilemulerc", false, false);
    config->setGroup("MobileMule");
    mm->activateBox->setChecked(config->readBoolEntry("Autostart", false));
    mm->portEntry->setValue(config->readNumEntry("Port", 4081));
    mm->passwordEntry->setText(config->readEntry("Password", ""));
    delete config;

    if (foo) save();

    emit changed(false);
}

void KCMDonkey::setDefaultEntry(DonkeyEntry* entry)
{
    if (defaultEntry) defaultEntry->setDefault(false);
    entry->setDefault(true);
    defaultEntry = entry;
    hostList->triggerUpdate(false);
}

void KCMDonkey::entrySelected(QListBoxItem* item)
{
    if (!processSignals) return;
    processSignals = false;
    if (item) {
	DonkeyEntry* entry = static_cast<DonkeyEntry*>(item);
	nameEntry->setText(entry->getName());
	addressEntry->setText(entry->getAddress());
	telnetPortEntry->setValue(entry->getTelnetPort());
	guiPortEntry->setValue(entry->getGuiPort());
	usernameEntry->setText(entry->getUsername());
	passwordEntry->erase();
	passwordEntry->insert(entry->getPassword());
	localMode->setChecked(entry->getLocalMode());
	tempPath->setText(entry->getTemp());
	incomingPath->setText(entry->getIncoming());
	defaultButton->setEnabled(!entry->getDefault());
	deleteButton->setEnabled(true);
	activeEntry = entry;
	if (!entry->isSelected())
	    hostList->setSelected(entry, true);
    } else {
	nameEntry->setText(QString::null);
	addressEntry->setText(QString::null);
	telnetPortEntry->setValue(0);
	guiPortEntry->setValue(0);
	usernameEntry->setText(QString::null);
	passwordEntry->erase();
	localMode->setChecked(false);
	tempPath->setText(QString::null);
	incomingPath->setText(QString::null);
	defaultButton->setEnabled(false);
	deleteButton->setEnabled(false);
	activeEntry = 0;
    }
    processSignals = true;
}

DonkeyEntry* KCMDonkey::getActiveEntry()
{
    return activeEntry;
}

void KCMDonkey::nameChanged(const QString& text)
{
    DonkeyEntry* entry = getActiveEntry(); if (!entry || !processSignals) return;
    entry->setName(text);
    hostList->triggerUpdate(false);
    configChanged();
}

void KCMDonkey::addressChanged(const QString& text)
{
    DonkeyEntry* entry = getActiveEntry(); if (!entry || !processSignals) return;
    entry->setAddress(text);
    configChanged();
}

void KCMDonkey::usernameChanged(const QString& text)
{
    DonkeyEntry* entry = getActiveEntry(); if (!entry || !processSignals) return;
    entry->setUsername(text);
    configChanged();
}

void KCMDonkey::passwordChanged(const QString&)
{
    DonkeyEntry* entry = getActiveEntry(); if (!entry || !processSignals) return;
    entry->setPassword(passwordEntry->password());
    configChanged();
}

void KCMDonkey::telnetPortChanged(int value)
{
    DonkeyEntry* entry = getActiveEntry(); if (!entry || !processSignals) return;
    entry->setTelnetPort(value);
    configChanged();
}

void KCMDonkey::guiPortChanged(int value)
{
    DonkeyEntry* entry = getActiveEntry(); if (!entry || !processSignals) return;
    entry->setGuiPort(value);
    configChanged();
}

void KCMDonkey::localModeChanged(bool value)
{
    DonkeyEntry* entry = getActiveEntry(); if (!entry || !processSignals) return;
    entry->setLocalMode(value);
    configChanged();
}

void KCMDonkey::tempChanged(const QString& value)
{
    DonkeyEntry* entry = getActiveEntry(); if (!entry || !processSignals) return;
    entry->setTemp(value);
    configChanged();
}

void KCMDonkey::incomingChanged(const QString& value)
{
    DonkeyEntry* entry = getActiveEntry(); if (!entry || !processSignals) return;
    entry->setIncoming(value);
    configChanged();
}

void KCMDonkey::defaultButtonClicked()
{
    DonkeyEntry* entry = getActiveEntry(); if (!entry || entry->getDefault()) return;
    setDefaultEntry(entry);
    configChanged();
}

void KCMDonkey::newButtonClicked()
{
    DonkeyEntry* entry = new DonkeyEntry(hostList, i18n( "New host" ), "", 4000, 4001, "", "", false);
    if (!defaultEntry) setDefaultEntry(entry);
    hostList->setSelected(entry, true);
    configChanged();
}

void KCMDonkey::deleteButtonClicked()
{
    DonkeyEntry* entry = getActiveEntry(); if (!entry) return;
    if (defaultEntry == entry) {
	delete entry;
	defaultEntry = 0;
	if (hostList->count())
	    setDefaultEntry(static_cast<DonkeyEntry*>(hostList->firstItem()));
	else
	    defaultEntry = 0;
    } else {
	delete entry;
    }
    configChanged();
}

void KCMDonkey::tempPathDialog()
{
    KURL dir = KDirSelectDialog::selectDirectory(tempPath->text(), false, this, i18n("Select Temp Directory"));
    if (!dir.isEmpty()) tempPath->setText(dir.url());
}

void KCMDonkey::incomingPathDialog()
{
    KURL dir = KDirSelectDialog::selectDirectory(incomingPath->text(), false, this, i18n("Select Incoming Directory"));
    if (!dir.isEmpty()) incomingPath->setText(dir.url());
}

void KCMDonkey::defaults()
{
    configChanged();
}

void KCMDonkey::save()
{
    KConfig *config = new KConfig("mldonkeyrc", false, false);

    QStringList list = config->groupList();
    QStringList::iterator git;
    for (git = list.begin(); git != list.end(); ++git)
	config->deleteGroup(*git);

    QListBoxItem* it = hostList->firstItem();
    for (; it; it = it->next()) {
	DonkeyEntry* entry = static_cast<DonkeyEntry*>(it);
	config->setGroup(entry->getName());
	config->writeEntry("DonkeyHost", entry->getAddress());
	config->writeEntry("DonkeyPort", entry->getTelnetPort());
	config->writeEntry("DonkeyGuiPort", entry->getGuiPort());
	config->writeEntry("DonkeyUsername", entry->getUsername());
	config->writeEntry("DonkeyPassword", entry->getPassword());
	config->writeEntry("LocalHost", entry->getLocalMode());
	config->writePathEntry("TempPath", entry->getTemp());
	config->writePathEntry("IncomingPath", entry->getIncoming());
	config->writeEntry("Default", entry->getDefault());
    }

    config->sync();
    delete config;
    
    config = new KConfig("mobilemulerc", false, false);
    config->setGroup("MobileMule");
    config->writeEntry("Autostart", mm->activateBox->isChecked());
    config->writeEntry("Port", mm->portEntry->value());
    config->writeEntry("Password", mm->passwordEntry->text());
    config->sync();
    delete config;

    startMobileMuleService(mm->activateBox->isChecked());

    emit changed(true);
}

int KCMDonkey::buttons()
{
    return KCModule::Default | KCModule::Apply;
// | KCModule::Help;
}

void KCMDonkey::configChanged()
{
    emit changed(true);
}

bool KCMDonkey::startMobileMuleService(bool start)
{
    DCOPClient* client = DCOPClient::mainClient();
    QByteArray data;
    QDataStream arg(data, IO_WriteOnly);
    arg << start;
    return client->send("kded", "mobilemule", "setEnabled(bool)", data);
}

QString KCMDonkey::quickHelp() const
{
    return i18n("This is where you configure the connectivity for the KMLDonkey suite - "
		"meaning the network location of your mldonkey core, and your username and password. "
		"If none of that makes any sense to you, the defaults are probably good.");
}

const KAboutData* KCMDonkey::aboutData() const
{
    KAboutData *about = new KAboutData("kcmdonkey", I18N_NOOP("KMLDonkey Configuration"), KMLDONKEY_VERSION,
				       I18N_NOOP("An MLDonkey frontend for the KDE desktop."),
				       KAboutData::License_GPL_V2, I18N_NOOP("Copyright (C) 2003, 2004 Petter E. Stokke"),
				       I18N_NOOP("Part of the KMLDonkey package."),
				       "http://www.gibreel.net/projects/kmldonkey/");
    about->addAuthor("Petter E. Stokke", I18N_NOOP("Lead developer"), "gibreel@gibreel.net");
    about->addAuthor("Sebastian Sauer", I18N_NOOP("Developer"), "mail@dipesh.de");
    about->setTranslator(I18N_NOOP("_: NAME OF TRANSLATORS\nYour names"), I18N_NOOP("_: EMAIL OF TRANSLATORS\nYour emails"));

    return about;
}

