// Sine_test.cpp
//
// Copyright 2011-2013 Roan Trail, Inc.
//
// This file is part of Kinetophone.
//
// Kinetophone is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published
// by the Free Software Foundation, either version 2 of the License,
// or (at your option) any later version.
//
// Kinetophone is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// General Public License for more details.  You should have received
// a copy of the GNU General Public License along with Kinetophone. If
// not, see <http://www.gnu.org/licenses/>.

//
// This code was based on the PortAudio file patest_sine.c:
//
//   This program uses the PortAudio Portable Audio Library.
//   For more information see: http://www.portaudio.com/
//   Copyright (c) 1999-2000 Ross Bencina and Phil Burk
//
//   Permission is hereby granted, free of charge, to any person obtaining
//   a copy of this software and associated documentation files
//   (the "Software"), to deal in the Software without restriction,
//   including without limitation the rights to use, copy, modify, merge,
//   publish, distribute, sublicense, and/or sell copies of the Software,
//   and to permit persons to whom the Software is furnished to do so,
//   subject to the following conditions:
//
//   The above copyright notice and this permission notice shall be
//   included in all copies or substantial portions of the Software.
//
//   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
//   EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
//   MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
//   IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR
//   ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF
//   CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
//   WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

#include <kinetophone/error/Portaudio_error.hpp>
#include <portaudio.h>
#include <iostream>
#include <string>
#include <cmath>

using std::cout;
using std::endl;
using std::string;
using namespace Roan_trail::Kinetophone;
//
// Internal constants and helper functions
//

namespace
{
  const int ic_channels = 2;
  const int ic_device = 6;
  const int ic_number_seconds = 10000;
  const int ic_sample_rate = 44100;
  const double ic_sine_frequency = 1000.0;
  const double ic_sine_amplitude = 1.0;
  const int ic_frames_per_buffer = 64;
  const int ic_pi = 3.14159265;
  const int ic_table_size = ic_sample_rate; // could be optimized smaller
  typedef struct is_PA_test_data_struct
  {
    is_PA_test_data_struct();
    double sine[ic_table_size];
    int pos[ic_channels];
    float amplitude[ic_channels];
    string message;
  } is_PA_test_data;

  is_PA_test_data_struct::is_PA_test_data_struct()
  {
    for (int channel = 0; channel < ic_channels; ++channel)
    {
      pos[channel] = 0;
      amplitude[channel] = 1.0;
    }
  }

  // This routine will be called by the PortAudio engine when audio is needed.
  // It may called at interrupt level on some machines so don't do anything
  // that could mess up the system like calling malloc() or free().
  static int ih_audio_callback(const void* input_buffer,
                               void* output_buffer,
                               unsigned long frames_per_buffer,
                               const PaStreamCallbackTimeInfo* time_info,
                               PaStreamCallbackFlags status_flags,
                               void* user_data)
  {
    // Avoid unused warnings
    static_cast<void>(input_buffer);
    static_cast<void>(time_info);
    static_cast<void>(status_flags);

    is_PA_test_data* data = reinterpret_cast<is_PA_test_data*>(user_data);
    for (int channel = 0; channel < ic_channels; ++channel)
    {
      float* out = reinterpret_cast<float*>(output_buffer) + channel; // interleaved samples
      float amplitude = data->amplitude[channel];
      int &position = data->pos[channel];
      for (unsigned long i = 0; i < frames_per_buffer; ++i)
      {
        *out = amplitude * data->sine[position];
        ++position;
        if (position >= ic_table_size)
        {
          position -= ic_table_size;
        }
        out += ic_channels;
      }
    }

    return paContinue;
  }

  // This routine is called by portaudio when playback is done.
  static void ih_stream_finished(void* user_data)
  {
    is_PA_test_data* data = reinterpret_cast<is_PA_test_data*>(user_data);
    cout << "Stream completed: " << data->message << endl;
  }
}

int main()
{
    PaError PA_error;

    start_error_block();

    cout << "Sine test: output sine wave. " << endl;
    cout << "Sample rate = " << ic_sample_rate << endl;
    cout << "Frequency = " << ic_sine_frequency << endl;
    cout << "Amplitude = " << ic_sine_amplitude << endl;
    cout << "Buffer size = " << ic_frames_per_buffer << endl;

    // initialize sine wave table
    is_PA_test_data data;
    for(int i = 0; i < ic_table_size; i++)
    {
      double t = (static_cast<double>(i) / static_cast<double>(ic_sample_rate));
      data.sine[i] = ic_sine_amplitude * sin(t * ic_pi * 2.0 * ic_sine_frequency);
    }

    if (ic_channels > 1)
    {
      // data.pos[1] = 10; // different phase for Ch. 1, can be removed
      data.amplitude[1] = 0.5 * ic_sine_amplitude; // half amplitude for Ch. 1, can be removed
    }

    PaError PA_error = Pa_Initialize();
    on_error(paNoError != PA_error, new Portaudio_error(PA_error, "Pa_Initialize"));

    PaStreamParameters output_param;
    output_param.device = ic_device; // Pa_GetDefaultOutputDevice();
    on_error(output_param.device == paNoDevice, new Portaudio_error(PA_error, "Pa_GetDefaultOutputDevice"));

    output_param.channelCount = ic_channels;
    output_param.sampleFormat = paFloat32; // 32 bit floating point output
    output_param.suggestedLatency = Pa_GetDeviceInfo(output_param.device)->defaultLowOutputLatency;
    output_param.hostApiSpecificStreamInfo = 0;

    PaStream *stream;
    PA_error = Pa_OpenStream(&stream,
                             0, // no input
                             &output_param,
                             ic_sample_rate,
                             ic_frames_per_buffer,
                             paClipOff, // out of range samples not output, so don't bother clipping
                             ih_audio_callback,
                             &data);
    on_error(paNoError != PA_error, new Portaudio_error(PA_error, "Pa_OpenStream"));

    data.message = "No message.";

    PA_error = Pa_SetStreamFinishedCallback( stream, &ih_stream_finished );
    on_error(paNoError != PA_error, new Portaudio_error(PA_error, "Pa_SetStreamFinishedCallback"));

    PA_error = Pa_StartStream(stream);
    on_error(paNoError != PA_error, new Portaudio_error(PA_error, "Pa_StartStream"));

    cout << "Play for " << ic_number_seconds << " seconds." << endl;
    Pa_Sleep(ic_number_seconds * 1000);

    PA_error = Pa_StopStream(stream);
    on_error(paNoError != PA_error, new Portaudio_error(PA_error, "Pa_StopStream"));

    PA_error = Pa_CloseStream(stream);
    on_error(paNoError != PA_error, new Portaudio_error(PA_error, "Pa_CloseStream"));

    cout << "Test finished." << endl;

    goto exit_point;

    end_error_block();

 error_handler:
    cout << "An error occured while using the portaudio stream." << endl;
    cout << "  Error code: " << handler_error->code() << endl;
    cout << "  Error message: " << handler_error->error_dictionary()[Error::diagnostic_error_key] << endl;
    goto error_cleanup;

 error_cleanup:
    delete handler_error;
    goto exit_point;

 exit_point:
    Pa_Terminate();
    return PA_error;
}
