// Kinetophone_vox_config.cpp
//
// Copyright 2011-2013 Roan Trail, Inc.
//
// This file is part of Kinetophone.
//
// Kinetophone is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published
// by the Free Software Foundation, either version 2 of the License,
// or (at your option) any later version.
//
// Kinetophone is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// General Public License for more details.  You should have received
// a copy of the GNU General Public License along with Kinetophone. If
// not, see <http://www.gnu.org/licenses/>.

#include "Kinetophone_vox_config.hpp"
#include <kinetophone/Speech_synthesizer_config.hpp>
#include <kinetophone/Sound_file_config.hpp>
#include <kinetophone/File_manager.hpp>
#include <kinetophone/option_checks.hpp>
#include <kinetophone/Option_validators.hpp>
#include <kinetophone/error/Boost_error.hpp>
#include <kinetophone/error/Kinetophone_error.hpp>
#include <boost/program_options.hpp>
#include <boost/concept/assert.hpp>
#include <boost/concept_check.hpp>
#include <fstream>
#include <sstream>
#include <string>
#include <cstdlib>

using std::endl;
using std::ifstream;
using std::logic_error;
using std::string;
using std::stringstream;
using Roan_trail::Kinetophone::File_manager;
using Roan_trail::Kinetophone::check_conflicting_options;
using Roan_trail::Kinetophone::check_option_dependency;
using Roan_trail::Kinetophone::Directory_validator;
using Roan_trail::Kinetophone::Positive_int_validator;
using Roan_trail::Kinetophone::Str_to_str_lookup_validator;
using Roan_trail::Kinetophone::Validator;
using Roan_trail::Kinetophone::Sound_file_config;
using Roan_trail::Kinetophone::Speech_synthesizer_config;
using Roan_trail::Kinetophone::Boost_error;
using Roan_trail::Kinetophone::Error_param;
using Roan_trail::Kinetophone::Kinetophone_error;
using namespace Roan_trail::Kinetophone_app;

namespace PO = boost::program_options;

//
// Internal helpers
//

namespace
{

  //
  // some custom validation code for this configuration
  //

  typedef struct Str_to_str_lookup_validator<string, Speech_synthesizer_config::setup_command_for_voice>
  Ih_voice_validator;

  //
  // the main validator struct for this configuration
  //

  struct Ih_validator_struct
  {
    Ih_validator_struct(const Kinetophone_vox_config& s)
      : image_source_path(s.image_source_path),
        image_directory(),
        vox_generation(s.vox_generation),
        detailed_error(Validator<bool>(s.detailed_error)),
        quit_on_pregenerate_warnings(Validator<bool>(s.quit_on_pregenerate_warnings)),
        custom_installation_dir(Directory_validator(s.custom_installation_dir)),
        speech_synthesizer_config(*s.speech_synthesizer_config) {}
    void update(Kinetophone_vox_config& return_cfg)
    {
      return_cfg.detailed_error = detailed_error.v;
      vox_generation.update(return_cfg.vox_generation);
      return_cfg.detailed_error = detailed_error.v;
      return_cfg.quit_on_pregenerate_warnings = quit_on_pregenerate_warnings.v;
      return_cfg.custom_installation_dir = custom_installation_dir.v;
      speech_synthesizer_config.update(*return_cfg.speech_synthesizer_config);
      // calculate and assign remaining members
      if ("" == image_source_path)
      {
        if ("" != image_directory.v)
        {
          return_cfg.image_source_path = image_directory.v;
        }
        else
        {
          // left blank to indicate that the source in the session file should be used
          return_cfg.image_source_path = "";
        }
      }
      else
      {
        return_cfg.image_source_path = image_source_path;
      }
    }
    string image_source_path;
    Directory_validator image_directory;
    struct Vox_generation_validator_struct
    {
      Vox_generation_validator_struct(const Kinetophone_vox_config::Vox_generation_struct& s)
        : input_session_file(s.input_session_file),
          output_session_file(s.output_session_file),
          input_session_file_is_PDF(Validator<bool>(s.input_session_file_is_PDF)),
          quiet(Validator<bool>(s.quiet)),
          verbose(Validator<bool>(s.verbose)),
          skip_generate_warnings(Validator<bool>(s.skip_generate_warnings)),
          overwrite_output_session_file(Validator<bool>(s.overwrite_output_session_file)),
          output_empty_slides(Validator<bool>(s.output_empty_slides)) {}
      void update(Kinetophone_vox_config::Vox_generation_struct& return_s)
      {
        return_s.input_session_file = input_session_file;
        return_s.output_session_file = output_session_file;
        return_s.input_session_file_is_PDF = input_session_file_is_PDF.v;
        return_s.quiet = quiet.v;
        return_s.verbose = verbose.v;
        return_s.skip_generate_warnings = skip_generate_warnings.v;
        return_s.overwrite_output_session_file = overwrite_output_session_file.v;
        return_s.output_empty_slides = output_empty_slides.v;
      }
      string input_session_file;
      string output_session_file;
      Validator<bool> input_session_file_is_PDF;
      Validator<bool> quiet;
      Validator<bool> verbose;
      Validator<bool> skip_generate_warnings;
      Validator<bool> overwrite_output_session_file;
      Validator<bool> output_empty_slides;
    } vox_generation;
    Validator<bool> detailed_error;
    Validator<bool> quit_on_pregenerate_warnings;
    Directory_validator custom_installation_dir;
    struct Speech_synthesizer_config_validator_struct
    {
      Speech_synthesizer_config_validator_struct(const Speech_synthesizer_config& s)
        : voice(Ih_voice_validator(s.voice)),
          sound_file(*s.sound_file),
          file_overwrite(s.file_overwrite) {}
      void update(Speech_synthesizer_config& return_s)
      {
        return_s.voice = voice.v;
        sound_file.update(*return_s.sound_file);
        return_s.file_overwrite = file_overwrite.v;
      }
      //
      Ih_voice_validator voice;
      struct Sound_file_config_validator_struct
      {
        Sound_file_config_validator_struct(const Sound_file_config& s)
          : sample_rate(Positive_int_validator(static_cast<int>(s.sample_rate))),
            file_name(s.file_name) {}
        void update(Sound_file_config& return_s)
        {
          return_s.sample_rate = sample_rate.v;
          return_s.file_name = file_name;
        }
        //
        Positive_int_validator sample_rate;
        string file_name;
      } sound_file;
      Validator<bool> file_overwrite;
    } speech_synthesizer_config;
  };

  //
  // Auxiliary functions for checking input validity.
  //

  void ih_check_option_consistency(const Kinetophone_vox_config& cfg,
                                   const PO::variables_map& VM)
  {
    // conflicting option checks
    check_conflicting_options(VM,
                              "quiet",
                              "verbose");
    check_conflicting_options(VM,
                              "image-directory",
                              "image-source-path");
    // required option checks
    if ("" == cfg.vox_generation.input_session_file)
    {
      throw logic_error(string("input session file not specified"));
    }
    if ("" == cfg.vox_generation.output_session_file)
    {
      throw logic_error(string("output session file not specified"));
    }
  }

  typedef struct
  {
    PO::options_description& cmd_line_desc;
    PO::options_description& basic_desc;
    PO::options_description& advanced_desc;
  } Ih_desc_params;

  // sets up the program option description object
  void ih_setup_descriptions(Ih_desc_params& return_desc_params,
                             Ih_validator_struct& return_v,
                             string& return_config_file)
  {
    const char *home_dir = getenv("HOME"); // (check_code_ignore)
    string default_config_file = (!home_dir || (*home_dir == 0) || !File_manager::path_is_directory(home_dir)) ?
      string("") : string(home_dir) + string("/.kinetophone");

    return_desc_params.cmd_line_desc.add_options()
      ("help,h", "output help message")
      ("config,i", PO::value<string>(&return_config_file)->default_value(default_config_file),
       "specify configuration file")
      ("version,v", "print program version")
      ;

    return_desc_params.basic_desc.add_options()
      ("voice,w", PO::value<Ih_voice_validator>(&return_v.speech_synthesizer_config.voice),
       "set voice for speech synthesizer\n"
       "{ en_br_m (British English male)\n"
       "| en_us_m (American English male)\n"
       "| en_us_m_alt (American English male alternate)\n"
       "| it_m (Italian male)\n"
       "| it_f (Italian female)\n"
       "| es_m (Spanish male)}")
      ("input-session-file,I", PO::value<string>(&return_v.vox_generation.input_session_file)
       ->default_value(""),
       "set input session file")
      ("output-session-file,O", PO::value<string>(&return_v.vox_generation.output_session_file)
       ->default_value(""),
       "set input session file")
      ("image-directory,D", PO::value<Directory_validator>(&return_v.image_directory),
       "set image source directory (overrides session)")
      ("image-source-path,P", PO::value<string>(&return_v.image_source_path)
       ->default_value(return_v.image_source_path),
       "set image source path (directory or PDF file, overrides session file)")
      ("sound-file,f", PO::value<string>(&return_v.speech_synthesizer_config.sound_file.file_name)
       ->default_value(return_v.speech_synthesizer_config.sound_file.file_name),
       "set destination sound file path")
      ("overwrite,o", PO::value<Validator<bool> >(&return_v.speech_synthesizer_config.file_overwrite)
       ->implicit_value(Validator<bool>(true)),
       "overwrite output sound recording file")
      ("overwrite-session-file,W",
       PO::value<Validator<bool> >(&return_v.vox_generation.overwrite_output_session_file)
       ->implicit_value(Validator<bool>(true)),
       "overwrite output session file")
      ("sample-rate,s",
       PO::value<Positive_int_validator>(&return_v.speech_synthesizer_config.sound_file.sample_rate)
       ->default_value(return_v.speech_synthesizer_config.sound_file.sample_rate),
       "set samples per second (must be > 0)")
      ("input-session-file-is-PDF,S",
       PO::value<Validator<bool> >(&return_v.vox_generation.input_session_file_is_PDF)
       ->implicit_value(Validator<bool>(true)),
       "the file specified as the input session file is in PDF format (use its text annotations as notes)")
      ;

    return_desc_params.advanced_desc.add_options()
      ("skip-generate-warnings,k",
       PO::value<Validator<bool> >(&return_v.vox_generation.skip_generate_warnings)
       ->implicit_value(Validator<bool>(true)),
       "do not output vox generation warnings (cannot use with --quit-on-pregenerate-warnings)")
      ("quiet,Q", PO::value<Validator<bool> >(&return_v.vox_generation.quiet)
       ->implicit_value(Validator<bool>(true)),
       "little output, if any, from vox generation (cannot use with --verbose)")
      ("verbose,V", PO::value<Validator<bool> >(&return_v.vox_generation.verbose)
       ->implicit_value(Validator<bool>(true)),
       "talkative vox generation (cannot use with --quiet)")
      ("quit-on-pregenerate-warnings,p",
       PO::value<Validator<bool> >(&return_v.quit_on_pregenerate_warnings)
       ->implicit_value(Validator<bool>(true)),
       "quit, do not continue, if there are pre-generate warning(s) " // (check_code_ignore)
       "(cannot use with --skip-generate-warnings)")
      ("detailed-error,e", PO::value<Validator<bool> >(&return_v.detailed_error)
       ->implicit_value(Validator<bool>(true)),
       "display detailed diagnostics on error")
      ("config-directory,C", PO::value<Directory_validator>(&return_v.custom_installation_dir),
       "set custom program configuration directory")
      ("output-empty-slides,E", PO::value<Validator<bool> >(&return_v.vox_generation.output_empty_slides)
       ->implicit_value(Validator<bool>(true)),
       "output slides with no notes as segments")
      ;
  }
}

//
// Constructor/destructor/copy
//

Kinetophone_vox_config::Kinetophone_vox_config()
  // setup some reasonable defaults
  : help_message(""),
    image_source_path(""), // left blank to indicate that the directory in the session file should be used
    vox_generation(Vox_generation_struct()),
    detailed_error(false),
    quit_on_pregenerate_warnings(false),
    custom_installation_dir(""),
    speech_synthesizer_config(new Speech_synthesizer_config)
{
}

Kinetophone_vox_config::~Kinetophone_vox_config()
{
  delete speech_synthesizer_config;
}

Kinetophone_vox_config::Vox_generation_struct::Vox_generation_struct()
  // setup some reasonable defaults
  : input_session_file(""),
    output_session_file(""),
    input_session_file_is_PDF(false),
    quiet(false),
    verbose(false),
    skip_generate_warnings(false),
    overwrite_output_session_file(false),
    output_empty_slides(false)
{
}

//
//  Class constants
//

//   default
//     command
const int Kinetophone_vox_config::command_error;
const int Kinetophone_vox_config::command_synthesize;
const int Kinetophone_vox_config::command_help;
const int Kinetophone_vox_config::command_output_version;

int Kinetophone_vox_config::parse_program_options(int argc,
                                                  const char** argv,
                                                  Error_param& return_error)
{
  precondition(!return_error());

  int return_value = Kinetophone_vox_config::command_error;

  help_message = "";

  start_error_block();

  try
  {
    PO::options_description cmd_line_desc("Command line only");
    stringstream basic_desc_label;
    basic_desc_label << "Command line and configuration file" << endl << "  Basic";
    PO::options_description basic_desc(basic_desc_label.str());
    PO::options_description advanced_desc("  Advanced");
    Ih_desc_params desc_params =
    {
      cmd_line_desc,
      basic_desc,
      advanced_desc
    };

    Ih_validator_struct validator(*this);
    string config_file;
    ih_setup_descriptions(desc_params,
                          validator,
                          config_file);

    // parse the command line to get config file
    PO::variables_map VM_config;
    PO::store(PO::command_line_parser(argc, const_cast<char**>(argv)).
              options(cmd_line_desc).allow_unregistered().run(), VM_config);
    notify(VM_config);

    PO::options_description config_desc;
    config_desc.add(basic_desc).add(advanced_desc);

    stringstream help_stream;
    help_stream << "Usage: " << endl;
    help_stream << "kinetophone_vox --help" << endl << endl;
    help_stream << "kinetophone_vox --list-devices" << endl << endl;
    help_stream << "kinetophone_vox input-session-file output-session-file [sound-file ] [ options ]";
    help_stream << endl << endl;

    PO::options_description all_desc("Options");
    all_desc.add(cmd_line_desc).add(basic_desc).add(advanced_desc);
    help_stream << all_desc;
    help_message = help_stream.str();

    PO::variables_map VM;

    // parse the command line (overrides config file)
    PO::positional_options_description pos_desc;
    pos_desc.add("input-session-file", 1);
    pos_desc.add("output-session-file", 1);
    pos_desc.add("sound-file", 1);
    PO::store(PO::command_line_parser(argc, const_cast<char**>(argv)).
              options(all_desc).positional(pos_desc).run(), VM);
    PO::notify(VM);

    // parse the config file
    if ("" != config_file)
    {
      ifstream ifs(config_file.c_str());
      bool have_error =  (!ifs && !VM_config["config"].defaulted());
      on_error(have_error, new Boost_error(error_location(), string("cannot open config file: ") + config_file));
      const bool allow_unregistered_options = true;
      PO::store(PO::parse_config_file(ifs, config_desc, allow_unregistered_options), VM);
      PO::notify(VM);
    }

    if (VM.count("help"))
    { // HELP
      return_value = Kinetophone_vox_config::command_help;
    }
    else if (VM.count("version"))
    {
      // PROGRAM VERSION
      return_value = Kinetophone_vox_config::command_output_version;
    }
    else
    { // SYNTHESIZE
      // update the config from the validator
      validator.update(*this);
      ih_check_option_consistency(*this, VM);
      return_value = Kinetophone_vox_config::command_synthesize;
    }
  }
  catch (const std::exception& e)
  {
    string diagnostic = e.what();
    stringstream s;
    s << "Reason: " << diagnostic << std::endl << std::endl;
    s << "Enter kinetophone_vox --help for valid program options.";
    Kinetophone_error* error = new Kinetophone_error(error_location(),
                                                     Kinetophone_error::command_line,
                                                     new Boost_error(error_location(), s.str()));
    on_error(true, error);
  }
  catch (...)
  {
    string diagnostic = "exception parsing command line";
    Kinetophone_error* error = new Kinetophone_error(error_location(),
                                                     Kinetophone_error::command_line,
                                                     diagnostic);
    on_error(true, error);
  }

  goto exit_point;

  end_error_block();

  default_error_handler(return_error);

 error_cleanup:
  if (!return_error.need_error())
  {
    delete handler_error;
  }
  return_value = Kinetophone_vox_config::command_error;
  goto exit_point;

 exit_point:
  return return_value;
}
