// Kinetophone_narrator_model.cpp
//
// Copyright 2011-2013 Roan Trail, Inc.
//
// This file is part of Kinetophone.
//
// Kinetophone is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published
// by the Free Software Foundation, either version 2 of the License,
// or (at your option) any later version.
//
// Kinetophone is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// General Public License for more details.  You should have received
// a copy of the GNU General Public License along with Kinetophone. If
// not, see <http://www.gnu.org/licenses/>.

#include "Kinetophone_narrator_model.hpp"

// include *mm first to avoid conflicts
#include <gdkmm/pixbuf.h>

#include "Kinetophone_narrator_config.hpp"
#include <kinetophone/File_manager.hpp>
#include <kinetophone/Narration_recorder.hpp>
#include <kinetophone/Slide_collection.hpp>
#include <kinetophone/error/Kinetophone_error.hpp>
#include <kinetophone/error/Posix_error.hpp>
#include <kinetophone/Dir_slide_collection_glib.hpp>
#include <kinetophone/PDF_slide_collection_glib.hpp>
#include <libxml++/document.h>
#include <libxml++/exceptions/exception.h>
#include <libxml++/nodes/node.h>
#include <libxml++/parsers/domparser.h>
#include <fstream>
#include <string>
#include <cerrno> // (check_code_ignore)

using xmlpp::Document;
using xmlpp::DomParser;
using xmlpp::Node;
using std::ios_base;
using std::ofstream;
using std::string;
using Roan_trail::Kinetophone::Error_param;
using Roan_trail::Kinetophone::File_manager;
using Roan_trail::Kinetophone::Narration_recorder;
using Roan_trail::Kinetophone::Slide_collection;
using Roan_trail::Kinetophone::source_type_from_XML;
using Roan_trail::Kinetophone::Kinetophone_error;
using Roan_trail::Kinetophone::Posix_error;
using Roan_trail::Kinetophone_glib::Dir_slide_collection_glib;
using Roan_trail::Kinetophone_glib::PDF_slide_collection_glib;
using namespace Roan_trail::Kinetophone_app;

//
// Constructor/destructor
//

Kinetophone_narrator_model::Kinetophone_narrator_model(const Kinetophone_narrator_config& config)
  : m_slides(0),
    m_slides_source(directory_source),
    m_recorder(new Narration_recorder),
    m_dirty(false)
{
  m_slides_source = (File_manager::path_is_directory(config.image_source_path) ?
                     directory_source : PDF_source);

  switch (m_slides_source)
  {
  case directory_source:
    {
      m_slides = new Dir_slide_collection_glib;
    }
    break;
  case PDF_source:
    {
      PDF_slide_collection_glib* slides = new PDF_slide_collection_glib;
      slides->set_import_PDF_notes(config.import_PDF_notes);
      slides->set_fill_color(*config.PDF_fill_color);
      m_slides = slides;
    }
    break;
  default:
    assert(false && "invalid slides source type when creating Kinetophone narrator model");
    break;
  }


  postcondition(mf_invariant(false));
}

Kinetophone_narrator_model::~Kinetophone_narrator_model()
{
  precondition(mf_invariant(false));

  delete m_recorder;
  delete m_slides;
}

//
// Protected member functions
//

bool Kinetophone_narrator_model::mf_invariant(bool check_base_class) const
{
  bool return_value = false;

  if (!m_recorder || !m_slides)
  {
    goto exit_point;
  }

  return_value = true;

 exit_point:
  return return_value;
}

//
// Commands
//

bool Kinetophone_narrator_model::import_from_XML(const string& file_path, Error_param& return_error)
{
  precondition(!return_error()
               && mf_invariant());

  bool return_value = false;

  start_error_block();

  Error_param error;
  string diagnostic_prefix = "error importing XML file: ";

  try
  {
    DomParser parser(file_path); // no XML validation by default
    parser.set_substitute_entities(); // substitute text for entity references
    parser.parse_file(file_path);
    on_error(!parser, new Kinetophone_error(error_location(),
                                            Kinetophone_error::file_IO,
                                            "error importing XML file, could not parse",
                                            file_path));

    const Node* root_node = parser.get_document()->get_root_node();
    on_error(!root_node, new Kinetophone_error(error_location(),
                                               Kinetophone_error::file_IO,
                                               "error importing XML file, could not parse",
                                               file_path));

    // import notes into the slide collection
    const bool slides_imported = m_slides->import_notes_from_XML(file_path,
                                                                 root_node,
                                                                 error);
    on_error(!slides_imported, new Kinetophone_error(error_location(),
                                                     Kinetophone_error::general,
                                                     error()));
  }
  catch (const xmlpp::exception& e)
  {
    const string diagnostic = diagnostic_prefix + e.what();
    on_error(true, new Kinetophone_error(error_location(),
                                         Kinetophone_error::file_IO,
                                         diagnostic,
                                         file_path));
  }

  return_value = true;

  goto exit_point;
  end_error_block();

  default_error_handler_and_cleanup(return_error, return_value, false);

  goto exit_point;

 exit_point:
  postcondition(return_error.is_valid_at_return(return_value)
                && mf_invariant());
  return return_value;
}

bool Kinetophone_narrator_model::export_as_XML(const string& file_path, Error_param& return_error) const
{
  precondition(!return_error()
               && mf_invariant());

  bool return_value = false;

  start_error_block();

  Error_param error;

  Document kinetophone_narrator_document;
  kinetophone_narrator_document.create_root_node("kinetophone_narrator_model");
  Node* XML_root_node = kinetophone_narrator_document.get_root_node();
  assert(XML_root_node && "error getting root node from kinetophone narrator XML document");

  const bool slides_saved = m_slides->save_as_XML(file_path,
                                                  XML_root_node,
                                                  error);
  on_error(!slides_saved, new Kinetophone_error(error_location(),
                                                Kinetophone_error::general,
                                                error()));

  const bool recorder_exported = m_recorder->export_as_XML(file_path,
                                                           XML_root_node,
                                                           error);
  on_error(!recorder_exported, new Kinetophone_error(error_location(),
                                                     Kinetophone_error::general,
                                                     error()));

  const string XML = kinetophone_narrator_document.write_to_string_formatted(); // UTF-8 encoding by default

  ofstream XML_file(file_path.c_str(), ios_base::out | ios_base::trunc);
  on_error(!XML_file, new Kinetophone_error(error_location(),
                                            Kinetophone_error::file_IO,
                                            new Posix_error(errno, // (check_code_ignore)
                                                            "open for write",
                                                            file_path.c_str())));

  XML_file << XML;

  // check for write error
  on_error(!XML_file, new Kinetophone_error(error_location(),
                                            Kinetophone_error::file_IO,
                                            new Posix_error(errno, // (check_code_ignore)
                                                            "write",
                                                            file_path.c_str())));

  return_value = true;

  goto exit_point;
  end_error_block();

  default_error_handler_and_cleanup(return_error,
                                    return_value,
                                    false);
  goto exit_point;

 exit_point:
  postcondition(mf_invariant());
  return return_value;
}
