// Fraction_bar_widget.cpp
//
// Copyright 2011-2013 Roan Trail, Inc.
//
// This file is part of Kinetophone.
//
// Kinetophone is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published
// by the Free Software Foundation, either version 2 of the License,
// or (at your option) any later version.
//
// Kinetophone is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// General Public License for more details.  You should have received
// a copy of the GNU General Public License along with Kinetophone. If
// not, see <http://www.gnu.org/licenses/>.

#include "Fraction_bar_widget.hpp"

// include *mm first to avoid conflicts
#include <gdkmm/drawable.h>
#include <gdkmm/general.h>
#include <gtkmm/container.h>

#include <kinetophone/common.hpp>
#include <kinetophone/error/Error.hpp>
#include <iostream>
#include <cstring>

using Glib::RefPtr;
using Gtk::Allocation;
using Gtk::Container;
using Gtk::Requisition;
using Gtk::Widget;
using Gdk::Rectangle;
using Gdk::Window;
using Roan_trail::Kinetophone::Error_param;
using namespace Gdk::Cairo;
using namespace Roan_trail::Kinetophone_app;

namespace
{
  const int ic_default_bar_length = 120;
  const int ic_default_bar_thickness = 10;
  const int ic_default_border_padding = 1;
  const double ic_default_fraction = 1.0;
  const double ic_default_warning_fraction = 1.0 / 3.0;
  const double ic_default_critical_fraction = 1.0 / 6.0;
  const double ic_default_empty_fraction = 0.05;
  Error_param iv_ignore_error(false);
  bool ih_is_valid_rect(const Rectangle& rect)
  {
    return ((rect.get_x() >= 0)
            && (rect.get_y() >= 0)
            && (rect.get_width() >= 0)
            && (rect.get_height() >= 0));
  }
}

// Note: The GType name will be gtkmm__CustomObject_Fraction_bar_widget
//
// Note: This shows the GType name, which must be used in the RC file.
// shown with  std::cout << "GType name: " << G_OBJECT_TYPE_NAME(gobj()) << std::endl;
//
// Note: This show that the GType still derives from GtkWidget:
// std::cout << "Gtype is a GtkWidget?:" << GTK_IS_WIDGET(gobj()) << std::endl;

Fraction_bar_widget::Fraction_bar_widget(bool enabled,
                                         bool vertical,
                                         double fraction,
                                         double warning_fraction,
                                         double critical_fraction,
                                         double empty_fraction,
                                         int bar_thickness,
                                         int bar_length,
                                         int border_padding)
  : ObjectBase("Fraction_bar_widget"),
    Widget(),
    m_ref_window(0),
    m_enabled(enabled),
    m_vertical(vertical),
    m_fraction(fraction < 0.0 ? ic_default_fraction : fraction),
    m_warning_fraction(warning_fraction < 0.0 ? ic_default_warning_fraction: warning_fraction),
    m_critical_fraction(critical_fraction < 0.0 ? ic_default_critical_fraction: critical_fraction),
    m_empty_fraction(empty_fraction < 0.0 ? ic_default_empty_fraction: empty_fraction),
    m_show_warning(m_fraction <= m_warning_fraction),
    m_show_critical(m_fraction <= m_critical_fraction),
    m_show_empty(m_fraction <= m_empty_fraction),
    m_window_rect(),
    m_bar_background_rect(),
    m_bar_rect(),
    m_normal_color(),
    m_empty_color(),
    m_warning_color(),
    m_critical_color(),
    m_background_color(),
    m_disabled_color(),
    m_bar_thickness(bar_thickness < 1 ? ic_default_bar_thickness : bar_thickness),
    m_bar_length(bar_length < 1 ? ic_default_bar_length : bar_length),
    m_border_padding(border_padding < 0 ? ic_default_border_padding : border_padding),
    m_widget_sized(false)
{
  set_has_window(false);

  mf_setup_colors();

  postcondition(mf_invariant(false));
}

Fraction_bar_widget::~Fraction_bar_widget()
{
  precondition(mf_invariant(false));
}

//
// View update
//

bool Fraction_bar_widget::update(Error_param& return_error)
{
  precondition(!return_error()
               && mf_invariant());

  static_cast<void>(return_error); // avoid unused warning

  mf_update_bar();

  queue_draw(); // invalidate entire area of widget, triggering an expose event to redraw

  postcondition(!return_error()
                && mf_invariant());
  return true;
}

//
// Mutators
//

void Fraction_bar_widget::set_enabled(bool enabled)
{
  precondition(mf_invariant());

  m_enabled = enabled;

  if (!m_ref_window)
  {
    goto exit_point;
  }

  mf_calc_fixed_rects(m_window_rect);

  update(iv_ignore_error); // ignore error

 exit_point:
  postcondition(mf_invariant());
}

void Fraction_bar_widget::set_fraction(double fraction)
{
  precondition(mf_invariant());

  m_fraction = fraction;

  update(iv_ignore_error); // ignore error;

  postcondition(mf_invariant());
}

void Fraction_bar_widget::set_warning_fraction(double warning_fraction)
{
  precondition(mf_invariant());

  m_warning_fraction = warning_fraction;

  if (!m_ref_window)
  {
    goto exit_point;
  }

  mf_calc_fixed_rects(m_window_rect);

  update(iv_ignore_error); // ignore error

 exit_point:
  postcondition(mf_invariant());
}

void Fraction_bar_widget::set_critical_fraction(double critical_fraction)
{
  precondition(mf_invariant());

  m_critical_fraction = critical_fraction;

  if (!m_ref_window)
  {
    goto exit_point;
  }

  mf_calc_fixed_rects(m_window_rect);

  update(iv_ignore_error); // ignore error

 exit_point:
  postcondition(mf_invariant());
}

//
// Protected member functions
//

bool Fraction_bar_widget::mf_invariant(bool check_base_class) const
{
  bool return_value = false;

  {
    if (get_realized() && !m_ref_window)
    {
      goto exit_point;
    }

    // check attributes
    if ((m_bar_thickness < 1)
        || (m_bar_length < 1)
        || (m_border_padding < 0))
    {
      goto exit_point;
    }
    // range check
    if ((m_fraction < 0.0) || (m_fraction > 1.0)
        || (m_warning_fraction < 0.0) || (m_warning_fraction > 1.0)
        || (m_critical_fraction < 0.0) || (m_critical_fraction > 1.0)
        || (m_empty_fraction < 0.0) || (m_empty_fraction > 1.0))
    {
      goto exit_point;
    }

    if (!ih_is_valid_rect(m_window_rect)
        || !ih_is_valid_rect(m_bar_background_rect)
        || !ih_is_valid_rect(m_bar_rect))
    {
      goto exit_point;
    }
  }

  return_value = true;

 exit_point:
  return return_value;
}

//
// Signal handlers
//

void Fraction_bar_widget::on_size_request(Requisition* return_requisition)
{
  *return_requisition = Requisition();

  // set minimum space requested
  int height;
  int width;
  mf_calc_widget_size(width, height);
  return_requisition->width = width;
  return_requisition->height = height;
}

void Fraction_bar_widget::on_size_allocate(Allocation& allocation)
{
  // we will get height/width >= what was requested, use what is given
  set_allocation(allocation);

  m_window_rect = Rectangle(allocation.get_x(),
                            allocation.get_y(),
                            allocation.get_width(),
                            allocation.get_height());

  if (m_ref_window)
  {
    m_ref_window->move_resize(m_window_rect.get_x(),
                              m_window_rect.get_y(),
                              m_window_rect.get_width(),
                              m_window_rect.get_height());
  }

  mf_calc_fixed_rects(m_window_rect);
  mf_update_bar();

  m_widget_sized = true;
}

void Fraction_bar_widget::on_map()
{
  Widget::on_map();
}

void Fraction_bar_widget::on_unmap()
{
  Widget::on_unmap();
}

void Fraction_bar_widget::on_realize()
{
  Widget::on_realize();

  ensure_style();

  if (!m_ref_window)
  {
    GdkWindowAttr attributes;
    memset(&attributes, 0, sizeof(attributes));

    Allocation allocation = get_allocation();

    // set initial position and size of the Gdk::Window:
    attributes.x = allocation.get_x();
    attributes.y = allocation.get_y();
    attributes.width = allocation.get_width();
    attributes.height = allocation.get_height();

    attributes.event_mask = get_events () | Gdk::EXPOSURE_MASK;
    attributes.window_type = GDK_WINDOW_CHILD;
    attributes.wclass = GDK_INPUT_OUTPUT;

    RefPtr<Window> parent_window = get_window();
    m_ref_window = Window::create(parent_window,
                                  &attributes,
                                  GDK_WA_X | GDK_WA_Y);
    set_has_window();
    set_window(m_ref_window);

    m_ref_window->reference(); // Note: will create a leak, but can be ignored

    // setup the widget to receive expose events
    m_ref_window->set_user_data(gobj());
  }

}

void Fraction_bar_widget::on_unrealize()
{
  m_ref_window.clear();

  Widget::on_unrealize();
}

bool Fraction_bar_widget::on_expose_event(GdkEventExpose* event)
{
  {
    if (!m_ref_window)
    {
      goto exit_point;
    }

    Cairo::RefPtr<Cairo::Context> context = m_ref_window->create_cairo_context();

    if (event)
    {
      // clip to the area that needs to be re-exposed so we don't draw any
      // more than we need to.
      context->rectangle(event->area.x,
                         event->area.y,
                         event->area.width,
                         event->area.height);
      context->clip();
    }

    // paint the background
    set_source_color(context, get_parent()->get_style()->get_bg(Gtk::STATE_NORMAL));
    context->paint();

    // draw the bar from what was calculated when the bar was updated

    if (!m_enabled)
    {
      // draw disabled
      set_source_color(context, m_disabled_color);
      add_rectangle_to_path(context, m_bar_background_rect);
      context->fill();
    }
    else if (m_show_empty)
    {
      // draw empty
      set_source_color(context, m_empty_color);
      add_rectangle_to_path(context, m_bar_background_rect);
      context->fill();
    }
    else
    {
      // draw background
      set_source_color(context, m_background_color);
      add_rectangle_to_path(context, m_bar_background_rect);
      context->fill();

      // draw bar
      if (m_show_critical)
      {
        set_source_color(context, m_critical_color);
      }
      else if (m_show_warning)
      {
        set_source_color(context, m_warning_color);
      }
      else
      {
        set_source_color(context, m_normal_color);
      }
      add_rectangle_to_path(context, m_bar_rect);
      context->fill();
    }
  }

exit_point:
  return true;
}

//
// Private member functions
//

// calculate the tight bounds that the widget needs
void Fraction_bar_widget::mf_calc_widget_size(int& return_width, int& return_height)
{
  if (m_vertical)
  {
    return_height = m_bar_length + m_border_padding * 2;
    return_width = m_bar_thickness + m_border_padding * 2;
  }
  else
  {
    return_height = m_bar_thickness + m_border_padding * 2;
    return_width = m_bar_length + m_border_padding * 2;
  }
}

// calculate rectangles that are fixed between widget resize events
void Fraction_bar_widget::mf_calc_fixed_rects(const Rectangle& rect)
{
  int widget_width;
  int widget_height;
  mf_calc_widget_size(widget_width, widget_height);
  const int extra_x = (rect.get_width() > widget_width) ? (rect.get_width() - widget_width) : 0;
  const int extra_y = (rect.get_height() > widget_height) ? (rect.get_height() - widget_height) : 0;
  const int extra_x_padding = extra_x / 2;
  const int extra_y_padding = extra_y / 2;
  const int bar_length = m_bar_length;
  const bool vertical = m_vertical;

  int current_bar_origin_x = m_border_padding + (vertical ? extra_x_padding : 0);
  int current_bar_origin_y = m_border_padding + (vertical ? extra_y : extra_y_padding);

  // the bar rectangle is the bounds for the main bar
  Rectangle& bar_background_rect = m_bar_background_rect;
  bar_background_rect = Rectangle(current_bar_origin_x,
                                  current_bar_origin_y,
                                  (vertical ? m_bar_thickness : bar_length),
                                  (vertical ? bar_length : m_bar_thickness));
}

void Fraction_bar_widget::mf_setup_colors()
{
  // green for normal
  m_normal_color.set_rgb(0,
                         65535,
                         0);
  // yellow for warning
  m_warning_color.set_rgb(65535,
                          65535,
                          0);
  // red for critical
  m_critical_color.set_rgb(65535,
                           0,
                           0);
  // dim red for empty
  m_empty_color.set_rgb(65535 / 2,
                        0,
                        0);
  // light grey for normal background
  m_background_color.set_rgb(65535 / 4,
                             65535 / 4,
                             65535 / 4);
  // grey for disabled background
  m_disabled_color.set_rgb(65535 / 3 * 2,
                           65535 / 3 * 2,
                           65535 / 3 * 2);
}

// calculate rectangles and colors needed when bar is updated
void Fraction_bar_widget::mf_update_bar()
{
  const bool vertical = m_vertical;

  if (m_enabled)
  {
    const bool show_empty = m_fraction < m_empty_fraction;
    m_show_empty = show_empty;

    const bool show_critical = !show_empty && (m_fraction <= m_critical_fraction);
    m_show_critical = show_critical;

    const bool show_warning = !show_critical && (m_fraction <= m_warning_fraction);
    m_show_warning = show_warning;

    if (!show_empty)
    {
      // calculate rect for the bar
      const int bar_length = m_bar_length;
      Rectangle& bar_rect = m_bar_rect;
      bar_rect = m_bar_background_rect;

      const int current_bar_length = round(bar_length * m_fraction);
      if (vertical)
      {
        bar_rect.set_height(current_bar_length);
        bar_rect.set_y(m_bar_background_rect.get_y()
                       + (m_bar_background_rect.get_height() - current_bar_length));
      }
      else
      {
        bar_rect.set_width(current_bar_length);
      }
    }
  }
}
