// Kinetophone_error.cpp
//
// Copyright 2011-2013 Roan Trail, Inc.
//
// This file is part of Kinetophone.
//
// Kinetophone is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published
// by the Free Software Foundation, either version 2 of the License,
// or (at your option) any later version.
//
// Kinetophone is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// General Public License for more details.  You should have received
// a copy of the GNU General Public License along with Kinetophone. If
// not, see <http://www.gnu.org/licenses/>.

#include <kinetophone/error/Kinetophone_error.hpp>
#include <kinetophone/common.hpp>
#include <string>

using std::string;
using namespace Roan_trail::Kinetophone;

//
// Constructor/destructor/copy
//

Kinetophone_error::Kinetophone_error(const char* file,
                                     const char* function,
                                     int line,
                                     int code,
                                     const string& diagnostic,
                                     const Error* base_error)
  : Error(file,
          function,
          line,
          code,
          base_error)
{
  mf_set_error_code_string(code);

  if ("" != diagnostic)
  {
    error_dictionary()[diagnostic_error_key] = diagnostic;
  }

  if (mf_is_user_error())
  {
    mf_set_descriptions_from_file("Kinetophone_errors.xml");
  }

  postcondition(mf_invariant(false));
}

Kinetophone_error::Kinetophone_error(const char* file,
                                     const char* function,
                                     int line,
                                     int code,
                                     const string& diagnostic,
                                     const string& file_path,
                                     const Error* base_error)
  : Error(file,
          function,
          line,
          code,
          base_error)
{
  mf_set_error_code_string(code);

  if ("" != diagnostic)
  {
    error_dictionary()[diagnostic_error_key] = diagnostic;
  }

  if ("" != file_path)
  {
    error_dictionary()[file_path_error_key] = file_path;
  }

  if (mf_is_user_error())
  {
    mf_set_descriptions_from_file("Kinetophone_errors.xml");
  }

  postcondition(mf_invariant(false));
}

Kinetophone_error::Kinetophone_error(const char* file,
                                     const char* function,
                                     int line,
                                     int code,
                                     const Error* base_error)
  : Error(file,
          function,
          line,
          code,
          base_error)
{
  mf_set_error_code_string(code);

  if (mf_is_user_error())
  {
    mf_set_descriptions_from_file("Kinetophone_errors.xml");
  }

  postcondition(mf_invariant(false));
}

//
// Error codes
//

const int Kinetophone_error::none;
const int Kinetophone_error::general;
const int Kinetophone_error::startup;
const int Kinetophone_error::record;
const int Kinetophone_error::shutdown;
const int Kinetophone_error::command_line;
const int Kinetophone_error::list_devices;
const int Kinetophone_error::console;
const int Kinetophone_error::window;
const int Kinetophone_error::server;
const int Kinetophone_error::client;
const int Kinetophone_error::source_load;
const int Kinetophone_error::build;
const int Kinetophone_error::build_warning;
const int Kinetophone_error::file_IO;
const int Kinetophone_error::vox;
const int Kinetophone_error::vox_warning;

//
// Protected member functions
//

bool Kinetophone_error::mf_invariant(bool check_base_class) const
{
  bool return_value = false;

  if ((code() < Kinetophone_error::none) || (code() > Kinetophone_error::vox_warning))
  {
    goto exit_point;
  }
  else if (m_code_string_map.empty())
  {
    goto exit_point;
  }
  else
  {
    const Error_dictionary& dict = error_dictionary();
    Error_dictionary::const_iterator i = dict.find(Error::code_string_error_key);
    if (i == dict.end())
    {
      goto exit_point;
    }
    else
    {
      return_value = (!check_base_class || Error::mf_invariant(check_base_class));
    }
  }

 exit_point:

  return return_value;
}

string Kinetophone_error::mf_set_error_code_string(int code)
{
  if (m_code_string_map.empty())
  {
    // error code string map is a singleton for this class
    m_code_string_map[none]           = "kinetophone_error_none";
    m_code_string_map[general]        = "kinetophone_error_general";
    m_code_string_map[startup]        = "kinetophone_error_startup";
    m_code_string_map[record]         = "kinetophone_error_record";
    m_code_string_map[shutdown]       = "kinetophone_error_shutdown";
    m_code_string_map[list_devices]   = "kinetophone_error_list_devices";
    m_code_string_map[command_line]   = "kinetophone_error_command_line";
    m_code_string_map[console]        = "kinetophone_error_console";
    m_code_string_map[window]         = "kinetophone_error_window";
    m_code_string_map[server]         = "kinetophone_error_server";
    m_code_string_map[client]         = "kinetophone_error_client";
    m_code_string_map[source_load]    = "kinetophone_error_source_load";
    m_code_string_map[build]          = "kinetophone_error_build";
    m_code_string_map[build_warning]  = "kinetophone_error_build_warning";
    m_code_string_map[file_IO]        = "kinetophone_error_file_IO";
    m_code_string_map[vox]            = "kinetophone_error_vox";
    m_code_string_map[vox_warning]    = "kinetophone_error_vox_warning";
  }

  string code_string = m_code_string_map[code];
  if (!code_string.length())
  {
    code_string = m_code_string_map[general];
  }
  error_dictionary()[code_string_error_key] = code_string;

  return code_string;
}

//
// Static class data members
//

Error::Error_code_string_map Kinetophone_error::m_code_string_map;
