// common.hpp
//
// Copyright 2011-2013 Roan Trail, Inc.
//
// This file is part of Kinetophone.
//
// Kinetophone is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published
// by the Free Software Foundation, either version 2 of the License,
// or (at your option) any later version.
//
// Kinetophone is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// General Public License for more details.  You should have received
// a copy of the GNU General Public License along with Kinetophone. If
// not, see <http://www.gnu.org/licenses/>.

#ifndef ROAN_TRAIL_KINETOPHONE_COMMON_HPP_
#define ROAN_TRAIL_KINETOPHONE_COMMON_HPP_

#include <cassert>
#include <boost/cstdint.hpp>
#include <boost/concept/assert.hpp>
#include <boost/concept_check.hpp>
#include <boost/static_assert.hpp>

namespace Roan_trail
{
  namespace Kinetophone
  {
    // templatized max/min functions
    template <class T> const T& min(const T& a, const T& b)
    {
      BOOST_CONCEPT_ASSERT((boost::LessThanComparable<T>));
      return (b < a) ? b : a;
    }
    template <class T> const T& max(const T& a, const T& b)
    {
      BOOST_CONCEPT_ASSERT((boost::LessThanComparable<T>));
      return (b > a) ? b : a;
    }

    // to avoid warnings about equality comparisions of floating point values
    template <class T> bool equal_fp(const T& a, const T& b)
    {
      BOOST_CONCEPT_ASSERT((boost::LessThanComparable<T>));
      return !((a < b) || (a > b));
    }

#ifdef BOOST_NO_INT64_T
#error This platform is not supported, this software requires a 64 bit integer
#endif

    // typedef aliases for special numerical types
    typedef int64_t Long_int;
    typedef float Stream_float; // has to be same type as internal audio stream code
    typedef int32_t Stream_conv_int; // has to be compatible with __sync... calls

    // platform checks (assumptions about hardware)
    //   system has sufficiently large int for our assumptions
    BOOST_STATIC_ASSERT((sizeof(int) >= 4)
                        && "Error, int type needs be at least 4 bytes");
    //   checks on the audio stream format data types
    BOOST_STATIC_ASSERT((sizeof(Stream_float) == 4)
                        && "Error, audio stream float format is not 4 bytes (currently supported size)");
    BOOST_STATIC_ASSERT((sizeof(Stream_float) == sizeof(Stream_conv_int))
                        && "Error, size of audio stream float format not equal to integer conversion size");
    //   system has sufficiently large "long" int for our assumptions
    BOOST_STATIC_ASSERT((sizeof(Long_int) >= 8)
                        && "Error, long integer type needs be at least 8 bytes");

    template <class T> T round_up_to_pow2(const T& n)
    {
      BOOST_CONCEPT_ASSERT((boost::Integer<T>));
      T value = n;
      T converted_value = 1;
      bool remainder;
      while (true)
      {
        remainder = ((value % 2) != 0);
        if (value /= 2)
        {
          converted_value *= 2;
        }
        else
        {
          break;
        }
      }
      if (remainder)
      {
        converted_value *= 2;
      }

      return converted_value;
    }

    template <class T> bool is_pow2(const T& n)
    {
      BOOST_CONCEPT_ASSERT((boost::Integer<T>));
      return !((n - 1) & n);
    }

    // pre/post conditions
#define precondition(cond) assert("precondition failed" && (cond))
#define postcondition(cond) assert("postcondition failed" && (cond))
#ifdef NDEBUG
#define declare_old(decl) static_cast<void>(0);
#else
#define declare_old(decl) decl
#endif

  }
}

#endif // ROAN_TRAIL_KINETOPHONE_COMMON_HPP_
