// Video_track.cpp
//
// Copyright 2011-2014 Roan Trail, Inc.
//
// This file is part of Kinetophone.
//
// Kinetophone is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published
// by the Free Software Foundation, either version 2 of the License,
// or (at your option) any later version.
//
// Kinetophone is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// General Public License for more details.  You should have received
// a copy of the GNU General Public License along with Kinetophone. If
// not, see <http://www.gnu.org/licenses/>.

#include <kinetophone/Video_track.hpp>
#include <kinetophone/AV_compat.hpp>
#include <kinetophone/Movie_config.hpp>
#include <kinetophone/error/Build_error.hpp>
#include <limits>
#include <cstddef>

using std::numeric_limits;
using namespace Roan_trail::Kinetophone;

//
// Internal helpers
//

namespace
{
  const double ic_size_estimate_adjustment = 1.01;
  const int ic_max_int = numeric_limits<int>::max();
}

Video_track::Video_track(int track_ID,
                         const Frame_rate& frame_rate,
                         const Rect_size& output_frame_size,
                         Movie_preset_type movie_preset,
                         int keyframe_period)
  : m_setup(false),
    m_started(false),
    m_track_ID(track_ID),
    m_frame_rate(frame_rate),
    m_output_frame_size(output_frame_size),
    m_movie_preset(movie_preset),
    m_current_track_time(0),
    m_format_context(0),
    m_stream(0),
    m_codec_context(0),
    m_output_frame(0),
    m_output_buffer(0),
    m_output_buffer_size(0),
    m_keyframe_period(keyframe_period)
{
  postcondition(mf_invariant(false));
}

Video_track::~Video_track()
{
  precondition(mf_invariant(false));

  mf_cleanup();
}

bool Video_track::setup(AVFormatContext* format_context, Error_param& return_error)
{
  precondition(format_context
               && !return_error()
               && mf_invariant());

  // used in error cleanup
  AVStream* stream = 0;
  AVCodecContext* codec_context = 0;
  AVCodec* codec = 0;
  int codec_opened = -1;
  AVFrame* output_frame = 0;
  uint8_t* output_buffer = 0;
  Long_int output_buffer_size = 0;

  bool return_value = false;

  start_error_block();

  if (m_setup)
  {
    return_value = true;
    goto exit_point;
  }

  // initialization for FFMPEG
  stream = AV_compat::new_stream(format_context, m_track_ID);
  on_error(!stream, new Build_error(error_location(),
                                    Build_error::setup,
                                    "",
                                    "",
                                    "could not allocate stream for video track"));

  AVRational time_base = { m_frame_rate.frame_length(), m_frame_rate.time_scale() };

  codec_context = stream->codec;
  codec_context->width = m_output_frame_size.width;
  codec_context->height = m_output_frame_size.height;
  codec_context->time_base = time_base;
  codec_context->codec_type = AV_compat::codec_type_video;

  mf_setup_codec_context_from_preset(m_movie_preset, *codec_context);
  // flag need for separate stream headers, if necessary
  if (format_context->oformat->flags & AVFMT_GLOBALHEADER)
  {
    codec_context->flags |= CODEC_FLAG_GLOBAL_HEADER;
  }

  // open codec
  codec = avcodec_find_encoder(codec_context->codec_id);
  on_error(!codec, new Build_error(error_location(),
                                   Build_error::setup,
                                   "",
                                   "",
                                   "could not find codec for video track"));

  codec_opened = AV_compat::codec_open(codec_context, codec);
  on_error(codec_opened, new Build_error(error_location(),
                                         Build_error::setup,
                                         "",
                                         "",
                                         "could not open codec for video track"));
  // setup output
  //   frame
  output_frame = avcodec_alloc_frame();
  on_error(!output_frame, new Build_error(error_location(),
                                          Build_error::setup,
                                          "",
                                          "",
                                          "out of memory allocating output frame for video track"));
  //   frame buffer
  if (AV_compat::need_encoder_output_buffer())
  {
    const Long_int frame_buffer_size = avpicture_get_size(codec_context->pix_fmt,
                                                          codec_context->width,
                                                          codec_context->height);
    //   output buffer
    output_buffer_size = max(2 * frame_buffer_size, 4 * static_cast<Long_int>(FF_MIN_BUFFER_SIZE));
    output_buffer = static_cast<uint8_t*>(av_malloc(output_buffer_size));
    on_error(!output_buffer, new Build_error(error_location(),
                                             Build_error::setup,
                                             "",
                                             "",
                                             "out of memory allocating output buffer for video track"));
  }

  // success, update data members
  mf_cleanup();
  //
  m_format_context = format_context;
  m_stream = stream;
  m_codec_context = codec_context;
  m_output_frame = output_frame;
  m_output_buffer = output_buffer;
  m_output_buffer_size = output_buffer_size;
    //
  m_setup = true;

  return_value = true;
  goto exit_point;

  end_error_block();

  default_error_handler(return_error);

 error_cleanup:
  if (!return_error.need_error())
  {
    delete handler_error;
  }
  if (!codec_opened)
  {
    avcodec_close(codec_context);
  }
  if (output_buffer)
  {
    av_free(output_buffer);
  }
  if (output_frame)
  {
    av_free(output_frame);
  }
  if (codec_context)
  {
    av_free(codec_context);
  }
  if (stream)
  {
    av_free(stream);
  }
  return_value = false;
  goto exit_point;

 exit_point:
  postcondition(// success
                ((return_value
                  && m_codec_context
                  && m_output_frame
                  && m_setup)
                // failure
                 || !return_value)
                && return_error.is_valid_at_return(return_value)
                && mf_invariant());
  return return_value;
}


//
// Build member functions
//

bool Video_track::start(Error_param& return_error)
{
  precondition(m_setup
               && !m_started
               && !return_error()
               && mf_invariant());

  // success, update data members
  m_current_track_time = 0;
  //
  m_started = true;

  postcondition(m_setup
                && m_started
                && (0 == m_current_track_time)
                && !return_error()
                && mf_invariant());
  return true;
}

bool Video_track::add(const uint8_t* raw_frame,
                      Long_int frame_count,
                      Error_param& return_error)
{
  precondition(raw_frame
               && (frame_count >= 0)
               && !return_error()
               && m_setup
               && m_started
               && mf_invariant());

  bool return_value = false;

  start_error_block();

  if (frame_count < 1)
  {
    // nothing to do
    return_value = true;
    goto exit_point;
  }

  // force RLE encoder to output I-Frame/key frame for the first frame
  m_codec_context->gop_size = 0;

  for (Long_int i = 0; i < frame_count; ++i)
  {
    AVPacket packet;
    bool got_packet;
    const bool encode_return = mf_encode(raw_frame,
                                         packet,
                                         got_packet);
    on_error(!encode_return, new Build_error(error_location(),
                                             Build_error::temporary_movie_write,
                                             "",
                                             "",
                                             "error encoding video frames"));
    assert(got_packet && "error, frame unexpectedly buffered");

    packet.pts = m_current_track_time;
    packet.duration = m_frame_rate.frame_length();
    packet.stream_index = m_stream->index;

    // write the compressed frame
    const int write_return = av_interleaved_write_frame(m_format_context, &packet);
    on_error(write_return, new Build_error(error_location(),
                                           Build_error::temporary_movie_write,
                                           "",
                                           "",
                                           "error writing video frames"));

    m_current_track_time += packet.duration;
    // check for subsequent frames being I-Frame/keyframes or P-Frame/non-key frames
    if (m_keyframe_period)
    {
      // check that RLE encoder produced the correct frame type
      //   current frame is in the key frame period
      assert(((i % m_keyframe_period) || m_codec_context->coded_frame->key_frame)
             && "error, frame should be a key frame in key frame period");
      assert(((i % m_keyframe_period)
              || (AV_compat::picture_type_I == m_codec_context->coded_frame->pict_type))
             && "error, frame should be an I-frame in key frame period");
      //   current frame is not in the key frame period
      assert((!(i % m_keyframe_period) || !m_codec_context->coded_frame->key_frame)
             && "error, frame should be not be a key frame outside the key frame period");
      assert((!(i % m_keyframe_period)
              || (AV_compat::picture_type_P == m_codec_context->coded_frame->pict_type))
             && "error, frame should be a P-frame outside the key frame period");
      //   for the next frame, force I-Frame/keyframe for the frames is key frame period, otherwise
      //   force P-Frames
      m_codec_context->gop_size = ((i + 1) % m_keyframe_period) ? ic_max_int : 0;
    }
    else
    {
      // check that RLE encoder produced the correct frame type
      //   first frame should be an I-Frame/key frame
      assert(((0 != i) || m_codec_context->coded_frame->key_frame)
             && "error, first frame should be a key frame");
      assert(((0 != i) || (AV_compat::picture_type_I == m_codec_context->coded_frame->pict_type))
             && "error, first frame should be a I-frame");
      assert(((0 == i) || !m_codec_context->coded_frame->key_frame)
             && "error, remaining frames should not be key frames");
      assert(((0 == i) || (AV_compat::picture_type_P == m_codec_context->coded_frame->pict_type))
             && "error, remaining frames should be P-frames");
      //   from now on, frames added in this call should be an P-Frame/non-key frames
      m_codec_context->gop_size = ic_max_int;
    }

    // if we have allocated m_output_buffer, free the packet, but don't free the data buffer
    if (m_output_buffer)
    {
      packet.data = 0;
      packet.size = 0;
    }
    av_free_packet(&packet);
  }

  return_value = true;
  goto exit_point;

  end_error_block();

  default_error_handler_and_cleanup(return_error,
                                    return_value,
                                    false);
  goto exit_point;

 exit_point:
  postcondition(m_setup
                && m_started
                && return_error.is_valid_at_return(return_value)
                && mf_invariant());
  return return_value;
}


bool Video_track::end(Error_param& return_error)
{
  precondition(m_setup
               && m_started
               && !return_error()
               && mf_invariant());

  m_started = false;

  postcondition(m_setup
                && !m_started
                && !return_error()
                && mf_invariant());
  return true;
}

Long_int Video_track::estimate_frame_size(const uint8_t* raw_frame)
{
  precondition(m_setup
               && mf_invariant());

  Long_int return_value = 0;
  Long_int size_estimate;
  if (raw_frame && mf_compressed_size_of_pixels(raw_frame, size_estimate))
  {
    return_value = size_estimate;
  }
  else
  {
    return_value = avpicture_get_size(m_codec_context->pix_fmt,
                                      m_codec_context->width,
                                      m_codec_context->height);
  }

  return_value *= ic_size_estimate_adjustment;

  precondition((return_value >= 0)
               && m_setup
               && mf_invariant());
  return return_value;
}

Long_int Video_track::estimate_segment_size(Long_int frame_size, Long_int frame_count)
{
  precondition(m_setup
               && mf_invariant());

  Long_int return_value = frame_size * (m_keyframe_period ?
                                        (frame_count % m_keyframe_period)
                                        : 1);
  return_value *= ic_size_estimate_adjustment;

  precondition((return_value >= 0)
               && m_setup
               && mf_invariant());
  return return_value;
}

//
// Protected member functions
//

bool Video_track::mf_invariant(bool check_base_class) const
{
  bool return_value = false;

  if ((m_setup
       && (!m_format_context
           || !m_stream
           || !m_codec_context
           || !m_output_frame))
      || (!m_setup
          && (m_format_context
              || m_stream
              || m_output_frame
              || m_output_buffer
              || m_output_buffer_size)))
  {
    goto exit_point;
  }

  if ((m_output_buffer && !m_output_buffer_size)
      || (!m_output_buffer && m_output_buffer_size))
  {
    goto exit_point;
  }

  if (m_started && !m_setup)
  {
    goto exit_point;
  }

  if (m_current_track_time < 0)
  {
    goto exit_point;
  }

  return_value = true;
  goto exit_point;

 exit_point:
  return return_value;
}

//
// Private member functions
//

bool Video_track::mf_compressed_size_of_pixels(const uint8_t* raw_frame,
                                               Long_int& return_compressed_size)
{
  precondition(m_setup);

  AVPacket packet;
  bool got_packet;
  const bool encode_return = mf_encode(raw_frame,
                                       packet,
                                       got_packet);

  return_compressed_size = static_cast<Long_int>(packet.size);

  if (encode_return && got_packet)
  {
    // if we have allocated m_output_buffer, free the packet, but don't free the data buffer
    if (m_output_buffer)
    {
      packet.data = 0;
      packet.size = 0;
    }
    av_free_packet(&packet);
  }

  return true;
}

// if function returns true and return_got_packet is returned as true, caller needs free packet data
bool Video_track::mf_encode(const uint8_t* raw_frame,
                            AVPacket& packet,
                            bool& return_got_packet)
{
  precondition(m_setup);

  return_got_packet = false;

  if (AV_compat::pixel_format_RGB24 == m_codec_context->pix_fmt)
  {
    mf_fill_in_output_frame(raw_frame);
  }
  else
  {
    assert(false && "error, only RGB colorspace supported");
  }

  av_init_packet(&packet);
  packet.data = m_output_buffer;
  packet.size = m_output_buffer_size;

  // encode the image
  const bool encode_return = AV_compat::encode_video(m_codec_context,
                                                     &packet,
                                                     m_output_frame,
                                                     return_got_packet);

  return encode_return;
}

void Video_track::mf_setup_codec_context_from_preset(Movie_preset_type preset,
                                                     AVCodecContext& return_codec_context)
{
  if (Movie_config::movie_preset_qtrle == preset)
  {
    return_codec_context.codec_id = AV_compat::codec_ID_QTRLE;
    return_codec_context.gop_size = ic_max_int;
    return_codec_context.pix_fmt = AV_compat::pixel_format_RGB24;
  }
  else
  {
    assert(false && "invalid movie preset");
  }
}

void Video_track::mf_cleanup()
{
  if (m_started)
  {
    m_started = false;
  }
  m_output_buffer_size = 0;
  if (m_output_buffer)
  {
    av_free(m_output_buffer);
    m_output_buffer = 0;
  }
  if (m_output_frame)
  {
    av_free(m_output_frame);
    m_output_frame = 0;
  }
  if (m_codec_context)
  {
    avcodec_close(m_codec_context);
    av_free(m_codec_context);
    m_codec_context = 0;
  }
  if (m_stream)
  {
    av_free(m_stream);
    m_stream = 0;
  }
  m_format_context = 0;
}

void Video_track::mf_fill_in_output_frame(const uint8_t* raw_frame)
{
  AVPicture output_picture;
  avpicture_fill(&output_picture,
                 const_cast<uint8_t*>(raw_frame),
                 m_codec_context->pix_fmt,
                 m_codec_context->width,
                 m_codec_context->height);
  for (size_t i = 0; i < 4; ++i)
  {
    m_output_frame->data[i] = output_picture.data[i];
    m_output_frame->linesize[i] = output_picture.linesize[i];
  }
}
