// Speech_synthesizer.hpp
//
// Copyright 2011-2013 Roan Trail, Inc.
//
// This file is part of Kinetophone.
//
// Kinetophone is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published
// by the Free Software Foundation, either version 2 of the License,
// or (at your option) any later version.
//
// Kinetophone is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// General Public License for more details.  You should have received
// a copy of the GNU General Public License along with Kinetophone. If
// not, see <http://www.gnu.org/licenses/>.

#ifndef ROAN_TRAIL_KINETOPHONE_SPEECH_SYNTHESIZER_HPP_
#define ROAN_TRAIL_KINETOPHONE_SPEECH_SYNTHESIZER_HPP_

#include <kinetophone/common.hpp>
#include <string>

typedef struct SNDFILE_tag SNDFILE;

namespace Roan_trail
{
  namespace Kinetophone
  {
    class Async_audio_file_writer;
    class Error_param;
    class Logger;
    struct Speech_synthesizer_config;

    class Speech_synthesizer
    {
    public:
      // constructor/destructor
      Speech_synthesizer();
      virtual ~Speech_synthesizer();
      // control
      //
      //   startup/shutdown functions
      bool startup(const Speech_synthesizer_config &config, Error_param& return_error);
      bool shutdown(bool completed, Error_param& return_error);
      bool speak(const std::string& text, Error_param& return_error);
      // accessors
      bool is_started() const;
      bool can_record() const;
      Long_int frames_written_count() const
      { return __sync_fetch_and_add(const_cast<int64_t*>(&m_frames_written_count), 0x0); }
      void frames_to_recorded_time(const Long_int frames,
                                   char separator,
                                   std::string& return_recorded_time) const;
      void recorded_time(std::string &return_recorded_time, char separator = ':') const;
      const Speech_synthesizer_config* config() const { return m_config; }
      bool have_recording() const { return m_have_recording; }
      std::string output_file() const;
      bool space_available(Long_int& return_bytes_available,
                           double& return_fraction_available,
                           double& return_seconds_remaining) const;
      bool output_file_size(Long_int& return_file_size, double& return_data_rate) const;
    protected:
      // invariant check
      bool mf_invariant(bool check_base_class = true) const;
    private:
      // aligned data members used for synchronized access
      // TODO: can this be mutable for const accessor?
      int64_t __attribute__ ((aligned (8))) m_frames_written_count;
      //
      bool m_speech_initialized;
      SNDFILE* m_output_file;
      std::string m_temporary_file_path;
      Speech_synthesizer_config* m_config;
      Logger& m_logger;
      bool m_have_recording;
      Long_int m_frames_per_second;
      Long_int m_frames_per_minute;
      Long_int m_frames_per_hour;
      //
      SNDFILE* mf_open_sound_file(std::string& return_output_file_path,
                                  bool& return_created_temp_file,
                                  Error_param& return_error);
      // prevent compiler from generating
      Speech_synthesizer(const Speech_synthesizer& recorder);
      Speech_synthesizer& operator=(const Speech_synthesizer& recorder);
    };
  }
}

#endif // ROAN_TRAIL_KINETOPHONE_SPEECH_SYNTHESIZER_HPP_
