// Sound_track.hpp
//
// Copyright 2011-2013 Roan Trail, Inc.
//
// This file is part of Kinetophone.
//
// Kinetophone is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published
// by the Free Software Foundation, either version 2 of the License,
// or (at your option) any later version.
//
// Kinetophone is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// General Public License for more details.  You should have received
// a copy of the GNU General Public License along with Kinetophone. If
// not, see <http://www.gnu.org/licenses/>.

#ifndef ROAN_TRAIL_KINETOPHONE_SOUND_TRACK_HPP_
#define ROAN_TRAIL_KINETOPHONE_SOUND_TRACK_HPP_

#include <kinetophone/common.hpp>
#include <kinetophone/Movie_types.hpp>

struct AVFormatContext;
struct AVStream;
struct AVCodecContext;

namespace Roan_trail
{
  namespace Kinetophone
  {
    class Error_param;
    struct Sound_file_config;
    class Sound_source;

    class Sound_track
    {
    public:
      // constructor/destructor/initialization
      Sound_track(int track_ID,
                  const Frame_rate& reference_frame_rate,
                  Sound_source& sound);
      virtual ~Sound_track();
      // accessors
      bool endian_conversion() const { return m_need_endian_conversion; }
      bool setup(AVFormatContext* format_context, Error_param& return_error);
      // control
      virtual bool start(Error_param& return_error);
      virtual bool add(Long_int reference_start_frame,
                       Long_int reference_frame_count,
                       Error_param& return_error);
      virtual bool add_silence(Long_int reference_frame_count,
                               Error_param& return_error);
      virtual bool end(Error_param& return_error);
      // helpers
      Long_int estimate_segment_size(Long_int reference_frame_count);
    protected:
      // invariant check
      bool mf_invariant(bool check_base_class = true) const;
    private:
      Sound_source& m_sound_source;
      const Sound_file_config& m_config;
      Frame_rate m_reference_frame_rate;
      Long_int m_audio_frames_per_reference_frame;
      //
      bool m_setup;
      bool m_started;
      int m_track_ID;
      Long_int m_current_track_time;
      bool m_need_endian_conversion;
      // FFmpeg
      //   format reference
      AVFormatContext* m_format_context;
      //   stream
      AVStream* m_stream;
      //   codec
      AVCodecContext* m_codec_context;
      //
      uint8_t* m_output_conversion_buffer;
      //
      void mf_cleanup();
      bool mf_write_frames(const uint8_t* input_buffer,
                           Long_int write_frame_count,
                           Error_param& return_error);
      void mf_convert_endianness(const uint8_t* input_buffer, Long_int frame_count);
      // prevent compiler from generating
      Sound_track(const Sound_track& m);
      Sound_track& operator=(const Sound_track& m);
    };
  }
}

#endif // ROAN_TRAIL_KINETOPHONE_SOUND_TRACK_HPP_
