// Sound_track.cpp
//
// Copyright 2011-2013 Roan Trail, Inc.
//
// This file is part of Kinetophone.
//
// Kinetophone is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published
// by the Free Software Foundation, either version 2 of the License,
// or (at your option) any later version.
//
// Kinetophone is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// General Public License for more details.  You should have received
// a copy of the GNU General Public License along with Kinetophone. If
// not, see <http://www.gnu.org/licenses/>.

#include <kinetophone/Sound_track.hpp>
#include <kinetophone/AV_compat.hpp>
#include <kinetophone/Sound_source.hpp>
#include <kinetophone/Sound_file_config.hpp>
#include <kinetophone/File_manager.hpp>
#include <kinetophone/error/Build_error.hpp>

using namespace Roan_trail::Kinetophone;

//
// Internal constants and helpers
//

namespace
{
  const int ic_output_conversion_buffer_size = 32768;

  // endian swapping
  inline void ih_endian_swap16(uint16_t& x)
  {
    x = (x >> 8)
      | (x << 8);
  }

  inline void ih_endian_swap32(uint32_t& x)
  {
    x = (x >> 24)
      | ((x << 8) & 0x00FF0000)
      | ((x >> 8) & 0x0000FF00)
      | (x << 24);
  }

  inline void ih_endian_swap64(uint64_t& x)
  {
    x = (x >> 56)
      | ((x << 40) & 0x00FF000000000000ULL)
      | ((x << 24) & 0x0000FF0000000000ULL)
      | ((x << 8)  & 0x000000FF00000000ULL)
      | ((x >> 8)  & 0x00000000FF000000ULL)
      | ((x >> 24) & 0x0000000000FF0000ULL)
      | ((x >> 40) & 0x000000000000FF00ULL)
      | (x << 56);
  }
}

Sound_track::Sound_track(int track_ID,
                         const Frame_rate& reference_frame_rate,
                         Sound_source& sound)
  : m_sound_source(sound),
    m_config(sound.config()),
    m_reference_frame_rate(reference_frame_rate),
    m_audio_frames_per_reference_frame(m_config.sample_rate * m_reference_frame_rate.frame_length()
                                       / m_reference_frame_rate.time_scale()),
    m_setup(false),
    m_started(false),
    m_track_ID(track_ID),
    m_current_track_time(0),
    m_need_endian_conversion(false),
    m_format_context(0),
    m_stream(0),
    m_codec_context(0),
    m_output_conversion_buffer(0)
{
  postcondition(mf_invariant(false));
}

Sound_track::~Sound_track()
{
  precondition(mf_invariant(false));

  mf_cleanup();
}

bool Sound_track::setup(AVFormatContext* format_context, Error_param& return_error)
{
  precondition(format_context
               && !return_error()
               && mf_invariant());

  bool return_value = false;

  // used in error cleanup
  AVStream* stream = 0;
  AVCodecContext* codec_context = 0;
  AVCodec* codec = 0;
  int codec_opened = -1;
  uint8_t* output_conversion_buffer = 0;

  start_error_block();

  if (m_setup)
  {
    return_value = true;
    goto exit_point;
  }

  // initialization for FFmpeg
  stream = AV_compat::new_stream(format_context, m_track_ID);
  on_error(!stream, new Build_error(error_location(),
                                    Build_error::setup,
                                    "",
                                    "",
                                    "count not allocate stream for sound track"));
  //   setup the codec context
  codec_context = stream->codec;
  codec_context->codec_type = AV_compat::codec_type_audio;
  codec_context->sample_rate = m_config.sample_rate;
  codec_context->channels = m_config.channels;
  if (!AV_compat::codec_ID_for_sound_file_config(m_config, codec_context->codec_id))
  {
    assert(false && "unsupported sound file config when setting codec ID");
  }

  codec_context->channel_layout = AV_compat::default_channel_layout(codec_context->channels);

  // Note: we currently don't need the sample format because
  //   no audio encoding is done through FFmpeg, but this is here for completeness
  if (!AV_compat::sample_fmt_for_sound_file_config(m_config, codec_context->sample_fmt))
  {
    assert(false && "unsupported data format when setting sample format");
  }
  //     flag need for separate stream headers, if necessary
  if (format_context->oformat->flags & AVFMT_GLOBALHEADER)
  {
    codec_context->flags |= CODEC_FLAG_GLOBAL_HEADER;
  }

  //   open codec
  codec = avcodec_find_encoder(codec_context->codec_id);
  on_error(!codec, new Build_error(error_location(),
                                   Build_error::setup,
                                   "",
                                   "",
                                   "could not find audio codec"));

  codec_opened = AV_compat::codec_open(codec_context, codec);
  on_error(codec_opened, new Build_error(error_location(),
                                         Build_error::setup,
                                         "",
                                         "",
                                         "could not open audio codec"));

  const int sample_size = m_sound_source.frame_size() / m_config.channels;
  Sound_file_config::Endianness output_endian;
  if (!AV_compat::endianness_for_output_format(format_context->oformat,
                                               m_config,
                                               output_endian))
  {
    assert(false && "unsupported sound file configuration when getting output endianness");
  }
  const bool need_endian_conversion = ((output_endian != m_config.endianness) && (sample_size > 1));
  if (need_endian_conversion)
  {
    output_conversion_buffer = static_cast<uint8_t*>(av_malloc(ic_output_conversion_buffer_size));
  }

  // success, update data members
  mf_cleanup();
  //
  m_format_context = format_context;
  m_stream = stream;
  m_codec_context = codec_context;
  //
  m_need_endian_conversion = need_endian_conversion;
  m_output_conversion_buffer = output_conversion_buffer;

  m_setup = true;

  return_value = true;
  goto exit_point;

  end_error_block();

  default_error_handler(return_error);

 error_cleanup:
  if (!return_error.need_error())
  {
    delete handler_error;
  }
  if (output_conversion_buffer)
  {
    av_free(output_conversion_buffer);
  }
  if (!codec_opened)
  {
    avcodec_close(codec_context);
  }
  if (codec_context)
  {
    av_free(codec_context);
  }
  if (stream)
  {
    av_free(stream);
  }
  return_value = false;
  goto exit_point;

 exit_point:
  postcondition(( // success
                 (return_value
                  && m_setup
                  && m_codec_context)
                 // failure
                 || !return_value)
                && return_error.is_valid_at_return(return_value)
                && mf_invariant());
  return return_value;
}

//
// Build member functions
//

bool Sound_track::start(Error_param& return_error)
{
  precondition(!return_error()
               && m_setup
               && !m_started
               && mf_invariant());

  // success, update data members
  m_current_track_time = 0;
  //
  m_started = true;

  postcondition(m_setup
                && m_started
                && (0 == m_current_track_time)
                && !return_error()
                && mf_invariant());
  return true;
}

bool Sound_track::add(Long_int reference_start_frame,
                      Long_int reference_frame_count,
                      Error_param& return_error)
{
  precondition((reference_start_frame >= 0)
               && (reference_frame_count >= 0)
               && !return_error()
               && m_setup
               && m_started
               && mf_invariant());

  bool return_value = false;

  start_error_block();

  Error_param error;

  Long_int remaining_audio_frame_count = reference_frame_count * m_audio_frames_per_reference_frame;
  Long_int audio_start_frame = reference_start_frame * m_audio_frames_per_reference_frame;
  bool need_silence = false;

  while (remaining_audio_frame_count > 0)
  {
    Long_int current_audio_frame_count;
    const uint8_t* output_buffer;
    if (need_silence)
    {
      // fill in with silence if we are past the end of the audio data
      output_buffer = m_sound_source.silence(remaining_audio_frame_count,
                                             current_audio_frame_count,
                                             error);
      on_error(!output_buffer, new Build_error(error_location(),
                                               Build_error::general,
                                               error()));
    }
    else
    {
      // get data from the sound source
      output_buffer = m_sound_source.sound(audio_start_frame,
                                           remaining_audio_frame_count,
                                           current_audio_frame_count,
                                           error);
      on_error(!output_buffer, new Build_error(error_location(),
                                               Build_error::general,
                                               error()));
    }

    if (current_audio_frame_count > 0)
    {
      const bool frames_written = mf_write_frames(output_buffer,
                                                  current_audio_frame_count,
                                                  error);
      on_error(!frames_written, new Build_error(error_location(),
                                                Build_error::general,
                                                error()));
      remaining_audio_frame_count -= current_audio_frame_count;
      audio_start_frame += current_audio_frame_count;
    }
    else
    {
      need_silence = true;
    }
  }

  return_value = true;
  goto exit_point;

  end_error_block();

  default_error_handler_and_cleanup(return_error,
                                    return_value,
                                    false);
  goto exit_point;

 exit_point:
  postcondition(m_setup
                && m_started
                && return_error.is_valid_at_return(return_value)
                && mf_invariant());
  return return_value;
}

bool Sound_track::add_silence(Long_int reference_frame_count, Error_param& return_error)
{
  precondition((reference_frame_count >= 0)
               && !return_error()
               && m_setup
               && m_started
               && mf_invariant());

  bool return_value = false;

  start_error_block();

  Error_param error;

  Long_int remaining_audio_frame_count = reference_frame_count * m_audio_frames_per_reference_frame;

  while (remaining_audio_frame_count > 0)
  {
    Long_int current_audio_frame_count;
    const uint8_t* output_buffer = m_sound_source.silence(remaining_audio_frame_count,
                                                          current_audio_frame_count,
                                                          error);
    on_error(!output_buffer, new Build_error(error_location(),
                                             Build_error::general,
                                             error()));
    assert((current_audio_frame_count > 0) && "error, sound source silence returned 0 frames");

    const bool frames_written = mf_write_frames(output_buffer,
                                                current_audio_frame_count,
                                                error);
    on_error(!frames_written, new Build_error(error_location(),
                                              Build_error::general,
                                              error()));

    remaining_audio_frame_count -= current_audio_frame_count;
  }

  return_value = true;
  goto exit_point;

  end_error_block();

  default_error_handler_and_cleanup(return_error,
                                    return_value,
                                    false);
  goto exit_point;

 exit_point:
  postcondition(m_setup
                && m_started
                && return_error.is_valid_at_return(return_value)
                && mf_invariant());
  return return_value;
}

bool Sound_track::end(Error_param& return_error)
{
  precondition(!return_error()
               && m_setup
               && m_started
               && mf_invariant());

  m_started = false;

  postcondition(m_setup
                && !m_started
                && !return_error()
                && mf_invariant());
  return true;
}

Long_int Sound_track::estimate_segment_size(Long_int reference_frame_count)
{
  precondition(m_setup
               && (reference_frame_count >= 0)
               && mf_invariant());

  const Long_int audio_frame_count = reference_frame_count * m_audio_frames_per_reference_frame;

  Long_int total_size = audio_frame_count * m_sound_source.frame_size();

  postcondition((total_size >= 0)
                && m_setup
                && mf_invariant());
  return total_size;
}

//
// Protected member functions
//

bool Sound_track::mf_invariant(bool check_base_class) const
{
  bool return_value = false;

  if ((m_setup
       && (!m_format_context
           || !m_stream
           || !m_codec_context))
      || (!m_setup
          && (m_format_context
              || m_stream
              || m_codec_context)))
  {
    goto exit_point;
  }

  if (m_current_track_time < 0)
  {
    goto exit_point;
  }

  if (m_setup && (m_audio_frames_per_reference_frame < 0))
  {
    goto exit_point;
  }

  return_value = true;
  goto exit_point;

 exit_point:
  return return_value;
}

//
// Private member functions
//

void Sound_track::mf_cleanup()
{
  if (m_started)
  {
    m_started = false;
  }
  if (m_output_conversion_buffer)
  {
    av_free(m_output_conversion_buffer);
    m_output_conversion_buffer = 0;
  }
  if (m_codec_context)
  {
    avcodec_close(m_codec_context);
    av_free(m_codec_context);
    m_codec_context = 0;
  }
  if (m_stream)
  {
    av_free(m_stream);
    m_stream = 0;
  }
  m_format_context = 0;
}

bool Sound_track::mf_write_frames(const uint8_t* output_buffer,
                                  Long_int write_frame_count,
                                  Error_param& return_error)
{
  precondition(m_setup
               && m_started
               && output_buffer
               && (write_frame_count >= 0)
               && !return_error());

  bool return_value = false;

  start_error_block();

  uint8_t* write_buffer;
  Long_int frame_block_size;
  const int frame_size = m_sound_source.frame_size();
  if (m_need_endian_conversion)
  {
    write_buffer = m_output_conversion_buffer;
    frame_block_size = min(static_cast<Long_int>(ic_output_conversion_buffer_size / frame_size),
                           write_frame_count);
  }
  else
  {
    write_buffer = const_cast<uint8_t*>(output_buffer);
    frame_block_size = write_frame_count;
  }
  uint8_t* output_buffer_pos = const_cast<uint8_t*>(output_buffer);

  Long_int remaining_frame_count = write_frame_count;
  Long_int current_frame_count = 0;
  while (remaining_frame_count > 0)
  {
    current_frame_count = min(frame_block_size, remaining_frame_count);
    if (m_need_endian_conversion)
    {
      mf_convert_endianness(output_buffer_pos, current_frame_count);
      output_buffer_pos += current_frame_count * frame_size;
    }
    AVPacket packet;
    av_init_packet(&packet);
    packet.pts = m_current_track_time;
    packet.duration = current_frame_count;
    packet.flags |= AV_compat::pkt_flag_key;
    packet.stream_index = m_stream->index;
    packet.data = const_cast<uint8_t*>(write_buffer);
    packet.size = current_frame_count * m_sound_source.frame_size();
    // write the audio data
    const int write_return = av_interleaved_write_frame(m_format_context, &packet);
    on_error(write_return, new Build_error(error_location(),
                                           Build_error::temporary_movie_write,
                                           "",
                                           "",
                                           "error writing audio frames"));
    m_current_track_time += packet.duration;
    remaining_frame_count -= current_frame_count;
  }

  return_value = true;
  goto exit_point;

  end_error_block();

  default_error_handler_and_cleanup(return_error,
                                    return_value,
                                    false);
  goto exit_point;

 exit_point:
  postcondition(m_setup
                && m_started
                && return_error.is_valid_at_return(return_value)
                && mf_invariant());
  return return_value;
}

void Sound_track::mf_convert_endianness(const uint8_t* input_buffer, Long_int frame_count)
{
  const Long_int sample_count = frame_count * m_config.channels;

  const int sample_size = m_sound_source.frame_size() / m_config.channels;
  if (2 == sample_size)
  {
    const uint16_t* input_buffer_pos = reinterpret_cast<const uint16_t*>(input_buffer);
    uint16_t* output_buffer_pos = reinterpret_cast<uint16_t*>(m_output_conversion_buffer);
    for (Long_int sample_number = 0; sample_number < sample_count; ++sample_number)
    {
      *output_buffer_pos = *input_buffer_pos;
      ih_endian_swap16(*output_buffer_pos);
      ++output_buffer_pos;
      ++input_buffer_pos;
    }
  }
  else if (4 == sample_size)
  {
    const uint32_t* input_buffer_pos = reinterpret_cast<const uint32_t*>(input_buffer);
    uint32_t* output_buffer_pos = reinterpret_cast<uint32_t*>(m_output_conversion_buffer);
    for (Long_int sample_number = 0; sample_number < sample_count; ++sample_number)
    {

      *output_buffer_pos = *input_buffer_pos;
      ih_endian_swap32(*output_buffer_pos);
      ++output_buffer_pos;
      ++input_buffer_pos;
    }
  }
  else if (8 == sample_size)
  {
    const uint64_t* input_buffer_pos = reinterpret_cast<const uint64_t*>(input_buffer);
    uint64_t* output_buffer_pos = reinterpret_cast<uint64_t*>(m_output_conversion_buffer);
    for (Long_int sample_number = 0; sample_number < sample_count; ++sample_number)
    {
      *output_buffer_pos = *input_buffer_pos;
      ih_endian_swap64(*output_buffer_pos);
      ++output_buffer_pos;
      ++input_buffer_pos;
    }
  }
  else
  {
    assert(false && "error, invalid sample length for conversion");
  }
}
