// RingBuffer.hpp
//
// Copyright 2011-2013 Roan Trail, Inc.
//
// This file is part of Kinetophone.
//
// Kinetophone is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published
// by the Free Software Foundation, either version 2 of the License,
// or (at your option) any later version.
//
// Kinetophone is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// General Public License for more details.  You should have received
// a copy of the GNU General Public License along with Kinetophone. If
// not, see <http://www.gnu.org/licenses/>.

//
// This code was based on the PortAudio file pa_ringbuffer.h:
//
//    This program uses the PortAudio Portable Audio Library.
//    For more information see: http://www.portaudio.com
//    Copyright (c) 1999-2000 Ross Bencina and Phil Burk
//
//    Permission is hereby granted, free of charge, to any person obtaining
//    a copy of this software and associated documentation files
//    (the "Software"), to deal in the Software without restriction,
//    including without limitation the rights to use, copy, modify, merge,
//    publish, distribute, sublicense, and/or sell copies of the Software,
//    and to permit persons to whom the Software is furnished to do so,
//    subject to the following conditions:
//
//    The above copyright notice and this permission notice shall be
//    included in all copies or substantial portions of the Software.
//
//    THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
//    EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
//    MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
//    IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR
//    ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF
//    CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
//    WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

// Ring_buffer is a ring buffer used to transport samples between
// different execution contexts (threads, OS callbacks, interrupt handlers)
// without requiring the use of locks. This only works when there is
// a single reader and a single writer (ie. one thread or callback writes
// to the ring buffer, another thread or callback reads from it).

// The Ring_buffer class manages a ring buffer containing N
// elements, where N must be a power of two. An element may be any size
// (specified in bytes).

#ifndef ROAN_TRAIL_KINETOPHONE_RING_BUFFER_HPP_
#define ROAN_TRAIL_KINETOPHONE_RING_BUFFER_HPP_

#include <kinetophone/common.hpp>

namespace Roan_trail
{
  namespace Kinetophone
  {
    class Ring_buffer
    {
    public:
      // Constructor/destructor/copy
      explicit Ring_buffer(Long_int num_bytes);
      virtual ~Ring_buffer();
      // read/write
      Long_int read_ahead_data(void* data, Long_int num_bytes) const;
      Long_int write_data(const void* data, Long_int num_bytes);
      Long_int read_data(void* data, Long_int num_bytes);
      // accessors
      // (Note: read_available() has a benign race condition due to lock free algorithm)
      Long_int read_available() const
      { __sync_synchronize(); return ((m_write_index - m_read_index) &  m_big_mask); }
      Long_int write_available() const { return (m_buffer_size - read_available()); }
      Long_int buffer_size() const { return m_buffer_size; }
      // mutators
      //   should only be called when buffer is NOT being read.
      void flush() { m_write_index = m_read_index = 0; }
      void clear(); // zero out
    protected:
      // invariant check
      bool mf_invariant(bool check_base_class = true) const;
    private:
      // read/write
      Long_int mf_write_regions(Long_int num_bytes,
                                void* *data_ptr_1,
                                Long_int* size_ptr_1,
                                void* *data_ptr_2,
                                Long_int* size_ptr_2);
      Long_int mf_read_regions(Long_int num_bytes,
                               void* *data_ptr_1,
                               Long_int* size_ptr_1,
                               void* *data_ptr_2,
                               Long_int* size_ptr_2) const;
      // mutators
      Long_int mf_advance_write_index(Long_int num_bytes);
      Long_int mf_advance_read_index(Long_int num_bytes);
      //
      const Long_int m_buffer_size; // Number of elements in FIFO. Power of 2.
      Long_int m_write_index;       // Index of next writable element.
      Long_int m_read_index;        // Index of next readable element.
      const Long_int m_big_mask;    // Used for wrapping indices with extra bit to distinguish full/empty.
      const Long_int m_small_mask;  // Used for fitting indices to buffer.
      char* m_buffer;               // Pointer to the buffer containing the actual data.
      // prevent the compiler from generating
      Ring_buffer(const Ring_buffer& rb);
      Ring_buffer& operator=(const Ring_buffer& rb);
    };
  }
}

#endif // ROAN_TRAIL_KINETOPHONE_RING_BUFFER_HPP_
