// Kinetophone_PDF_slide_collection.cpp
//
// Copyright 2011-2013 Roan Trail, Inc.
//
// This file is part of Kinetophone.
//
// Kinetophone is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published
// by the Free Software Foundation, either version 2 of the License,
// or (at your option) any later version.
//
// Kinetophone is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// General Public License for more details.  You should have received
// a copy of the GNU General Public License along with Kinetophone. If
// not, see <http://www.gnu.org/licenses/>.

#include <kinetophone/PDF_slide_collection_glib.hpp>
#include <kinetophone/PDF_slide_collection_glib_impl.hpp>

// include *mm first to avoid conflicts
#include <gdk/gdk.h>
#include <gdkmm/pixbuf.h>
#include <gtkmm/invisible.h>
#include <gtkmm/stock.h>

#include <kinetophone/common.hpp>
#include <kinetophone/Image_types.hpp>
#include <kinetophone/PDF_slide_collection.hpp>
#include <string>

using Glib::RefPtr;
using Gdk::Pixbuf;
using std::string;
using Roan_trail::Kinetophone::min;
using Roan_trail::Kinetophone::max;
using Roan_trail::Kinetophone::Rect_size;
using Roan_trail::Kinetophone::PDF_slide_collection;
using Roan_trail::Kinetophone::Error_param;
using Roan_trail::Kinetophone::Source_error;
using Roan_trail::Kinetophone_impl::PDF_slide_collection_glib_impl;
using namespace Roan_trail::Kinetophone_glib;

//
// Constructor/destructor
//
PDF_slide_collection_glib::PDF_slide_collection_glib()
  : PDF_slide_collection<Glib::RefPtr<Gdk::Pixbuf> >(),
    m_impl(new PDF_slide_collection_glib_impl)
{
  postcondition(mf_invariant(false));
}

PDF_slide_collection_glib::~PDF_slide_collection_glib()
{
  precondition(mf_invariant(false));

  delete m_impl;
}

//
// Accessors
//

bool PDF_slide_collection_glib::image_for_slide_at(int slide_index,
                                                   const Rect_size& suggested_size,
                                                   RefPtr<Pixbuf>& return_image) const
{
  precondition(((slide_index >= 0) && (slide_index < slide_count()))
               && !return_image
               && mf_invariant());

  const bool return_value = m_impl->image_for_slide_at(slide_index,
                                                       suggested_size,
                                                       fill_color(),
                                                       return_image);

  postcondition(return_image
                && mf_invariant());

  return return_value;
}

void PDF_slide_collection_glib::thumbnail_for_slide_at(int slide_index, RefPtr<Pixbuf>& return_image) const
{
  precondition(((slide_index >= 0) && (slide_index < slide_count()))
               && !return_image
               && mf_invariant());

  m_impl->thumbnail_for_slide_at(slide_index,
                                 fill_color(),
                                 return_image);

  postcondition(return_image
                && mf_invariant());
}

Rect_size PDF_slide_collection_glib::max_bounds() const
{
  precondition(mf_invariant());

  return m_impl->max_bounds(slide_count(), resolution());
}

//
// Protected member functions
//

bool PDF_slide_collection_glib::mf_invariant(bool check_base_class) const
{
  bool return_value = false;

  if (!m_impl)
  {
    goto exit_point;
  }

  return_value = !check_base_class
    || PDF_slide_collection<RefPtr<Pixbuf> >::mf_invariant(check_base_class);

  goto exit_point;

 exit_point:
  return return_value;
}

// virtual function for specific image (Gdk::Pixbuf in this case)
bool PDF_slide_collection_glib::mf_process_source_with_notes(const string& source_path,
                                                                    Error_param& return_error)
{
  precondition(!return_error());

  bool return_value = false;

  start_error_block();

  Error_param error;
  const bool impl_processed = m_impl->process_source_with_notes(source_path,
                                                                import_PDF_notes(),
                                                                m_slides,
                                                                error);
  on_error(!impl_processed, new Source_error(error_location(),
                                             Source_error::general,
                                             error()));

  return_value = true;

  goto exit_point;

  end_error_block();

  default_error_handler_and_cleanup(return_error,
                                    return_value,
                                    false);

  goto exit_point;

 exit_point:
  postcondition(return_error.is_valid_at_return(return_value));
  return return_value;
}


// virtual function for specific PDF document class
bool PDF_slide_collection_glib::mf_load_document(const string& source_path,
                                                 Error_param& return_error)
{
  precondition(!return_error());

  bool return_value = false;

  start_error_block();

  Error_param error;
  const bool impl_loaded = m_impl->load_document(source_path, error);
  on_error(!impl_loaded, new Source_error(error_location(),
                                          Source_error::general,
                                          error()));

  return_value = true;

  goto exit_point;

  end_error_block();

  default_error_handler_and_cleanup(return_error,
                                    return_value,
                                    false);
  goto exit_point;

 exit_point:
  postcondition(return_error.is_valid_at_return(return_value));
  return return_value;
}
