// Option_validators.hpp
//
// Copyright 2011-2013 Roan Trail, Inc.
//
// This file is part of Kinetophone.
//
// Kinetophone is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published
// by the Free Software Foundation, either version 2 of the License,
// or (at your option) any later version.
//
// Kinetophone is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// General Public License for more details.  You should have received
// a copy of the GNU General Public License along with Kinetophone. If
// not, see <http://www.gnu.org/licenses/>.

// Program option validation support

#ifndef ROAN_TRAIL_KINETOPHONE_OPTION_VALIDATORS_HPP_
#define ROAN_TRAIL_KINETOPHONE_OPTION_VALIDATORS_HPP_

#include <kinetophone/Image_types.hpp>
#include <boost/algorithm/string.hpp>
#include <boost/program_options.hpp>
#include <boost/concept/assert.hpp>
#include <boost/concept_check.hpp>
#include <iomanip>
#include <iostream>
#include <sstream>
#include <string>
#include <vector>

namespace Roan_trail
{
  namespace Kinetophone
  {
    //
    // Validator template
    //

    template<class T> struct Validator
    {
      BOOST_CONCEPT_ASSERT((boost::CopyConstructible<T>));
      BOOST_CONCEPT_ASSERT((boost::DefaultConstructible<T>));
      Validator(const T& a) : v(a) {}
      Validator() : v(T()) {}
      T v;
    };

    template<class T> void validate(boost::any& v, const std::vector<std::string>& xs, Validator<T>*, int)
    {
      boost::program_options::check_first_occurrence(v);
      v = boost::any(Validator<T>(boost::lexical_cast<T>(boost::program_options::get_single_string(xs))));
    }

    template<class T> std::ostream& operator<<(std::ostream& os, const Validator<T>& c)
    {
      return os << c.v;
    }

    //
    //   Boolean
    //

    std::ostream& operator<<(std::ostream& os, const Validator<bool>& c)
    {
      return os << (c.v ? "yes" : "no");
    }

    void validate(boost::any& v,
                  const std::vector<std::string>& values,
                  Validator<bool>*, int)
    {
      boost::program_options::check_first_occurrence(v);
      std::string s(boost::program_options::get_single_string(values, true));

      for (std::string::size_type i = 0; i < s.size(); ++i)
      {
        s[i] = char(tolower(s[i]));
      }

      if (s.empty() || (s == "on") || (s == "yes") || (s == "1") || (s == "true"))
      {
        v = boost::any(Validator<bool>(true));
      }
      else if ((s == "off") || (s == "no") || (s == "0") || (s == "false"))
      {
        v = boost::any(Validator<bool>(false));
      }
      else
      {
        throw boost::program_options::
          validation_error(boost::program_options::validation_error::invalid_bool_value, s);
      }
    }

    //
    // Color validator
    //

    struct Color_validator
    {
      Color_validator(const Color& c) : v(c) {}
      Color_validator() : v() {}
      Color v;
    };

    template <typename T> struct Hex_to
    {
      BOOST_CONCEPT_ASSERT((boost::DefaultConstructible<T>));
      T v;
      operator T() const { return v; }
      friend std::istream& operator>>(std::istream& in, Hex_to& out)
      {
        in >> std::hex >> out.v;
        return in;
      }
    };

    void validate(boost::any& v, const std::vector<std::string>& xs, Color_validator*, int)
    {
      boost::program_options::check_first_occurrence(v);
      const std::string& s = boost::program_options::validators::get_single_string(xs);
      uint32_t raw_value = boost::lexical_cast<Hex_to<uint32_t> >(s);
      Color c(raw_value, false);
      v = boost::any(Color_validator(c));
    }

    std::ostream& operator<<(std::ostream& os, const Color_validator& c)
    {
      uint32_t scalar_color = c.v.scalar_value();
      std::stringstream hex_stream;
      hex_stream << std::setw(6) << std::setfill('0') << std::hex << scalar_color;
      std::string upper_hex = hex_stream.str();
      boost::algorithm::to_upper(upper_hex);
      return os << upper_hex;
    }

    //
    // Condition validators
    //

    template<class T, int i> struct Condition_validator
    {
      BOOST_CONCEPT_ASSERT((boost::CopyConstructible<T>));
      BOOST_CONCEPT_ASSERT((boost::DefaultConstructible<T>));
      Condition_validator(const T& a) : v(a) {}
      Condition_validator() : v(T()) {}
      bool cond() const;
      T v;
    };

    template<class T, int i> std::ostream& operator<<(std::ostream& os, const Condition_validator<T, i>& c)
    {
      return os << c.v;
    }

    template<class T, int i> void validate(boost::any& v,
                                           const std::vector<std::string>& values,
                                           Condition_validator<T, i>*,
                                           int)
    {
      boost::program_options::check_first_occurrence(v);
      std::string s(boost::program_options::get_single_string(values, true));

      T a = boost::lexical_cast<T>(s);
      Condition_validator<T, i> cv(a);

      if (cv.cond())
      {
        v = boost::any(cv);
      }
      else
      {
        throw boost::program_options::
          validation_error(boost::program_options::validation_error::invalid_option_value);
      }
    }

    //
    //   Positive double validator
    //

    typedef struct Condition_validator<double, 0> Positive_double_validator;
    template<> bool Positive_double_validator::cond() const
    {
      return v > 0.0;
    }

    //
    //   Positive int validator
    //

    typedef struct Condition_validator<int, 0> Positive_int_validator;
    template<> bool Positive_int_validator::cond() const
    {
      return v > 0;
    }

    //
    //   Fraction validator
    //

    typedef struct Condition_validator<double, 1> Fraction_validator;
    template<> bool Fraction_validator::cond() const
    {
      return ((v >= 0.0) && (v <= 1.0));
    }

    //
    // Directory validator
    //

    struct Directory_validator
    {
      Directory_validator(const std::string& a) : v(a) {}
      Directory_validator() : v(std::string()) {}
      std::string v;
    };

    std::ostream& operator<<(std::ostream& os, const Directory_validator& d)
    {
      return os << d.v;
    }

    void validate(boost::any& v,
                  const std::vector<std::string>& values,
                  Directory_validator*,
                  int)
    {
      boost::program_options::check_first_occurrence(v);
      std::string s(boost::program_options::get_single_string(values, true));

      Directory_validator dv(s);

      if (File_manager::path_is_directory(dv.v))
      {
        v = boost::any(dv);
      }
      else
      {
        throw boost::program_options::
          validation_error(boost::program_options::validation_error::invalid_option_value);
      }
    }

    //
    // Lookup validators
    //

    //
    //   String to int
    //

    typedef bool (*Str_to_int_lookup_ptr)(const std::string&, int&);

    template<class T, Str_to_int_lookup_ptr F> struct Str_to_int_lookup_validator
    {
      BOOST_CONCEPT_ASSERT((boost::CopyConstructible<T>));
      Str_to_int_lookup_validator(const T& a) : v(a) {}
      T v;
    };

    template<class T, Str_to_int_lookup_ptr F>
    std::ostream& operator<<(std::ostream& os, const Str_to_int_lookup_validator<T, F>& c)
    {
      return os << c.v;
    }

    template<class T, Str_to_int_lookup_ptr F> void validate(boost::any& v,
                                                             const std::vector<std::string>& values,
                                                             Str_to_int_lookup_validator<T, F>*, int)
    {
      boost::program_options::validators::check_first_occurrence(v);
      // error if more than one string supplied
      const std::string& s = boost::program_options::validators::get_single_string(values);

      int type;
      if (F(s, type))
      {
        v = boost::any(Str_to_int_lookup_validator<T, F>(s));
      }
      else
      {
        throw boost::program_options::
          validation_error(boost::program_options::validation_error::invalid_option_value);
      }
    }

    //
    //   String to string
    //

    typedef std::string (*Str_to_str_lookup_ptr)(const std::string&);

    template<class T, Str_to_str_lookup_ptr F> struct Str_to_str_lookup_validator
    {
      BOOST_CONCEPT_ASSERT((boost::CopyConstructible<T>));
      Str_to_str_lookup_validator(const T& a) : v(a) {}
      T v;
    };

    template<class T, Str_to_str_lookup_ptr F> std::ostream& operator<<(std::ostream& os,
                                                                        const Str_to_str_lookup_validator<T, F>& c)
    {
      return os << c.v;
    }

    template<class T, Str_to_str_lookup_ptr F> void validate(boost::any& v,
                                                             const std::vector<std::string>& values,
                                                             Str_to_str_lookup_validator<T, F>*, int)
    {
      boost::program_options::validators::check_first_occurrence(v);
      // error if more than one string supplied
      const std::string& s = boost::program_options::validators::get_single_string(values);

      if ("" != F(s))
      {
        v = boost::any(Str_to_str_lookup_validator<T, F>(s));
      }
      else
      {
        throw boost::program_options::
          validation_error(boost::program_options::validation_error::invalid_option_value);
      }
    }
  }
}

#endif // ROAN_TRAIL_KINETOPHONE_OPTION_VALIDATORS_HPP_
