// Narration_synthesizer.cpp
//
// Copyright 2011-2013 Roan Trail, Inc.
//
// This file is part of Kinetophone.
//
// Kinetophone is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published
// by the Free Software Foundation, either version 2 of the License,
// or (at your option) any later version.
//
// Kinetophone is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// General Public License for more details.  You should have received
// a copy of the GNU General Public License along with Kinetophone. If
// not, see <http://www.gnu.org/licenses/>.

#include <kinetophone/Narration_synthesizer.hpp>
#include <kinetophone/Action_recorder.hpp>
#include <kinetophone/Segment.hpp>
#include <kinetophone/Speech_synthesizer.hpp>
#include <kinetophone/Speech_synthesizer_config.hpp>
#include <kinetophone/Sound_file_config.hpp>
#include <kinetophone/error/Record_error.hpp>
#include <libxml++/exceptions/exception.h>
#include <libxml++/nodes/element.h>
#include <libxml++/nodes/node.h>
#include <libxml2/libxml/tree.h>
#include <string>
#include <sstream>
#include <vector>
#include <cstddef>

using std::endl;
using std::string;
using std::stringstream;
using std::vector;
using xmlpp::Element;
using xmlpp::Node;
using namespace Roan_trail::Kinetophone;

//
// Internal helpers
//

namespace
{
  string ih_format_XML_file_line(int line)
  {
    stringstream line_stream;
    line_stream << ":" << line;

    return line_stream.str();
  }
}

//
// Constructor/destructor
//

Narration_synthesizer::Narration_synthesizer()
  : m_started(false),
    m_first_speech(true),
    m_action_recorder(new Action_recorder),
    m_speech_synthesizer(new Speech_synthesizer),
    m_segment_cache(),
    m_temporary_directory(),
    m_index_frames_written_count(0)
{
  postcondition(mf_invariant(false));
}

Narration_synthesizer::~Narration_synthesizer()
{
  precondition(mf_invariant(false));

  // shutdown, if not already
  if (m_started)
  {
    Error_param error(false); // ignore error
    shutdown(false, error);
  }

  delete m_speech_synthesizer;
  delete m_action_recorder;
}

//
// Control
//

//
//   ...startup/shutdown functions
//

bool Narration_synthesizer::startup(Speech_synthesizer_config& config, Error_param& return_error)
{
  precondition(!return_error()
               && mf_invariant());

  bool return_value = false;

  start_error_block();

  if (m_started)
  {
    return_value = true;
    goto exit_point;
  }

  // start speech synthesizer
  Error_param error;
  const bool started = m_speech_synthesizer->startup(config, error);
  on_error(!started, new Record_error(error_location(),
                                      Record_error::general,
                                      error()));

  const Long_int audio_frame_rate =
    static_cast<Long_int>(m_speech_synthesizer->config()->sound_file->sample_rate);
  m_action_recorder->set_audio_frame_rate(audio_frame_rate);
  m_index_frames_written_count = 0;
  m_segment_cache.clear();

  m_started = true;
  m_first_speech = true;
  return_value = true;
  goto exit_point;

  end_error_block();

  default_error_handler_and_cleanup(return_error,
                                    return_value,
                                    false);
  goto exit_point;

 exit_point:
  postcondition(( // success
                 (return_value && m_started)
                 // failure
                 || !return_value)
                && return_error.is_valid_at_return(return_value)
                && mf_invariant());
  return return_value;
}

bool Narration_synthesizer::shutdown(bool completed, Error_param& return_error)
{
  precondition(!return_error()
               && mf_invariant());

  bool return_value = false;

  start_error_block();

  Error_param error;

  if (!m_started)
  {
    return_value = true;
    goto exit_point;
  }

  if (Recorder::state_stopped != m_action_recorder->state())
  {
    const Long_int frame = m_speech_synthesizer->frames_written_count();
    m_index_frames_written_count = frame;
    m_action_recorder->stop(frame);
  }

  const bool ended = m_speech_synthesizer->shutdown(completed, error);
  m_started = false;
  on_error(!ended, new Record_error(error_location(),
                                    Record_error::general,
                                    error()));

  return_value = true;
  goto exit_point;

  end_error_block();

  default_error_handler_and_cleanup(return_error,
                                    return_value,
                                    false);

 exit_point:
  postcondition(!m_started
                && return_error.is_valid_at_return(return_value)
                && mf_invariant());
  return return_value;
}

//
// narration functions
//

bool Narration_synthesizer::speak(Long_int index,
                                  const string& text,
                                  Error_param& return_error)
{
  precondition(!return_error()
               && mf_invariant());

  bool return_value = false;

  start_error_block();

  on_error(!m_started, new Record_error(error_location(),
                                        Record_error::record,
                                        string("Error, narration synthesizer cannot speak")));

  if (m_first_speech)
  {
    m_action_recorder->record(index);
    m_first_speech = false;
  }
  else
  {
    const Long_int frame = m_speech_synthesizer->frames_written_count();
    m_index_frames_written_count = frame;
    m_action_recorder->new_index(frame, index);
  }

  Error_param error;
  const bool spoke = m_speech_synthesizer->speak(text, error);
  on_error(!spoke, new Record_error(error_location(),
                                    Record_error::general,
                                    error()));

  return_value = true;
  goto exit_point;

  end_error_block();

  default_error_handler_and_cleanup(return_error,
                                    return_value,
                                    false);
  goto exit_point;

 exit_point:
  postcondition(return_error.is_valid_at_return(return_value)
                && mf_invariant());
  return return_value;
}

bool Narration_synthesizer::end(Error_param& return_error)
{
  precondition(!return_error()
               && mf_invariant());

  if (!m_first_speech && m_started)
  {
    if (Recorder::state_stopped != m_action_recorder->state())
    {
      const Long_int frame = m_speech_synthesizer->frames_written_count();
      m_index_frames_written_count = frame;
      m_action_recorder->stop(frame);
    }
  }

  postcondition(m_started
                && !return_error()
                && mf_invariant());
  return true;
}


void Narration_synthesizer::recorded_time(string& return_recorded_time, char separator) const
{
  precondition(mf_invariant());

  m_speech_synthesizer->recorded_time(return_recorded_time, separator);
}

void Narration_synthesizer::recorded_time_for_current_index(string& return_recorded_time, char separator) const
{
  precondition(mf_invariant());

  Long_int frames = 0;
  if (Recorder::state_stopped != m_action_recorder->state())
  {
    const Long_int current_audio_frame = m_speech_synthesizer->frames_written_count();
    frames = current_audio_frame - m_index_frames_written_count;
    assert(frames >= 0 && "error, invalid frame count for current index");
  }

  m_speech_synthesizer->frames_to_recorded_time(frames, separator, return_recorded_time);
}

bool Narration_synthesizer::frames(Long_int& return_total_frames,
                                   Long_int& return_index_frames,
                                   Error_param& return_error) const
{
  precondition(!return_error()
               && mf_invariant());

  bool return_value = false;

  const Long_int current_audio_frame = m_speech_synthesizer->frames_written_count();
  return_total_frames = current_audio_frame;
  const Long_int index_audio_frame = current_audio_frame - m_index_frames_written_count;
  return_index_frames = index_audio_frame;

  return_value = true;
  goto exit_point;

 exit_point:
  postcondition(return_value
                && !return_error()
                && mf_invariant());
  return return_value;
}

const Speech_synthesizer_config* Narration_synthesizer::config() const
{
  precondition(mf_invariant());

  return m_speech_synthesizer->config();
}

string Narration_synthesizer::output_file() const
{
  precondition(mf_invariant());

  return m_speech_synthesizer->output_file();
}

bool Narration_synthesizer::have_recording() const
{
  precondition(mf_invariant());

  return m_speech_synthesizer->have_recording();
}

bool Narration_synthesizer::segments(vector<Segment>& return_segments, Error_param& return_error) const
{
  precondition(!return_error()
               && mf_invariant());

  bool return_value = false;

  start_error_block();

  Error_param error;
  const bool got_segments = m_action_recorder->segments(return_segments, error);
  if (!got_segments)
  {
    string sound_file = m_speech_synthesizer->output_file();
    if ("" == sound_file)
    {
      sound_file = "unknown file";
    }
    Error *record_error = new Record_error(error_location(),
                                           Record_error::general,
                                           error());
    record_error->error_dictionary()[Error::file_path_error_key] = sound_file;
    on_error(true, record_error);
  }

  return_value = true;
  goto exit_point;

  end_error_block();

  default_error_handler_and_cleanup(return_error,
                                    return_value,
                                    false);
  goto exit_point;

 exit_point:
  postcondition(return_error.is_valid_at_return(return_value));
  return return_value;
}

bool Narration_synthesizer::space_available(Long_int& return_bytes_available,
                                            double& return_fraction_available,
                                            double& return_seconds_remaining) const
{
  precondition(mf_invariant());

  return m_speech_synthesizer->space_available(return_bytes_available,
                                              return_fraction_available,
                                              return_seconds_remaining);
}

bool Narration_synthesizer::output_file_size(Long_int& return_file_size, double& return_data_rate) const
{
  precondition(mf_invariant());

  return m_speech_synthesizer->output_file_size(return_file_size, return_data_rate);
}

Long_int Narration_synthesizer::frames_written_count() const
{
  precondition(mf_invariant());

  return m_speech_synthesizer->frames_written_count();
}

//
// Export
//

bool Narration_synthesizer::save_as_XML(const string& file_path,
                                        Node* XML_node,
                                        Error_param& return_error)
{
  precondition(XML_node
               && !return_error());

  bool return_value = false;
  start_error_block();

  Error_param error;

  try
  {
    Node* recorder_node = XML_node->add_child("recorder");

    // audio recording path
    Element* audio_recording_path_element = recorder_node->add_child("audio_recording_path");
    audio_recording_path_element->add_child_text(output_file());

    // audio recording frame rate
    Element* audio_recording_frame_rate_element = recorder_node->add_child("audio_recording_frame_rate");
    stringstream frame_rate_stream;
    frame_rate_stream << config()->sound_file->sample_rate;
    audio_recording_frame_rate_element->add_child_text(frame_rate_stream.str());

    if (have_recording())
    {
      vector<Segment> recording_segments;
      bool got_segments = segments(recording_segments, error);
      on_error(!got_segments, new Record_error(error_location(),
                                               Record_error::general,
                                               error()));
      // segments
      Node* segments_node = recorder_node->add_child("segments");;
      for (size_t segment_number = 0; segment_number < recording_segments.size(); ++segment_number)
      {
        const Segment &segment = recording_segments[segment_number];
        Node* segment_node = segments_node->add_child("segment");
        // index
        Element* index_element = segment_node->add_child("index");
        stringstream segment_index_stream;
        segment_index_stream << segment.index();
        index_element->add_child_text(segment_index_stream.str());
        // start frame
        Element* start_frame_element = segment_node->add_child("start_frame");
        stringstream start_frame_stream;
        start_frame_stream << segment.start_frame();
        start_frame_element->add_child_text(start_frame_stream.str());
        // end frame
        Element* end_frame_element = segment_node->add_child("end_frame");
        stringstream end_frame_stream;
        end_frame_stream << segment.end_frame();
        end_frame_element->add_child_text(end_frame_stream.str());
        // ordinality
        Element* ordinality_element = segment_node->add_child("ordinality");
        stringstream ordinality_stream;
        ordinality_stream << segment.ordinality();
        ordinality_element->add_child_text(ordinality_stream.str());
        Element* retake_element = segment_node->add_child("retake");
        retake_element->add_child_text(segment.is_retake() ? "1" : "0");
      }
    }
  }
  catch (const xmlpp::exception& e)
  {
    string diagnostic = string("error exporting XML to file: ") + e.what();
    on_error(true, new Record_error(error_location(),
                                    Record_error::save,
                                    diagnostic,
                                    file_path));
  }

  return_value = true;

  goto exit_point;
  end_error_block();

  default_error_handler_and_cleanup(return_error,
                                    return_value,
                                    false);
  goto exit_point;

 exit_point:
  return return_value;
}

//
// Protected member functions
//

// invariant check
bool Narration_synthesizer::mf_invariant(bool check_base_class) const
{
  bool return_value = false;

  if (m_index_frames_written_count < 0)
  {
    goto exit_point;
  }

  // member objects which must exist during this object's lifetime
  if (!m_speech_synthesizer || !m_action_recorder)
  {
    goto exit_point;
  }

  return_value = true;
  goto exit_point;

 exit_point:
  return return_value;
}
