// Narration_recorder.hpp
//
// Copyright 2011-2013 Roan Trail, Inc.
//
// This file is part of Kinetophone.
//
// Kinetophone is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published
// by the Free Software Foundation, either version 2 of the License,
// or (at your option) any later version.
//
// Kinetophone is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// General Public License for more details.  You should have received
// a copy of the GNU General Public License along with Kinetophone. If
// not, see <http://www.gnu.org/licenses/>.

#ifndef ROAN_TRAIL_KINETOPHONE_NARRATION_RECORDER_HPP_
#define ROAN_TRAIL_KINETOPHONE_NARRATION_RECORDER_HPP_

#include <kinetophone/common.hpp>
#include <kinetophone/Recorder.hpp>
#include <string>
#include <vector>

namespace xmlpp
{
  class Node;
}

namespace Roan_trail
{
  namespace Kinetophone
  {
    class Action_recorder;
    class Error;
    class Error_param;
    class Sound_recorder;
    class Sound_recorder_config;
    class Segment;

    class Narration_recorder : public Recorder {
    public:
      // constructor/destructor
      Narration_recorder();
      virtual ~Narration_recorder();
      // control
      //
      //  Note: The Narration_recorder class is meant to be used directly from a library,
      //        so the control functions are forgiving of errors such as recording state.
      //        They will generally return an error rather than using an assertion.
      //
      //   startup/shutdown functions
      bool startup(Sound_recorder_config& config, Error_param& return_error);
      bool shutdown(Error_param& return_error);
      //   basic Recorder functions
      bool record(Error_param& return_error);
      bool pause(Error_param& return_error);
      bool stop(Error_param& return_error);
      //   narration functions
      bool record(Long_int index, Error_param& return_error);
      bool new_index(Long_int index, Error_param& return_error);
      bool retake(Error_param& return_error);
      bool mute(Error_param& return_error);
      // accessors
      bool is_muted() const;
      bool is_started() const;
      bool can_record() const;
      void recorded_time(std::string &return_recorded_time, char separator = ':') const;
      void recorded_time_for_current_index(std::string &return_recorded_time, char separator = ':') const;
      void sound_levels_RMS(std::vector<double>& return_sound_levels) const;
      void sound_levels_peak(std::vector<double>& return_sound_levels) const;
      bool frames(Long_int& return_total_frames,
                  Long_int& return_index_frames,
                  Error_param& return_error) const;
      bool segments(std::vector<Segment>& return_segments, Error_param& return_error) const;
      const Sound_recorder_config* config() const;
      std::string output_file() const;
      bool have_recording() const;
      const Error* record_error() const;
      Long_int overflow_count() const;
      Long_int input_overflow_count() const;
      Long_int output_overflow_count() const;
      bool space_available(Long_int& return_bytes_available,
                           double& return_fraction_available,
                           double& return_seconds_remaining) const;
      bool output_file_size(Long_int& return_file_size, double& return_data_rate) const;
      bool is_RMS_metering_enabled() const;
      bool is_peak_metering_enabled() const;
      int metering_channels() const;
      // mutators
      void reset_overflow_count();
      void enable_RMS_metering(bool enable);
      void enable_peak_metering(bool enable);
      void set_metering_channels(int channel_count);
      void cleanup_temporary_file();
      void clear_record_error();
      // export
      bool export_as_XML(const std::string& file_path,
                         xmlpp::Node* XML_node,
                         Error_param& return_error);
    protected:
      // invariant check
      bool mf_invariant(bool check_base_class = true) const;
    private:
      Action_recorder* m_action_recorder;
      Sound_recorder* m_sound_recorder;
      mutable std::vector<Segment> m_segment_cache;
      std::string m_temporary_directory;
      Long_int m_index_frames_written_count;
      // prevent compiler from generating
      Narration_recorder(const Narration_recorder& recorder);
      Narration_recorder& operator=(const Narration_recorder& recorder);
    };
  }
}

#endif // ROAN_TRAIL_KINETOPHONE_NARRATION_RECORDER_HPP_
