// Level_meter_view.cpp
//
// Copyright 2011-2013 Roan Trail, Inc.
//
// This file is part of Kinetophone.
//
// Kinetophone is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published
// by the Free Software Foundation, either version 2 of the License,
// or (at your option) any later version.
//
// Kinetophone is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// General Public License for more details.  You should have received
// a copy of the GNU General Public License along with Kinetophone. If
// not, see <http://www.gnu.org/licenses/>.

//
// This code was based on jackmeter.c:
//
//   jackmeter.c
//   Simple console based Digital Peak Meter for JACK
//   Copyright (C) 2005  Nicholas J. Humfrey
//
//   This program is free software; you can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your option) any later version.
//
//   This program is distributed in the hope that it will be useful,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//   GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program; if not, write to the Free Software
//   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

#include <kinetophone/Level_meter_view.hpp>
#include <kinetophone/error/Error.hpp>
#include <kinetophone/common.hpp>
#include <vector>
#include <cmath>
#include <cstddef>

using std::vector;
using namespace Roan_trail::Kinetophone;

namespace
{
  const double ic_RMS_factor = sqrt(2);
}

//
// Constructor/destructor
//

Level_meter_view::Level_meter_view(int channels,
                                   int update_rate,
                                   bool peak_meter_mode,
                                   bool vertical,
                                   bool model_ballistics,
                                   int attack_period,
                                   int decay_period,
                                   int peak_hold_period)
  : m_channels(channels),
    m_vertical(vertical),
    m_peak_meter_mode(peak_meter_mode),
    m_model_ballistics(model_ballistics),
    m_levels(),
    m_scaled_levels(),
    m_peak(),
    m_clipped(),
    m_attack_delta(0.0),
    m_decay_delta(0.0),
    m_peak_hold_count(0),
    m_peak_hold_counts()
{
  precondition((channels > 0)
               && (update_rate > 0));

  mf_set_channels();

  mf_set_rates(update_rate,
               ((attack_period < 1) ? default_attack_period : attack_period),
               ((decay_period < 1) ? default_decay_period : decay_period),
               ((peak_hold_period < 1) ? default_peak_hold_period : peak_hold_period));

  postcondition(mf_invariant(false));
}

Level_meter_view::~Level_meter_view()
{
  precondition(mf_invariant(false));
}

//
// Helpers
//

double Level_meter_view::scale_IEC(double dB)
{
  double deflection = 0.0; // meter deflection percentage

  if (dB < -70.0)
  {
    deflection = 0.0;
  }
  else if (dB < -60.0)
  {
    deflection = (dB + 70.0) * 0.25;
  }
  else if (dB < -50.0)
  {
    deflection = (dB + 60.0) * 0.5 + 2.5;
  }
  else if (dB < -40.0)
  {
    deflection = (dB + 50.0) * 0.75 + 7.5;
  }
  else if (dB < -30.0)
  {
    deflection = (dB + 40.0) * 1.5 + 15.0;
  }
  else if (dB < -20.0)
  {
    deflection = (dB + 30.0) * 2.0 + 30.0;
  }
  else if (dB < 0.0)
  {
    deflection = (dB + 20.0) * 2.5 + 50.0;
  }
  else
  {
    deflection = 100.0;
  }

  return (deflection / 100.0);
}

double Level_meter_view::convert_level_to_dB(double level) const
{
  // Full scale sine wave with RMS of 1.0 is 0.0 dBFS (ref. IEC 61606)
  // Note: this means .5 scale sine wave should be -6 dB

  precondition(!(level < 0.0)
               && mf_invariant());

  const double return_dB = 20.0 * log10(fabs(level));

  return return_dB;
}

//
// Mutators
//

void Level_meter_view::set_peak_meter_mode(bool peak_meter_mode)
{
  precondition(mf_invariant());

  m_peak_meter_mode = peak_meter_mode;

  mf_clear_vectors();

  postcondition(mf_invariant());
}


void Level_meter_view::set_levels(const vector<double>& levels)
{
  precondition(mf_invariant());

  const size_t channel_count = min(levels.size(), static_cast<size_t>(m_channels));
  for (size_t channel = 0; channel < channel_count; ++channel)
  {
    // scale by RMS factor if not in peak mode
    m_levels[channel] = (m_peak_meter_mode ? levels[channel] : (levels[channel] * ic_RMS_factor));
    if (!(m_levels[channel] < 1.0))
    {
      m_clipped[channel] = true;
    }
  }

  postcondition(mf_invariant());
}

void Level_meter_view::reset_clipped()
{
  precondition(mf_invariant());

  for (size_t channel = 0; channel < static_cast<size_t>(m_channels); ++channel)
  {
    m_clipped[channel] = false;
  }

  postcondition(mf_invariant());
}

//
// Class constants
//

const int Level_meter_view::default_attack_period = 300; // VU meter behavior
const int Level_meter_view::default_decay_period = 300; // VU meter behavior
const int Level_meter_view::default_peak_hold_period = 1600; // peak program meter behavior

//
// Protected class functions
//

bool Level_meter_view::mf_invariant(bool check_base_class) const
{
  static_cast<void>(check_base_class);

  bool return_value = false;

  // range checking
  if ((m_channels <= 0)
      || !(m_attack_delta > 0.0)
      || !(m_decay_delta < 0.0)
      || (m_peak_hold_count < 0))
  {
    goto exit_point;
  }
  else if (!m_levels.size()
           || !m_scaled_levels.size()
           || !m_peak.size()
           || !m_clipped.size()
           || !m_peak_hold_counts.size())
  {
    goto exit_point;
  }

  return_value = true;

 exit_point:
  return return_value;
}

//
// Private member functions
//

void Level_meter_view::mf_set_channels()
{
  const size_t number_channels = static_cast<size_t>(m_channels);

  m_levels.resize(number_channels);
  m_scaled_levels.resize(number_channels);
  m_peak.resize(number_channels);
  m_clipped.resize(number_channels);
  m_peak_hold_counts.resize(number_channels);

  mf_clear_vectors();
}

void Level_meter_view::mf_clear_vectors()
{
  const size_t number_channels = static_cast<size_t>(m_channels);

  for (size_t channel = 0; channel < number_channels; ++channel)
  {
    m_levels[channel] = 0.0;
    m_scaled_levels[channel] = 0.0;
    m_peak[channel] = 0.0;
    m_clipped[channel] = false;
    m_peak_hold_counts[channel] = 0;
  }
}

void Level_meter_view::mf_set_rates(int update_rate,
                                    int attack_period,
                                    int decay_period,
                                    int peak_hold_period)
{
  precondition((update_rate > 0)
               && (attack_period > 0)
               && (decay_period > 0)
               && (peak_hold_period >= 0));

  m_attack_delta = 1.0 / ((attack_period / 1000.0 / 0.99) * update_rate);
  m_decay_delta = -1.0 / ((decay_period / 1000.0 / 0.99) * update_rate);
  // calculate the decay length (expressed in update counts)
  m_peak_hold_count = static_cast<int>((peak_hold_period / 1000.0) * update_rate);
}

void Level_meter_view::update()
{
  size_t number_channels = static_cast<size_t>(m_channels);
  for (size_t channel = 0; channel < number_channels; ++channel)
  {
    const double dB = convert_level_to_dB(m_levels[channel]);
    const double target_scaled_level = scale_IEC(dB);
    double& scaled_level = m_scaled_levels[channel];
    if (m_model_ballistics)
    {
      double delta = target_scaled_level - scaled_level;
      double offset = 0.0;
      if (delta > 0.0)
      {
        offset = min(m_attack_delta, delta);
      }
      else if (delta < 0.0)
      {
        offset = max(m_decay_delta, delta);
      }
      // no last else needed

      scaled_level += offset;
    }
    else
    {
      scaled_level = target_scaled_level;
    }

    // update peak, if needed
    if (scaled_level > m_peak[channel])
    {
      m_peak[channel] = scaled_level;
      m_peak_hold_counts[channel] = 0;
    }
    else if (m_peak_hold_counts[channel]++ > m_peak_hold_count)
    {
      m_peak[channel] = scaled_level;
      m_peak_hold_counts[channel] = 0;
    }
    // no final else needed, we've updated as necessary
  }
}
