// Application.cpp
//
// Copyright 2011-2013 Roan Trail, Inc.
//
// This file is part of Kinetophone.
//
// Kinetophone is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published
// by the Free Software Foundation, either version 2 of the License,
// or (at your option) any later version.
//
// Kinetophone is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// General Public License for more details.  You should have received
// a copy of the GNU General Public License along with Kinetophone. If
// not, see <http://www.gnu.org/licenses/>.

#include <kinetophone/Application.hpp>
#include <kinetophone/common.hpp>
#include <kinetophone/File_manager.hpp>
#include <kinetophone/error/Error.hpp>
#include <string>
#include <new>
#include <iostream>
#include <sstream>
#include <csignal> // Standards Rule 21 deviation (check_code_ignore)

using std::cerr;
using std::endl;
using std::string;
using std::stringstream;
using std::set_new_handler;
using std::set_terminate;
using namespace Roan_trail::Kinetophone;

//
// Internal helper functions
//

namespace
{
  void ih_out_of_memory()
  {
    stringstream s;
    s << "Out of memory." << endl;
    Application::application()->terminate(1, s.str());
  }

  void ih_sig_handler(int sig_num)
  {
    static_cast<void>(sig_num); // avoid unused warning
    stringstream s;
    s << "Handling signal: " << sig_num << ", terminating." << endl;
    Application::application()->terminate(0, s.str());
  }

  void ih_uncaught_exception()
  {
    stringstream s;
    s << "Unhandled exception, terminating." << endl;
    Application::application()->terminate(1, s.str());
  }

  void ih_setup()
  {
    static bool have_setup = false;
    if (!have_setup)
    {
      // out of memory handler
      set_new_handler(ih_out_of_memory);
      // uncaught exeption handler
      set_terminate(ih_uncaught_exception);

      // set signal handlers for cleanup
      struct sigaction action;
      //   SIGABRT
      action.sa_handler = ih_sig_handler;
      action.sa_flags = 0;
      sigemptyset(&action.sa_mask);
      sigaction(SIGABRT, &action, 0);
      //   SIGINT
      action.sa_handler = ih_sig_handler;
      action.sa_flags = 0;
      sigemptyset(&action.sa_mask);
      sigaction(SIGINT, &action, 0);
      have_setup = true;

      // TODO: can call atexit() registration here
    }
  }
}

//
//
//

Application::~Application()
{
  precondition(mf_invariant(false));

  m_class_initialized = false;
  m_application = 0;

  postcondition(!m_application);
}

//
// Accessors
//

Application* Application::application()
{
  precondition(m_application);

  return m_application;
}

//
// Mutators
//

void Application::set_installation_dir(const string& installation_dir)
{
  string directory = installation_dir;
  if ("" == directory)
  {
    if (File_manager::executable_dir(directory))
    {
      directory += "/../share/kinetophone";
    }
    else
    {
      directory = ".";
    }
  }
  if (!File_manager::path_exists(directory))
  {
    directory = ".";
  }

  m_installation_dir = directory;
  Error::set_error_directory(m_installation_dir);
}

//
// Other
//

void Application::terminate(int code, const string& message)
{
  precondition(mf_invariant());

  if ("" != message)
  {
    cerr << message;
  }

  exit(code); // Standards Rule 24 deviation (check_code_ignore)

  // no postcondition, above call does not return
}

//
// Protected member functions
//

Application::Application(int argc, const char** argv)
  : m_argc(argc),
    m_argv(argv),
    m_installation_dir()
{
  // singleton by convention
  precondition(!m_application);

  m_application = this;

  set_installation_dir("");

  ih_setup();

  postcondition(mf_invariant(false));
}

//
//   Invariant
//

bool Application::mf_invariant(bool check_base_class) const
{
  static_cast<void>(check_base_class); // avoid unused warning

  return m_application;
}

//
// Static member data
//

Application *Application::m_application = 0;
bool Application::m_class_initialized = false;
