// AV_compat.hpp
//
// Copyright 2012-2014 Roan Trail, Inc.
//
// This file is part of Kinetophone.
//
// Kinetophone is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published
// by the Free Software Foundation, either version 2 of the License,
// or (at your option) any later version.
//
// Kinetophone is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// General Public License for more details.  You should have received
// a copy of the GNU General Public License along with Kinetophone. If
// not, see <http://www.gnu.org/licenses/>.

// Compatibility class for FFmpeg (supports Libav as well, although
// there may be some warnings when building in debug mode)

#ifndef ROAN_TRAIL_KINETOPHONE_AV_COMPAT_HPP_
#define ROAN_TRAIL_KINETOPHONE_AV_COMPAT_HPP_

#include <kinetophone/common.hpp>

extern "C"
{
#include <libavcodec/avcodec.h>
#include <libavformat/avformat.h>
#include <libavutil/avutil.h>

#if LIBAVCODEC_VERSION_INT >= AV_VERSION_INT(53, 42, 0) // FFmpeg 0.9
#if LIBAVUTIL_VERSION_INT >= AV_VERSION_INT(51, 32, 0) // FFmpeg 0.9
  // for av_get_default_channel_layout
#include <libavutil/audioconvert.h>
  //
#endif
#endif
}

namespace Roan_trail
{
  namespace Kinetophone
  {
    typedef int Endianness;

    struct Sound_file_config;

    class AV_compat
    {
    public:
      // typedefs
      typedef int Open_flags;

      // constructor/destructor
      AV_compat();
      ~AV_compat();

      // class member functions

#if LIBAVCODEC_VERSION_INT < AV_VERSION_INT(54, 59, 100) // FFmpeg 1.0
      static bool codec_ID_for_sound_file_config(const Sound_file_config& config,
                                                 enum CodecID& return_codec_ID);
#else
      static bool codec_ID_for_sound_file_config(const Sound_file_config& config,
                                                 enum AVCodecID& return_codec_ID);
#endif
      static bool endianness_for_output_format(const AVOutputFormat* output_format,
                                               const Sound_file_config& config,
                                               Endianness& return_endian);

#if LIBAVCODEC_VERSION_INT <= AV_VERSION_INT(52, 123, 0) // FFmpeg 0.7.13
      //
      static bool sample_fmt_for_sound_file_config(const Sound_file_config& config,
                                                   enum SampleFormat& return_sample_format);
      //
#else
      //
      static bool sample_fmt_for_sound_file_config(const Sound_file_config& config,
                                                   enum AVSampleFormat& return_sample_format);
      //
#endif

#if LIBAVFORMAT_VERSION_INT < AV_VERSION_INT(52, 64, 2) // FFmpeg 0.6
      //
      static AVOutputFormat* guess_format(const char* short_name,
                                          const char* file_name,
                                          const char* mime_type)
      {
        return ::guess_format(short_name,
                              file_name,
                              mime_type);
      }
      //
#else
      //
      static AVOutputFormat *guess_format(const char* short_name,
                                          const char* file_name,
                                          const char* mime_type)
      {
        return ::av_guess_format(short_name,
                              file_name,
                              mime_type);
      }
      //
#endif

#if LIBAVCODEC_VERSION_INT < AV_VERSION_INT(53, 8, 0) // FFmpeg 0.8.7
      //
      static int codec_open(AVCodecContext *context, AVCodec *codec)
      {
        return ::avcodec_open(context, codec);
      }
      //
#else
      //
      static int codec_open(AVCodecContext *context, AVCodec *codec)
      {
        return ::avcodec_open2(context,
                               codec,
                               0);
      }
      //
#endif

#if LIBAVFORMAT_VERSION_INT < AV_VERSION_INT(52, 104, 0) // FFmpeg 0.6.90-rc0
      //
      static int io_open(ByteIOContext** context,
                         const char* URL,
                         Open_flags flags)
      {
        return ::url_fopen(context,
                           URL,
                           flags);
      }
      static int io_close(ByteIOContext* context)
      {
        return ::url_fclose(context);
      }
      //
#else
      static int io_open(AVIOContext** context,
                         const char* file_name,
                           Open_flags flags)
      {
        return ::avio_open(context,
                           file_name,
                           flags);
      }
      static int io_close(AVIOContext* context)
      {
        return ::avio_close(context);
      }
      //
#endif

#if LIBAVFORMAT_VERSION_INT < AV_VERSION_INT(52, 110, 0) // FFmpeg 0.7
      //
      static int write_header(AVFormatContext* context)
      {
        return ::av_write_header(context);
      }
      //
#else
      //
      static int write_header(AVFormatContext* s)
      {
        return ::avformat_write_header(s, 0);
      }
      //
#endif

#if LIBAVCODEC_VERSION_INT < AV_VERSION_INT(53, 42, 0) // FFmpeg 0.9
      static int64_t default_channel_layout(int number_channels)
      {
        return 0;
      }
#elif LIBAVUTIL_VERSION_INT < AV_VERSION_INT(51, 32, 0) // FFmpeg 0.9
      static int64_t default_channel_layout(int number_channels)
      {
        return 0;
      }
#else
      static int64_t default_channel_layout(int number_channels)
      {
        return ::av_get_default_channel_layout(number_channels);
      }
#endif

#if LIBAVFORMAT_VERSION_INT < AV_VERSION_INT(53, 24, 0) // FFmpeg 0.9
      //
      static void zero_dts(AVFormatContext* context, AVStream* ref_stream)
      {
        return ::av_update_cur_dts(context,
                                   ref_stream,
                                   0);
      }
      static AVStream* new_stream(AVFormatContext *context, int track_ID)
      {
        return ::av_new_stream(context, track_ID);
      }
      //
#else
      //
      static void zero_dts(AVFormatContext* context, AVStream* ref_stream);
      static AVStream* new_stream(AVFormatContext* context, int track_ID)
      {
        AVStream* new_stream = ::avformat_new_stream(context, 0);
        new_stream->id = track_ID;
        return new_stream;
      }
      //
#endif

      // encode_video returns false and calls av_free_packet on error
      // it updates the packet size field with the number of bytes output to
      // the packet data field
#if LIBAVCODEC_VERSION_INT < AV_VERSION_INT(54, 23, 100) // FFmpeg 0.11
      // the caller must allocate an output buffer for avcodec_encode_video
      static bool need_encoder_output_buffer() { return true; }
      static bool encode_video(AVCodecContext* avctx,
                               AVPacket* pkt,
                               const AVFrame* pict,
                               bool& return_got_packet)
      {
        const int encode_return = ::avcodec_encode_video(avctx,
                                                         pkt->data,
                                                         pkt->size,
                                                         pict);
        pkt->size = encode_return;
        return_got_packet = (encode_return > 0);

        const bool return_value = (encode_return >= 0);
        if (!return_value)
        {
          av_free_packet(pkt);
        }

        return return_value;
      }
#else
      // avcodec_encode_video2 allocates an output buffer when the packet data field is null
      // so the caller does not necessarily need to allocate one
      static bool need_encoder_output_buffer() { return false; }
      static bool encode_video(AVCodecContext* avctx,
                               AVPacket* pkt,
                               const AVFrame* pict,
                               bool& return_got_packet)
      {
        int got_packet;
        const int encode_return = ::avcodec_encode_video2(avctx,
                                                          pkt,
                                                          pict,
                                                          &got_packet);
        return_got_packet = got_packet;

        return (encode_return == 0);
      }
#endif

      // class constants

#if LIBAVCODEC_VERSION_INT <= AV_VERSION_INT(52, 123, 0) // FFmpeg 0.7.13
      //
      static const enum CodecType codec_type_video = CODEC_TYPE_VIDEO;
      static const enum CodecType codec_type_audio = CODEC_TYPE_AUDIO;
      static const int pkt_flag_key = PKT_FLAG_KEY;
      //
#else
      //
      static const enum AVMediaType codec_type_video = AVMEDIA_TYPE_VIDEO;
      static const enum AVMediaType codec_type_audio = AVMEDIA_TYPE_AUDIO;
      static const int pkt_flag_key = AV_PKT_FLAG_KEY;
      //
#endif

#if LIBAVCODEC_VERSION_INT <= AV_VERSION_INT(52, 123, 0) // FFmpeg 0.7.13
      //
      static const enum SampleFormat sample_fmt_U8 = SAMPLE_FMT_U8;
      static const enum SampleFormat sample_fmt_S16 = SAMPLE_FMT_S16;
      static const enum SampleFormat sample_fmt_S32 = SAMPLE_FMT_S32;
      static const enum SampleFormat sample_fmt_FLT = SAMPLE_FMT_FLT;
      static const enum SampleFormat sample_fmt_DBL = SAMPLE_FMT_DBL;
      //
      static const int picture_type_I = FF_I_TYPE; // intra frame
      static const int picture_type_P = FF_P_TYPE; // predicted frame
      //
#else
      //
      static const enum AVSampleFormat sample_fmt_U8 = AV_SAMPLE_FMT_U8;
      static const enum AVSampleFormat sample_fmt_S16 = AV_SAMPLE_FMT_S16;
      static const enum AVSampleFormat sample_fmt_S32 = AV_SAMPLE_FMT_S32;
      static const enum AVSampleFormat sample_fmt_FLT = AV_SAMPLE_FMT_FLT;
      static const enum AVSampleFormat sample_fmt_DBL = AV_SAMPLE_FMT_DBL;
      //
      static const enum AVPictureType picture_type_I = AV_PICTURE_TYPE_I; // intra frame
      static const enum AVPictureType picture_type_P = AV_PICTURE_TYPE_P; // predicted frame
      //
#endif

      // for consistency, use the URL_* style flags until libavformat 53.4.0 (FFmpeg 0.8),
      // when the AVIO_FLAG_* style read/write flags were introduced--even though the avio_open and
      // avio_close calls were introduced in libavformat 52.104.0 (FFmpeg 0.6.90-rc0)
#if LIBAVFORMAT_VERSION_INT < AV_VERSION_INT(53, 4, 0) // FFmpeg 0.8
      //
      static const Open_flags read_only = URL_RDONLY;
      static const Open_flags write_only = URL_WRONLY;
      static const Open_flags read_write = URL_RDWR;
      //
#else
      //
      static const Open_flags read_only = AVIO_FLAG_READ;
      static const Open_flags write_only = AVIO_FLAG_WRITE;
      static const Open_flags read_write = AVIO_FLAG_READ_WRITE;
      //
#endif

#if LIBAVFORMAT_VERSION_INT <= AV_VERSION_INT(54, 29, 104) // FFmpeg 1.0.2
      static bool need_high_resolution_video_timescale() { return false; }
#else
      // timescale has to be at least .1 ms (10000 timescale units)
      static bool need_high_resolution_video_timescale() { return true; }
#endif

#if LIBAVCODEC_VERSION_INT < AV_VERSION_INT(54, 59, 100) // FFmpeg 1.0
      static const enum CodecID codec_ID_PCM_S8 = CODEC_ID_PCM_S8;
      static const enum CodecID codec_ID_PCM_S16LE = CODEC_ID_PCM_S16LE;
      static const enum CodecID codec_ID_PCM_S16BE = CODEC_ID_PCM_S16BE;
      static const enum CodecID codec_ID_PCM_S24LE = CODEC_ID_PCM_S24LE;
      static const enum CodecID codec_ID_PCM_S24BE = CODEC_ID_PCM_S24BE;
      static const enum CodecID codec_ID_PCM_S32LE = CODEC_ID_PCM_S32LE;
      static const enum CodecID codec_ID_PCM_S32BE = CODEC_ID_PCM_S32BE;
      static const enum CodecID codec_ID_PCM_F32LE = CODEC_ID_PCM_F32LE;
      static const enum CodecID codec_ID_PCM_F32BE = CODEC_ID_PCM_F32BE;
      static const enum CodecID codec_ID_PCM_F64LE = CODEC_ID_PCM_F64LE;
      static const enum CodecID codec_ID_PCM_F64BE = CODEC_ID_PCM_F64BE;
      //
      static const enum CodecID codec_ID_QTRLE = CODEC_ID_QTRLE;
#else
      static const enum AVCodecID codec_ID_PCM_S8 = AV_CODEC_ID_PCM_S8;
      static const enum AVCodecID codec_ID_PCM_S16LE = AV_CODEC_ID_PCM_S16LE;
      static const enum AVCodecID codec_ID_PCM_S16BE = AV_CODEC_ID_PCM_S16BE;
      static const enum AVCodecID codec_ID_PCM_S24LE = AV_CODEC_ID_PCM_S24LE;
      static const enum AVCodecID codec_ID_PCM_S24BE = AV_CODEC_ID_PCM_S24BE;
      static const enum AVCodecID codec_ID_PCM_S32LE = AV_CODEC_ID_PCM_S32LE;
      static const enum AVCodecID codec_ID_PCM_S32BE = AV_CODEC_ID_PCM_S32BE;
      static const enum AVCodecID codec_ID_PCM_F32LE = AV_CODEC_ID_PCM_F32LE;
      static const enum AVCodecID codec_ID_PCM_F32BE = AV_CODEC_ID_PCM_F32BE;
      static const enum AVCodecID codec_ID_PCM_F64LE = AV_CODEC_ID_PCM_F64LE;
      static const enum AVCodecID codec_ID_PCM_F64BE = AV_CODEC_ID_PCM_F64BE;
      //
      static const enum AVCodecID codec_ID_QTRLE = AV_CODEC_ID_QTRLE;
#endif

#if LIBAVUTIL_VERSION_INT < AV_VERSION_INT(52, 13, 100) // FFmpeg 1.1
      static const enum PixelFormat pixel_format_RGB24 = PIX_FMT_RGB24;
#else
      static const enum AVPixelFormat pixel_format_RGB24 = AV_PIX_FMT_RGB24;
#endif

    private:
      // prevent compiler from generating
      AV_compat(const AV_compat& c);
      AV_compat& operator=(const AV_compat& c);
    };
  }
}

#endif // ROAN_TRAIL_KINETOPHONE_AV_COMPAT_HPP_
