// Kinetophone_dbus_recorder.hpp
//
// Copyright 2011-2013 Roan Trail, Inc.
//
// This file is part of Kinetophone.
//
// Kinetophone is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published
// by the Free Software Foundation, either version 2 of the License,
// or (at your option) any later version.
//
// Kinetophone is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// General Public License for more details.  You should have received
// a copy of the GNU General Public License along with Kinetophone. If
// not, see <http://www.gnu.org/licenses/>.

#ifndef ROAN_TRAIL_KINETOPHONE_APP_KINETOPHONE_DBUS_RECORDER_HPP_
#define ROAN_TRAIL_KINETOPHONE_APP_KINETOPHONE_DBUS_RECORDER_HPP_

#include <dbus-c++/dbus.h>
#include "Kinetophone_dbus_server_glue.hpp"
#include <kinetophone/common.hpp>
#include <kinetophone/Narration_recorder.hpp>
#include <string>
#include <vector>
#include <map>

namespace Roan_trail
{
  namespace Kinetophone_app
  {
    typedef std::map<std::string, std::string> Error_dictionary;

    class Kinetophone_dbus_recorder
      : public com::roantrail::dbus::kinetophone::recorder_adaptor,
        public DBus::IntrospectableAdaptor,
        public DBus::ObjectAdaptor,
        public Roan_trail::Kinetophone::Narration_recorder
    {
    public:
      // constructor, destructor
      Kinetophone_dbus_recorder(DBus::Connection& connection, const std::string& path);
      // control
      void remove(bool& return_have_error, Error_dictionary& return_error);
      // helpers
      static const std::string& dbus_name();
      static std::string dbus_path(Roan_trail::Kinetophone::Long_int ID);
      //
      // DBUS interface
      //   control
      //     startup/shutdown functions
      void startup(const std::map<std::string, std::string>& settings,
                   bool& return_have_error,
                   Error_dictionary& return_error);
      //     basic Recorder functions
      void record(bool& return_have_error,
                  Error_dictionary& return_error);
      void pause(bool& return_have_error, Error_dictionary& return_error); // not a toggle
      void stop(bool& return_have_error, Error_dictionary& return_error);
      //     narration/sound recorder functions
      void record_with_index(const int64_t& index,
                             bool& return_have_error,
                             Error_dictionary& return_error);
      void new_index(const int64_t& index,
                     bool& return_have_error,
                     Error_dictionary& return_error);
      void retake(bool& return_have_error,
                  Error_dictionary& return_error);
      void mute(bool& return_have_error, Error_dictionary& return_error); // toggles muting
      //   accessors
      bool is_muted();
      bool is_started();
      bool can_record();
      std::string recorded_time(const uint8_t& separator);
      std::string recorded_time_for_current_index(const uint8_t& separator);
      std::vector<double> sound_levels_RMS();
      std::vector<double> sound_levels_peak();
      void frames(int64_t& return_total_frames,
                  int64_t& return_index_frames,
                  bool &return_have_error,
                  Error_dictionary& return_error);
      void segments(std::string& return_segments_XML,
                    bool& return_have_error,
                    Error_dictionary& return_error);
      void settings(std::map<std::string, std::string>& return_settings,
                    bool& return_have_error,
                    Error_dictionary& return_error);
      std::string output_file();
      bool have_recording();
      void record_error(bool& return_have_record_error, Error_dictionary& return_error);
      int64_t overflow_count();
      int64_t input_overflow_count();
      int64_t output_overflow_count();
      void space_available(int64_t& return_bytes_available,
                           double& return_fraction_available,
                           double& return_seconds_remaining,
                           bool& return_have_error);
      void output_file_size(int64_t& return_file_size,
                            double& return_data_rate,
                            bool& return_have_error);
      bool is_RMS_metering_enabled();
      bool is_peak_metering_enabled();
      int metering_channels();
      void values_for_update(std::string& return_recorded_time,
                             std::string& return_recorded_time_for_current_index,
                             std::vector<double>& return_sound_levels,
                             int64_t& return_overflow_count,
                             bool& return_have_record_error,
                             Error_dictionary& return_record_error);
      //   mutators
      void reset_overflow_counters();
      void enable_RMS_metering(const bool& enable);
      void enable_peak_metering(const bool& enable);
      void set_metering_channels(const int& channel_count);
      void cleanup_temporary_file();
      void clear_record_error();
      //   commands
      void list_devices(std::string& return_devices,
                        bool& return_have_error,
                        Error_dictionary& return_error);
    protected:
      bool mf_invariant(bool check_base_class = true) const;
    private:
      int m_channels;
      std::vector<double> m_sound_levels;
      // prevent compiler from generating
      Kinetophone_dbus_recorder(const Kinetophone_dbus_recorder& recorder);
      Kinetophone_dbus_recorder& operator=(const Kinetophone_dbus_recorder& recorder);
    };
  }
}

#endif // ROAN_TRAIL_KINETOPHONE_APP_KINETOPHONE_DBUS_RECORDER_HPP_
