// Kinetophone_test_app.cpp
//
// Copyright 2011-2013 Roan Trail, Inc.
//
// This file is part of Kinetophone.
//
// Kinetophone is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published
// by the Free Software Foundation, either version 2 of the License,
// or (at your option) any later version.
//
// Kinetophone is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// General Public License for more details.  You should have received
// a copy of the GNU General Public License along with Kinetophone. If
// not, see <http://www.gnu.org/licenses/>.

// Minimal application for automated/manual testing

#include <kinetophone/common.hpp>
#include "Console_app.hpp"
#include <kinetophone/Sound_recorder.hpp>
#include <kinetophone/Sound_recorder_config.hpp>
#include <kinetophone/Sound_file_config.hpp>
#include <kinetophone/error/Kinetophone_error.hpp>
#include <iostream>
#include <string>
#include <vector>
#include <cstddef>
#include <portaudio.h>

using std::cout;
using std::cerr;
using std::endl;
using std::string;
using std::vector;

using Roan_trail::Kinetophone::Long_int;
using Roan_trail::Kinetophone::Sound_recorder;
using Roan_trail::Kinetophone::Sound_recorder_config;
using Roan_trail::Kinetophone::Error;
using Roan_trail::Kinetophone::Error_param;
using Roan_trail::Kinetophone::Kinetophone_error;
using namespace Roan_trail::Kinetophone_app;

class Kinetophone_test_app : public Console_app
{
public:
  Kinetophone_test_app(int argc, const char** argv)
    : Console_app(argc, argv) {}
  virtual int run() { return 0; }
};

int main(int argc, const char** argv)
{
  static_cast<void>(argc); // avoid unused warning

  int return_value = 0;

  Kinetophone_test_app application(argc, argv);

  Sound_recorder sr;
  Error_param error;
  bool started = false;

  start_error_block();

  cout << "Starting recorder" << endl;

  Sound_recorder_config config;

  // fill in the details here (override defaults)
  config.input_device = 3;
  config.sound_file->channels = 1;
  vector<double> levels(static_cast<size_t>(config.sound_file->channels));

  sr.set_metering_channels(config.sound_file->channels);
  sr.enable_RMS_metering(true);

  started = sr.startup(config, error);
  on_error(!started, new Kinetophone_error(error_location(),
                                           Kinetophone_error::startup,
                                           error()));
  cout << "Recording for 1 sec" << endl;

  const bool recording = sr.record(error);
  on_error(!recording, new Kinetophone_error(error_location(),
                                             Kinetophone_error::record,
                                             error()));

  const Long_int sleep_msec = 1000 / 30;
  for (int i = 0; i < 30; ++i)
  {
    // do something
    // ...
    sr.sound_levels_RMS(levels);
    cout << "Level: " << levels[0] << endl;
    Pa_Sleep(sleep_msec);
  }

  cout << "Stopping recorder" << endl;

  const bool stopped = sr.stop(error);
  on_error(!stopped, new Kinetophone_error(error_location(),
                                           Kinetophone_error::record,
                                           error()));

  cout << "Shutting down recorder" << endl;
  const bool shutdown = sr.shutdown(error);
  on_error(!shutdown, new Kinetophone_error(error_location(),
                                            Kinetophone_error::shutdown,
                                            error()));
  cout << "Test complete" << endl;

  goto exit_point;

  end_error_block();

  {
  error_handler:
    const Error* user_error = handler_error->user_error_for_chain();
    assert(user_error && "no user error found");

    const Error::Error_dictionary& error_dict = user_error->error_dictionary();
    Error::Error_dictionary::const_iterator i;
    i = error_dict.find(Error::brief_description_error_key);
    if (error_dict.end() != i)
    {
      string brief_description = i->second;
      if (brief_description != "")
      {
        cerr << brief_description << endl;
      }
    }
    i = error_dict.find(Error::detailed_description_error_key);
    if (error_dict.end() != i)
    {
      string detailed_description = i->second;
      if (detailed_description != "")
      {
        cerr << detailed_description << endl;
      }
    }
    i = error_dict.find(Error::recovery_description_error_key);
    if (error_dict.end() != i)
    {
      string recovery_description = i->second;
      if (recovery_description != "")
      {
        cerr << recovery_description << endl;
      }
    }

    cerr << "Detailed error:" << endl;
    cerr << *user_error;
    goto error_cleanup;

  error_cleanup:
    delete handler_error;
    if (started)
    {
      Error_param e;
      sr.shutdown(e); // ignore error
    }
    const int error_code = user_error->code();
    return_value = error_code;
    goto exit_point;
  }

 exit_point:
  return return_value;
}
