// Kinetophone_console_view.cpp
//
// Copyright 2011-2013 Roan Trail, Inc.
//
// This file is part of Kinetophone.
//
// Kinetophone is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published
// by the Free Software Foundation, either version 2 of the License,
// or (at your option) any later version.
//
// Kinetophone is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// General Public License for more details.  You should have received
// a copy of the GNU General Public License along with Kinetophone. If
// not, see <http://www.gnu.org/licenses/>.

#include "Kinetophone_console_view.hpp"
#include <kinetophone/Model.hpp>
#include <kinetophone/Kinetophone_model.hpp>
#include <kinetophone/Sound_recorder_config.hpp>
#include <kinetophone/Sound_file_config.hpp>
#include <kinetophone/Sound_recorder.hpp>
#include "Kinetophone_console_config.hpp"
#include "Console_level_meter_view.hpp"
#include <kinetophone/Device_enum.hpp>
#include <kinetophone/error/Kinetophone_error.hpp>
#include <kinetophone/error/Curses_error.hpp>
#include <iomanip>
#include <sstream>
#include <string>
#include <ncurses.h>
#include <cmath>
#include <cstddef>

using std::cout;
using std::cerr;
using std::endl;
using std::ios_base;
using std::setw;
using std::setfill;
using std::stringstream;
using std::string;
using Roan_trail::Kinetophone::Device_enum;
using Roan_trail::Kinetophone::Kinetophone_model;
using Roan_trail::Kinetophone::Long_int;
using Roan_trail::Kinetophone::min;
using Roan_trail::Kinetophone::max;
using Roan_trail::Kinetophone::min;
using Roan_trail::Kinetophone::Sound_recorder;
using Roan_trail::Kinetophone::Sound_recorder_config;
using Roan_trail::Kinetophone::Curses_error;
using Roan_trail::Kinetophone::Error_param;
using Roan_trail::Kinetophone::Kinetophone_error;
using namespace Roan_trail::Kinetophone_app;

//
// Helper
//

namespace
{
  string ic_newline;
}

//
// Constructor/destructor/copy
//

Kinetophone_console_view::Kinetophone_console_view(const Kinetophone_model& model,
                                                   const Kinetophone_console_config& config)
  : View(),
    m_model(model),
    m_config(config),
    m_level_meter_view(0),
    m_console_width(m_default_console_width),
    m_console_height(m_default_console_height),
    m_display_string_cache(),
    m_file_size_string_cache(),
    m_available_string_cache(),
    m_audio_info_string_cache(),
    m_recording_file(),
    m_level(),
    m_confirm_overwrite(false),
    m_confirm_stop(false)
{
  precondition(m_config.sound_recorder_config->sound_file->channels > 0);

  const int channels = m_config.sound_recorder_config->sound_file->channels;
  const bool print_level = m_config.level_meter.has_level;
  if (config.has_level_meter)
  {
    m_level_meter_view = new Console_level_meter_view(channels,
                                                      m_config.update_rate,
                                                      m_console_width,
                                                      print_level,
                                                      m_config.level_meter.model_ballistics,
                                                      m_config.level_meter.attack_period,
                                                      m_config.level_meter.decay_period,
                                                      m_config.level_meter.peak_hold_period);
    m_level.resize(static_cast<size_t>(channels));
    for (int channel = 0; channel < channels; ++channel)
    {
      m_level[static_cast<size_t>(channel)] = 0.0;
    }
  }
  m_recording_file = config.sound_recorder_config->sound_file->file_name;
  mf_format_audio_info_string();

  if ("" == ic_newline)
  {
    stringstream s;
    s << endl;
    ic_newline = s.str();
  }

  postcondition(mf_invariant(false));
}

Kinetophone_console_view::~Kinetophone_console_view()
{
  precondition(mf_invariant(false));

  delete m_level_meter_view;
}

//
// Accessors
//

bool Kinetophone_console_view::level_meter_peak_mode() const
{
  precondition(mf_invariant());
  return (m_level_meter_view && m_level_meter_view->peak_meter_mode());
}

//
// Mutators
//

void Kinetophone_console_view::resize()
{
  precondition(mf_invariant());

  assert(stdscr && "Error, stdscr (ncurses standard window) is null");

  m_console_height = getmaxy(stdscr);
  m_console_width = max(m_minimum_console_width, getmaxx(stdscr));

  if (m_level_meter_view)
  {
    if (m_config.level_meter.use_max_width)
    {
      m_level_meter_view->set_width(m_console_width - 2);
    }
    else
    {
      m_level_meter_view->set_width(min(m_config.level_meter.width, m_console_width - 2));
    }
  }

  postcondition(mf_invariant());
}

void Kinetophone_console_view::set_level_meter_peak_mode(bool peak_mode)
{
  precondition(mf_invariant());

  if (m_level_meter_view)
  {
    m_level_meter_view->set_peak_meter_mode(peak_mode);
  }

  postcondition(mf_invariant());
}

//
// View update
//

bool Kinetophone_console_view::update(Error_param& return_error)
{
  precondition(!return_error()
               && mf_invariant());

  bool return_value = false;

  start_error_block();

  stringstream s;

  s << "Kinetophone Console Recorder" << endl;
  // audio information
  if ("" != m_audio_info_string_cache)
  {
    s << endl << m_audio_info_string_cache;
  }
  s << endl;

  // print meter(s) if needed
  if (m_level_meter_view)
  {
    if (m_level_meter_view->peak_meter_mode())
    {
      m_model.sound_recorder()->sound_levels_peak(m_level);
    }
    else
    {
      m_model.sound_recorder()->sound_levels_RMS(m_level);
    }

    m_level_meter_view->set_levels(m_level);
    Error_param e(false);
    m_level_meter_view->update(e); // ignore error
    s << m_level_meter_view->scale_label() << endl;
    s << m_level_meter_view->scale() << endl;
    s << m_level_meter_view->meter() << endl;
  }

  // print output file
  s << "Output file: " << m_recording_file << endl;
  if (m_config.show_file_size)
  {
    s << mf_format_file_size_string() << endl;
  }
  if (m_config.show_available)
  {
    s << mf_format_available_string() << endl;
  }

  // print control panel
  s << mf_format_display_string();

  const int height = getmaxy(stdscr);
  const int width = max(m_minimum_console_width, getmaxx(stdscr));
  if ((height != m_console_height) || (width != m_console_width))
  {
    resize();
    erase();
  }

  move(0,0); // (ncurses) reset screen position, ignore return code

  printw(s.str().c_str()); // ignore return code

  int code = refresh(); // (ncurses) update screen
  on_error(ERR == code, new Kinetophone_error(error_location(),
                                              Kinetophone_error::console,
                                              new Curses_error("refresh")));
  return_value = true;
  goto exit_point;

  end_error_block();

  default_error_handler_and_cleanup(return_error,
                                    return_value,
                                    false);
  goto exit_point;

 exit_point:
  postcondition(return_error.is_valid_at_return(return_value)
                && mf_invariant());
  return return_value;
}

//
// Commands
//

void Kinetophone_console_view::clear()
{
  precondition(mf_invariant());

  if (m_level_meter_view)
  {
    m_level_meter_view->reset_clipped();
  }

  postcondition(mf_invariant());
}

void Kinetophone_console_view::output(const string& message)
{
  cout << message;
}

void Kinetophone_console_view::output_error(const string& message)
{
  cerr << message;
}

bool Kinetophone_console_view::list_devices(Error_param& return_error)
{
  precondition(!return_error());

  bool return_value = false;

  start_error_block();

  string device_info_string;
  Error_param error;
  bool got_info = Device_enum::list_devices(paFloat32, // Note: paFloat32 declaration uses an old-style cast
                                            device_info_string,
                                            error);
  on_error(!got_info, new Kinetophone_error(error_location(),
                                            Kinetophone_error::list_devices,
                                            error()));

  cout << device_info_string;

  return_value = true;
  goto exit_point;

  end_error_block();

  default_error_handler_and_cleanup(return_error,
                                    return_value,
                                    false);
  goto exit_point;

 exit_point:
  return return_value;
}

//
// Protected member functions
//

bool Kinetophone_console_view::mf_invariant(bool check_base_class) const
{
  static_cast<void>(check_base_class); // avoid unused warning

  bool return_value = false;

  // range checking
  if ((m_console_height < 0)
      || (m_console_width < m_minimum_console_width))
  {
    goto exit_point;
  }
  // object consistency checking
  else if ((m_config.has_level_meter && !m_level_meter_view)
           || (!m_config.has_level_meter && m_level_meter_view))
  {
    goto exit_point;
  }
  else if ("" == m_recording_file)
  {
    goto exit_point;
  }
  else
  {
    return_value = true;
  }

 exit_point:
  return return_value;
}

//
// Private static constants
//

const int Kinetophone_console_view::m_default_console_width = 50;
const int Kinetophone_console_view::m_default_console_height = 25;
const int Kinetophone_console_view::m_minimum_console_width = 20;

//
// Private member functions
//

const string& Kinetophone_console_view::mf_format_display_string() const
{
  const Sound_recorder *sound_recorder = m_model.sound_recorder();

  string recorded_time;
  sound_recorder->recorded_time(recorded_time);

  stringstream display_string;

  // overflow count, if needed
  if (m_config.show_overflow_count)
  {
    Long_int input_overflow_count = sound_recorder->input_overflow_count();
    display_string << endl << "Overflow counts -- ";
    display_string << "input: " << setw(2) << setfill('0') << input_overflow_count % 100;
    if (input_overflow_count > 100)
    {
      display_string << "+ ";
    }
    else
    {
      display_string << "  ";
    }
    Long_int output_overflow_count = sound_recorder->output_overflow_count();
    display_string <<  "/ output: " << setw(2) << setfill('0') << output_overflow_count % 100;
    if (output_overflow_count > 100)
    {
      display_string << "+";
    }
    else
    {
      display_string << " ";
    }
  }
  display_string << endl;

  // time
  display_string << "Rec time: " << recorded_time << " ";

  // recorder state and commands
  const string level = m_level_meter_view ? "| (L)evel " : ""; // only needed if there is a level meter
  const int state = sound_recorder->state();
  string muted = (sound_recorder->is_muted() ? " *Muted*" : "        ");
  if (Roan_trail::Kinetophone::Recorder::state_stopped == state)
  {
    display_string << "[STOP ]" << muted << endl << endl;
    if (!m_confirm_overwrite && !m_confirm_stop)
    {
      display_string << "Cmd: [(Q)uit | (C)lear | (M)ute " << level << "|| ------ | ------- | (R)ecord]";
    }
  }
  else if (Roan_trail::Kinetophone::Recorder::state_paused == state)
  {
    display_string << "[PAUSE]" << muted << endl << endl;
    if (!m_confirm_overwrite && !m_confirm_stop)
    {
      display_string << "Cmd: [(Q)uit | (C)lear | (M)ute " << level << "|| (S)top | ------- | (R)ecord]";
    }
  }
  else if (Roan_trail::Kinetophone::Recorder::state_recording == state)
  {
    display_string << "[REC  ]" << muted << endl << endl;
    if (!m_confirm_overwrite && !m_confirm_stop)
    {
      display_string << "Cmd: [(Q)uit | (C)lear | (M)ute " << level << "|| (S)top | (P)ause | --------]";
    }
  }
  else
  {
    assert(false && "invalid recorder state");
  }

  if (m_confirm_overwrite)
  {
    display_string <<   "Overwrite? [Y | N]         ";
  }
  else if (m_confirm_stop)
  {
    display_string <<   "Stop? [Y | N]              ";
  }
  // No final else needed, handled confirm options

  display_string << endl;

  m_display_string_cache = display_string.str();

  return m_display_string_cache;
}

const string& Kinetophone_console_view::mf_format_file_size_string() const
{
  Long_int file_size = 0;
  double data_rate = 0.0;

  const Sound_recorder* sound_recorder = m_model.sound_recorder();
  sound_recorder->output_file_size(file_size, data_rate);
  stringstream file_size_stream;

  file_size_stream << "Output file size: ";
  file_size_stream.setf(ios_base::fixed, ios_base::floatfield);
  if (file_size < 1000L)
  {
    file_size_stream.precision(0);
    file_size_stream << file_size << " Bytes";
  }
  else if (file_size < 1000000L)
  {
    file_size_stream.precision(0);
    file_size_stream << static_cast<double>(file_size) / 1000.0 << " KB";
  }
  else if (file_size < 1000000000L)
  {
    file_size_stream.precision(2);
    file_size_stream << static_cast<double>(file_size) / 1000000.0 << " MB";
  }
  else
  {
    file_size_stream.precision(3);
    file_size_stream << static_cast<double>(file_size) / 1000000000.0 << " GB";
  }

  file_size_stream.precision(2);
  file_size_stream << " (" << data_rate / 1000.0 << " KB/sec)";

  m_file_size_string_cache = file_size_stream.str();

  return m_file_size_string_cache;
}

const string& Kinetophone_console_view::mf_format_available_string() const
{

  Long_int bytes_available = 0;
  double fraction_available = 0.0;
  double seconds_remaining = 0.0;
  const Sound_recorder* sound_recorder = m_model.sound_recorder();
  sound_recorder->space_available(bytes_available,
                                  fraction_available,
                                  seconds_remaining);

  // make 1MB reserve
  if (bytes_available > 1000000)
  {
    bytes_available -= 1000000;
  }
  else
  {
    bytes_available = 0;
  }

  stringstream available;
  available.setf(ios_base::fixed, ios_base::floatfield);
  available.precision(2);
  available << "Available space: ";
  if (bytes_available < 1000000000L)
  {
    available << static_cast<double>(bytes_available) / 1000000.0 << " MB";
  }
  else
  {
    available << static_cast<double>(bytes_available) / 1000000000.0 << " GB";
  }

  available.precision(0);
  available << " (";
  if (seconds_remaining < 60.0)
  {
    available << seconds_remaining << " sec)";
  }
  else if (seconds_remaining < 3600.0)
  {
    double seconds = fmod(round(seconds_remaining), 60.0);
    available << (seconds_remaining - seconds) / 60.0 << " min : " << seconds << " sec)";
  }
  else
  {
    available << seconds_remaining / 3600 << " hr ";
    available << fmod(seconds_remaining, 3600.0) / 60.0 << " min)";
  }

  m_available_string_cache = available.str();

  return m_available_string_cache;
}

const string& Kinetophone_console_view::mf_format_audio_info_string() const
{
  if (m_config.show_audio_info)
  {
    const Sound_recorder_config* config = m_config.sound_recorder_config;
    stringstream s;
    s << "| ";
    s << "DEV " << config->input_device << " | ";
    s << config->sound_file->format_string_brief() << " | ";
    double VBR_quality;
    if (config->sound_file->has_VBR(VBR_quality))
    {
      s << "VBRQ=" << VBR_quality << " | ";
    }
    s << "FPB=" << config->frames_per_buffer << " | ";
    s << "WBF=" << config->write_buffer_factor << " | ";
    s << (config->file_overwrite ? "OVWRT" : "NO OVWRT");
    s << " |" << endl;
    m_audio_info_string_cache = s.str();
  }
  else
  {
    m_audio_info_string_cache = "";
  }

  return m_audio_info_string_cache;
}
