// Kinetophone_console_app.cpp
//
// Copyright 2011-2013 Roan Trail, Inc.
//
// This file is part of Kinetophone.
//
// Kinetophone is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published
// by the Free Software Foundation, either version 2 of the License,
// or (at your option) any later version.
//
// Kinetophone is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// General Public License for more details.  You should have received
// a copy of the GNU General Public License along with Kinetophone. If
// not, see <http://www.gnu.org/licenses/>.

#include "Kinetophone_console_app.hpp"
#include <kinetophone/Kinetophone_model.hpp>
#include <kinetophone/File_manager.hpp>
#include <kinetophone/Sound_recorder.hpp>
#include <kinetophone/Sound_recorder_config.hpp>
#include <kinetophone/Sound_file_config.hpp>
#include "Kinetophone_console_view.hpp"
#include "Kinetophone_console_controller.hpp"
#include "Kinetophone_console_config.hpp"
#include <kinetophone/error/Kinetophone_error.hpp>
#include <portaudio.h>
#include <sstream>
#include <string>
#include <csignal> // Standards Rule 21 deviation (check_code_ignore)

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#ifndef PACKAGE_VERSION
#define PACKAGE_VERSION "(unknown)"
#endif

using std::endl;
using std::string;
using std::stringstream;
using Roan_trail::Kinetophone::Kinetophone_model;
using Roan_trail::Kinetophone::Sound_recorder;
using Roan_trail::Kinetophone::Error;
using Roan_trail::Kinetophone::Error_param;
using Roan_trail::Kinetophone::Kinetophone_error;
using namespace Roan_trail::Kinetophone_app;

//
// Constructor/destructor/copy
//

Kinetophone_console_app::Kinetophone_console_app(int argc, const char** argv)
  : Console_app(argc, argv),
    m_config(new Kinetophone_console_config),
    m_model(new Kinetophone_model),
    m_controller(0),
    m_view(0)
{
  precondition(argv);

  postcondition(mf_invariant(false));
}

Kinetophone_console_app::~Kinetophone_console_app()
{
  precondition(mf_invariant(false));

  delete m_view;
  delete m_controller;
  delete m_model;
  delete m_config;
}

int Kinetophone_console_app::run()
{
  precondition(mf_invariant());

  int return_value = 1;
  Error_param error;
  int user_error_code = 0;

  start_error_block();

  // see what the user wants to do
  int recorder_command = m_config->parse_program_options(argc(), argv(), error);
  on_error(Kinetophone_console_config::command_error == recorder_command,
           new Kinetophone_error(error_location(),
                                 Kinetophone_error::command_line,
                                 error()));

  // update the installation directory if the program options override the default one
  if ("" != m_config->custom_installation_dir)
  {
    Application::application()->set_installation_dir(m_config->custom_installation_dir);
  }

  switch (recorder_command)
  {
  case Kinetophone_console_config::command_help:
    // HELP
    Kinetophone_console_view::output(m_config->help_message);
    break;
  case Kinetophone_console_config::command_list_devices:
    // LIST DEVICES
    {
      const bool listed_devices = Kinetophone_console_view::list_devices(error);
      on_error(!listed_devices, new Kinetophone_error(error_location(),
                                                      Kinetophone_error::general,
                                                      error()));
    }
    break;
  case Kinetophone_console_config::command_output_version:
    // VERSION
    {
      stringstream version;
      version << "Kinetophone Console Recorder Version " << string(PACKAGE_VERSION) << endl;
      Kinetophone_console_view::output(version.str());
    }
    break;
  case Kinetophone_console_config::command_record:
    // RECORD
    {
      Error_param record_error;
      bool record_success = mf_record(record_error);
      on_error(!record_success, new Kinetophone_error(error_location(),
                                                    Kinetophone_error::general,
                                                    record_error()));
    }
    break;
  default:
    assert(false && "Invalid command");
    break;
  }

  return_value =  0;
  goto exit_point;

  end_error_block();

 error_handler:
  {
    string error_message;
    user_error_code = handler_error->format_message_for_chain(m_config->detailed_error, error_message);
    Kinetophone_console_view::output_error(error_message);
  }
  goto error_cleanup;

 error_cleanup:
  delete handler_error;
  return_value = user_error_code;
  goto exit_point;

 exit_point:
  postcondition(mf_invariant());
  return return_value;
}

//
// Other
//

void Kinetophone_console_app::terminate(int code, const string& message)
{
  precondition(mf_invariant());

  Sound_recorder *recorder = m_model->sound_recorder();
  if (Roan_trail::Kinetophone::Recorder::state_stopped != recorder->state())
  {
    Error_param e(false);
    recorder->stop(e); // ignore error
  }

  Console_app::terminate(code, message);

  // no postcondition, above terminate() call does not return
}

//
// Protected member functions
//

bool Kinetophone_console_app::mf_invariant(bool check_base_class) const
{
  static_cast<void>(check_base_class); // avoid unused warning

  bool return_value = false;

  // object allocation checking
  if (!m_config
      || !m_model)
  {
    goto exit_point;
  }

  return_value = true;

 exit_point:
  return return_value;
}

//
// Private member functions
//

bool Kinetophone_console_app::mf_record(Error_param& return_error)
{
  precondition(!return_error());

  bool return_value = false;

  bool sound_recorder_started_up = false;
  Error_param error;
  Sound_recorder* sound_recorder = m_model->sound_recorder();

  start_error_block();

  // start the sound recorder subsystem
  sound_recorder_started_up = sound_recorder->startup(*(m_config->sound_recorder_config), error);
  on_error(!sound_recorder_started_up, new Kinetophone_error(error_location(),
                                                             Kinetophone_error::startup,
                                                             error()));
  // update the Kinetophone console config from the recorder
  // before creating the view
  *(m_config->sound_recorder_config) = *sound_recorder->config();
  string output_file = sound_recorder->output_file();
  m_config->sound_recorder_config->sound_file->file_name = output_file;
  delete m_view;
  m_view = new Kinetophone_console_view(*m_model, *m_config);
  delete m_controller;
  m_controller = new Kinetophone_console_controller(*m_model, *m_view);

  // Note: minimize heap allocation after this point (should be limited to error handling).
  // Ref. Standards Rule #206

  // start I/O loop
  bool full_screen_set = set_full_screen(true, error);
  on_error(!full_screen_set, new Kinetophone_error(error_location(),
                                                   Kinetophone_error::console,
                                                   error()));

  m_view->resize();
  bool IO_loop_success = mf_run_IO_loop(error);
  on_error(!IO_loop_success, new Kinetophone_error(error_location(),
                                                   Kinetophone_error::general,
                                                   error()));
  Error_param e(false);
  set_full_screen(false, e); // ignore error

  if (sound_recorder->have_recording())
  {
    stringstream file_message;
    file_message << "Output to file: " << output_file << endl;
    Kinetophone_console_view::output(file_message.str());

    string formatted_time;
    sound_recorder->recorded_time(formatted_time);
    stringstream t;
    t << "Total time recorded: " << formatted_time << " (H:M:S)" << endl;
    Kinetophone_console_view::output(t.str());
  }
  else
  {
    sound_recorder->cleanup_temporary_file(); // clean up droppings
  }

  // shut down the sound recorder subsystem
  bool sound_recorder_shutdown = sound_recorder->shutdown(error);
  sound_recorder_started_up = false;
  on_error(!sound_recorder_shutdown,new Kinetophone_error(error_location(),
                                                          Kinetophone_error::shutdown,
                                                          error()));

  return_value = true;
  goto exit_point;

  end_error_block();

  default_error_handler(return_error);

 error_cleanup:
  if (!return_error.need_error())
  {
    delete handler_error;
  }
  if (sound_recorder_started_up)
  {
    Error_param e(false);
    sound_recorder->shutdown(e); // ignore error
  }
  if (full_screen())
  {
    Error_param e(false);
    set_full_screen(false, e); // ignore error
  }
  return_value = false;
  goto exit_point;

 exit_point:
  return return_value;
}

bool Kinetophone_console_app::mf_run_IO_loop(Error_param& return_error)
{
  precondition(!return_error());

  bool return_value = false;

  start_error_block();

  const long sleep_msec = static_cast<long int>(1000.0 / m_config->update_rate); // approximate update interval

  bool done;
  Sound_recorder* sound_recorder = m_model->sound_recorder();
  while (true)
  {
    // check for an record error between updates
    const Error* record_error = sound_recorder->record_error();
    on_error(record_error, new Kinetophone_error(error_location(),
                                                 Kinetophone_error::record,
                                                 record_error));
    Error_param error;
    bool processed = m_controller->process_input(done, error);
    on_error(!processed, new Kinetophone_error(error_location(),
                                               Kinetophone_error::record,
                                               error()));
    Error_param update_error;
    bool have_update = m_view->update(update_error);
    on_error(!have_update, new Kinetophone_error(error_location(),
                                                 Kinetophone_error::general,
                                                 update_error()));
    if (done)
    { // do view update first
      // then quit if done
      break;
    }

    Pa_Sleep(sleep_msec);
  }

  // one last check for record error
  const Error* record_error = sound_recorder->record_error();
  on_error(record_error, new Kinetophone_error(error_location(),
                                               Kinetophone_error::record,
                                               record_error));

  return_value = true;
  goto exit_point;

  end_error_block();

  default_error_handler_and_cleanup(return_error,
                                    return_value,
                                    false);
  goto exit_point;

 exit_point:
  return return_value;
}

//////////
// main //
//////////

int main(int argc, const char** argv)
{
  Kinetophone_console_app application(argc, argv);

  int return_code = application.run();

  return return_code;
}
