// Kinetophone_builder_config.cpp
//
// Copyright 2011-2013 Roan Trail, Inc.
//
// This file is part of Kinetophone.
//
// Kinetophone is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published
// by the Free Software Foundation, either version 2 of the License,
// or (at your option) any later version.
//
// Kinetophone is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// General Public License for more details.  You should have received
// a copy of the GNU General Public License along with Kinetophone. If
// not, see <http://www.gnu.org/licenses/>.

#include "Kinetophone_builder_config.hpp"
#include <kinetophone/File_manager.hpp>
#include <kinetophone/Image_types.hpp>
#include <kinetophone/Movie_builder_config.hpp>
#include <kinetophone/Movie_config.hpp>
#include <kinetophone/option_checks.hpp>
#include <kinetophone/Option_validators.hpp>
#include <kinetophone/Sound_file_config.hpp>
#include <kinetophone/error/Boost_error.hpp>
#include <kinetophone/error/Kinetophone_error.hpp>
#include <boost/program_options.hpp>
#include <boost/concept/assert.hpp>
#include <boost/concept_check.hpp>
#include <exception>
#include <fstream>
#include <sstream>
#include <string>
#include <cstdlib>

using std::endl;
using std::exception;
using std::ifstream;
using std::logic_error;
using std::string;
using std::stringstream;
using Roan_trail::Kinetophone::Color;
using Roan_trail::Kinetophone::File_manager;
using Roan_trail::Kinetophone::check_conflicting_options;
using Roan_trail::Kinetophone::check_option_dependency;
using Roan_trail::Kinetophone::Color_validator;
using Roan_trail::Kinetophone::Directory_validator;
using Roan_trail::Kinetophone::Fraction_validator;
using Roan_trail::Kinetophone::Positive_double_validator;
using Roan_trail::Kinetophone::Positive_int_validator;
using Roan_trail::Kinetophone::Str_to_int_lookup_validator;
using Roan_trail::Kinetophone::Validator;
using Roan_trail::Kinetophone::Movie_builder_config;
using Roan_trail::Kinetophone::Boost_error;
using Roan_trail::Kinetophone::Error_param;
using Roan_trail::Kinetophone::Kinetophone_error;
using namespace Roan_trail::Kinetophone_app;

namespace PO = boost::program_options;

//
// Internal helpers
//

namespace
{
  using Roan_trail::Kinetophone::Sound_file_config;
  using Roan_trail::Kinetophone::Movie_config;
  using Roan_trail::Kinetophone::Movie_builder_config;

  //
  // some custom validation code for this configuration
  //

  typedef struct Str_to_int_lookup_validator<string,
                                             Sound_file_config::file_type_for_string> Ih_file_type_validator;
  typedef struct Str_to_int_lookup_validator<string,
                                             Sound_file_config::data_format_for_string> Ih_data_format_validator;
  typedef struct Str_to_int_lookup_validator<string,
                                             Sound_file_config::endianness_for_string> Ih_endianness_validator;
  typedef struct Str_to_int_lookup_validator<string,
                                             Movie_config::movie_preset_for_string> Ih_movie_preset_validator;
  typedef struct Str_to_int_lookup_validator<string,
                              Movie_config::frame_rate_type_for_string> Ih_frame_rate_type_validator;
  typedef struct Str_to_int_lookup_validator<string,
                                             Movie_config::frame_format_for_string> Ih_frame_format_validator;

  //
  // the main validator struct for this configuration
  //

  struct Ih_validator_struct
  {
    Ih_validator_struct(const Kinetophone_builder_config& s)
      : image_source_path(s.image_source_path),
        image_directory(),
        session_file_path(s.session_file_path),
        movie_builder_config(*s.movie_builder_config),
        detailed_error(Validator<bool>(s.detailed_error)),
        quit_on_prebuild_warnings(Validator<bool>(s.quit_on_prebuild_warnings)),
        custom_installation_dir(Directory_validator(s.custom_installation_dir)) {}
    void update(Kinetophone_builder_config& return_cfg)
    {
      return_cfg.session_file_path = session_file_path;
      movie_builder_config.update(*return_cfg.movie_builder_config);
      return_cfg.detailed_error = detailed_error.v;
      return_cfg.quit_on_prebuild_warnings = quit_on_prebuild_warnings.v;
      return_cfg.custom_installation_dir = custom_installation_dir.v;
      // calculate and assign remaining members
      if ("" == image_source_path)
      {
        if ("" != image_directory.v)
        {
          return_cfg.image_source_path = image_directory.v;
        }
        else
        {
          // left blank to indicate that the source in the session file should be used
          return_cfg.image_source_path = "";
        }
      }
      else
      {
        return_cfg.image_source_path = image_source_path;
      }
      //   movie preset type from name
      string use_movie_preset = movie_builder_config.movie.movie_preset.v;
      if ("" == use_movie_preset)
      {
        // movie preset not specified
        // try to determine from output file extension
        const string extension = File_manager::file_extension_for_file
          (movie_builder_config.output.movie_name);
        if ("" != extension)
        {
          Movie_config::movie_preset_string_for_file_extension(extension, use_movie_preset);
        }
        else
        {
          // no extension, use default
          use_movie_preset =
            Movie_config::string_for_movie_preset(Movie_config::default_movie_preset);
        }
      }
      if (!Movie_config::movie_preset_for_string(use_movie_preset,
                                                 return_cfg.movie_builder_config->movie->movie_preset))
      {
        // not an an understood preset, this will be an error
        return_cfg.movie_builder_config->movie->movie_preset = Movie_config::movie_preset_unknown;
      }
      if (!Sound_file_config::file_type_for_string(movie_builder_config.movie.sound.file_type.v,
                                                   return_cfg.movie_builder_config->movie->sound->file_type))
      { // use default file type
        return_cfg.movie_builder_config->movie->sound->file_type = Sound_file_config::default_file_type;
      }
      if (!Sound_file_config::data_format_for_string(movie_builder_config.movie.sound.data_format.v,
                                                     return_cfg.movie_builder_config->movie->sound->data_format))
      { // use default data format
        return_cfg.movie_builder_config->movie->sound->data_format = Sound_file_config::default_data_format;
      }
      if (!Sound_file_config::endianness_for_string(movie_builder_config.movie.sound.endianness.v,
                                                    return_cfg.movie_builder_config->movie->sound->endianness))
      { // use default endianness
        return_cfg.movie_builder_config->movie->sound->endianness = Sound_file_config::default_endianness;
      }
      // output directory
      string directory_from_movie_name =
        File_manager::parent_directory_for_path(movie_builder_config.output.movie_name);
      if ("" != directory_from_movie_name)
      {
        // override the directory specified on the command line (if any)--we will
        // deal with conflicts later
        return_cfg.movie_builder_config->output.directory = directory_from_movie_name;
      }
      // strip parent path (if any) from movie name
      return_cfg.movie_builder_config->output.movie_name =
        File_manager::file_name_for_file_path(movie_builder_config.output.movie_name);
    }
    string image_source_path;
    Directory_validator image_directory;
    string session_file_path;
    struct Movie_builder_config_validator_struct
    {
      Movie_builder_config_validator_struct(const Movie_builder_config& m)
        : movie(*m.movie),
          output(m.output),
          movie_attributes(m.movie_attributes),
          quiet(Validator<bool>(m.quiet)),
          verbose(Validator<bool>(m.verbose)),
          skip_build_warnings(Validator<bool>(m.skip_build_warnings)) {}
      void update(Movie_builder_config& return_m)
      {
        movie.update(*return_m.movie);
        output.update(return_m.output);
        movie_attributes.update(return_m.movie_attributes);
        return_m.quiet = quiet.v;
        return_m.verbose = verbose.v;
        return_m.skip_build_warnings = skip_build_warnings.v;
      }
      struct Movie_validator_struct
      {
        Movie_validator_struct(const Movie_config& m)
          : movie_preset(Ih_movie_preset_validator("")),
            frame_rate(Ih_frame_rate_type_validator(Movie_config::string_for_frame_rate_type(m.frame_rate))),
            frame_format(Ih_frame_format_validator(Movie_config::
                                                   string_for_frame_format(m.frame_format))),
            keyframe_period(Positive_int_validator(m.keyframe_period)),
            sound(*m.sound) {}
        void update(Movie_config& return_m)
        {
          if (!Movie_config::movie_preset_for_string(movie_preset.v, return_m.movie_preset))
          {
            return_m.movie_preset = Movie_config::movie_preset_unknown;
          }
          if (!Movie_config::frame_rate_type_for_string(frame_rate.v, return_m.frame_rate))
          {
            assert(false && "invalid frame rate type string");
          }
          if (!Movie_config::frame_format_for_string(frame_format.v, return_m.frame_format))
          {
            assert(false && "invalid frame format string");
          }
          return_m.keyframe_period = keyframe_period.v;
          sound.update(*return_m.sound);
        }
        Ih_movie_preset_validator movie_preset;
        Ih_frame_rate_type_validator frame_rate;
        Ih_frame_format_validator frame_format;
        Positive_int_validator keyframe_period;
        // currently, none of these can be specified on the command line
        struct Sound_validator_struct
        {
          Sound_validator_struct(const Sound_file_config& s)
            : channels(Positive_int_validator(s.channels)),
              sample_rate(Positive_int_validator(static_cast<int>(s.sample_rate))),
              VBR_quality(Fraction_validator(s.VBR_quality)),
              file_name(s.file_name),
              file_type(Ih_file_type_validator(Sound_file_config::string_for_file_type(s.file_type))),
              data_format(Ih_data_format_validator(Sound_file_config::string_for_data_format(s.data_format))),
              endianness(Ih_endianness_validator(Sound_file_config::string_for_endianness(s.endianness))) {}
          void update(Sound_file_config& return_s)
          {
            return_s.channels = channels.v;
            return_s.sample_rate = sample_rate.v;
            return_s.VBR_quality = VBR_quality.v;
            return_s.file_name = file_name;
            if (!Sound_file_config::file_type_for_string(file_type.v, return_s.file_type))
            {
              assert(false && "invalid file type string");
            }
            if (!Sound_file_config::data_format_for_string(data_format.v, return_s.data_format))
            {
              assert(false && "invalid data format string");
            }
            if (!Sound_file_config::endianness_for_string(endianness.v, return_s.endianness))
            {
              assert(false && "invalid endianness string");
            }
          }
          Positive_int_validator channels;
          Positive_int_validator sample_rate;
          Fraction_validator VBR_quality;
          string file_name;
          Ih_file_type_validator file_type;
          Ih_data_format_validator data_format;
          Ih_endianness_validator endianness;
        } sound;
      } movie;
      struct Output_validator_struct
      {
        Output_validator_struct(const Movie_builder_config::Output_struct& o)
          : movie_name(o.movie_name),
            temporary_directory(o.temporary_directory),
            directory(o.directory),
            file_overwrite(Validator<bool>(o.file_overwrite)),
            create_multiple_movies(Validator<bool>(o.create_multiple_movies)),
            temporary_movies(Validator<bool>(o.temporary_movies)) {}
        void update(Movie_builder_config::Output_struct& return_o)
        {
          return_o.movie_name = movie_name;
          return_o.temporary_directory = temporary_directory;
          return_o.directory = directory;
          return_o.file_overwrite = file_overwrite.v;
          return_o.create_multiple_movies = create_multiple_movies.v;
          return_o.temporary_movies = temporary_movies.v;
        }
        string movie_name;
        string temporary_directory;
        string directory;
        Validator<bool> file_overwrite;
        Validator<bool> create_multiple_movies;
        Validator<bool> temporary_movies;
      } output;
      struct Movie_attributes_validator_struct
      {
        Movie_attributes_validator_struct(const Movie_builder_config::Movie_attributes_struct& a)
          : square_pixels(Validator<bool>(a.square_pixels)),
            silence_gaps(Validator<bool>(a.silence_gaps)),
            silence_gap_length(Positive_int_validator(a.silence_gap_length)),
            full_resolution_scale(Positive_double_validator(a.full_resolution_scale)),
            aspect_fill_color(Color_validator(a.aspect_fill_color)),
            PDF_fill_color(Color_validator(a.PDF_fill_color)),
            crop_image_to_frame(Validator<bool>(a.crop_image_to_frame)) {}
        void update(Movie_builder_config::Movie_attributes_struct& return_a)
        {
          return_a.square_pixels = square_pixels.v;
          return_a.silence_gaps = silence_gaps.v;
          return_a.silence_gap_length = silence_gap_length.v;
          return_a.full_resolution_scale = full_resolution_scale.v;
          return_a.aspect_fill_color = aspect_fill_color.v;
          return_a.PDF_fill_color = PDF_fill_color.v;
          return_a.crop_image_to_frame = crop_image_to_frame.v;
        }
        Validator<bool> square_pixels;
        Validator<bool> silence_gaps;
        Positive_int_validator silence_gap_length;
        Positive_double_validator full_resolution_scale;
        Color_validator aspect_fill_color;
        Color_validator PDF_fill_color;
        Validator<bool> crop_image_to_frame;
      } movie_attributes;
      Validator<bool> quiet;
      Validator<bool> verbose;
      Validator<bool> skip_build_warnings;
    } movie_builder_config;
    Validator<bool> detailed_error;
    Validator<bool> quit_on_prebuild_warnings;
    Directory_validator custom_installation_dir;
  };

  //
  // Auxiliary functions for checking input validity.
  //

  void ih_check_option_consistency(const Kinetophone_builder_config& cfg,
                                   const PO::variables_map& VM)
  {
    // conflicting option checks
    check_conflicting_options(VM,
                              "output-directory",
                              "movie-name",
                              ("" != File_manager::parent_directory_for_path
                               (cfg.movie_builder_config->output.movie_name)),
                              "output directory specified conflicts with "
                              "parent directory of movie name, omit one or the other");
    check_conflicting_options(VM,
                              "quiet",
                              "verbose");
    check_conflicting_options(VM,
                              "movie-name",
                              "temporary-movies");
    check_conflicting_options(VM,
                              "output-directory",
                              "temporary-movies");
    check_conflicting_options(VM,
                              "skip-build-warnings",
                              "quit-on-prebuild-warnings");
    check_conflicting_options(VM,
                              "image-directory",
                              "image-source-path");
    // required option checks
    if ("" == cfg.session_file_path)
    {
      throw logic_error(string("session file not specified"));
    }
    // option dependency checks
    check_option_dependency(VM,
                            "silence-gap-length",
                            "silence-gaps");
    // invalid movie preset
    if (Movie_config::movie_preset_unknown == cfg.movie_builder_config->movie->movie_preset)
    {
      throw logic_error("could not determine a movie preset");
    }
  }

  typedef struct
  {
    PO::options_description& cmd_line_desc;
    PO::options_description& basic_desc;
    PO::options_description& advanced_desc;
  } Ih_desc_params;

  // sets up the program option description object
  void ih_setup_descriptions(Ih_desc_params& return_desc_params,
                             Ih_validator_struct& return_v,
                             string& return_config_file)
  {
    const char *home_dir = getenv("HOME"); // (check_code_ignore)
    string default_config_file = (!home_dir || (*home_dir == 0) || !File_manager::path_is_directory(home_dir)) ?
      string("") : string(home_dir) + string("/.kinetophone");

    return_desc_params.cmd_line_desc.add_options()
      ("help,h", "output help message")
      ("config,i", PO::value<string>(&return_config_file)->default_value(default_config_file),
       "specify configuration file")
      ("version,v", "print program version")
      ;

    return_desc_params.basic_desc.add_options()
      ("session-file,x", PO::value<string>(&return_v.session_file_path)
       ->default_value(return_v.session_file_path),
       "set narrator session XML file for building movie(s)")
      ("image-directory,D", PO::value<Directory_validator>(&return_v.image_directory),
       "set image source directory (overrides session file)")
      ("image-source-path,P", PO::value<string>(&return_v.image_source_path)
       ->default_value(return_v.image_source_path),
       "set image source path (directory or PDF file, overrides session file)")
      ("sound-file,f", PO::value<string>(&return_v.movie_builder_config.movie.sound.file_name)
       ->default_value(return_v.movie_builder_config.movie.sound.file_name),
       "set sound file path for movie (overrides session)")
      ("frame-format,a",
       PO::value<Ih_frame_format_validator>(&return_v.movie_builder_config.movie.frame_format)
       ->default_value(return_v.movie_builder_config.movie.frame_format),
       "set output movie frame format\n"
       "{ 601_ntsc_4:3 | 601_ntsc_16:9 | dv_ntsc_4:3 | dv_ntsc_16:9_anamorphic | 601_dv_pal_4:3 |"
       " hd_720p | hd_1080 | original }") // (check_code_ignore)
      ("frame-rate,r", PO::value<Ih_frame_rate_type_validator>(&return_v.movie_builder_config.movie.frame_rate)
       ->default_value(return_v.movie_builder_config.movie.frame_rate),
       "set output movie frame rate\n"
       "{ 5_Hz | 10_Hz | 20_Hz | 24_Hz | 25_Hz | 29_97_Hz | 30_Hz | 50_Hz | 60_Hz }") // (check_code_ignore)
      ("movie-name,m", PO::value<string>(&return_v.movie_builder_config.output.movie_name)
       ->default_value(return_v.movie_builder_config.output.movie_name),
       "set base name for movie(s)")
      ("temporary-directory,t", PO::value<string>(&return_v.movie_builder_config.output.temporary_directory)
       ->default_value(return_v.movie_builder_config.output.temporary_directory),
       "set temporary directory for building movie(s)")
      ("output-directory,d", PO::value<string>(&return_v.movie_builder_config.output.directory)
       ->default_value(return_v.movie_builder_config.output.directory),
       "set output directory for movie(s)")
      ("overwrite,o", PO::value<Validator<bool> >(&return_v.movie_builder_config.output.file_overwrite)
       ->implicit_value(Validator<bool>(true)),
       "overwrite existing movies")
      ("multiple-movies,u",
       PO::value<Validator<bool> >(&return_v.movie_builder_config.output.create_multiple_movies)
       ->implicit_value(Validator<bool>(true)),
       "output each segment as a separate movie")
      ("temporary-movies,T",
       PO::value<Validator<bool> >(&return_v.movie_builder_config.output.temporary_movies)
       ->implicit_value(Validator<bool>(true)),
       "keep temporary movies in the temporary directory, do not move to destination")
      ;

    return_desc_params.advanced_desc.add_options()
      ("keyframe-period,K",
       PO::value<Positive_int_validator>(&return_v.movie_builder_config.movie.keyframe_period),
       "set period in frames between video keyframes (must be > 0, can make HUGE movies!)")
      ("square-pixels,q",
       PO::value<Validator<bool> >(&return_v.movie_builder_config.movie_attributes.square_pixels)
       ->implicit_value(Validator<bool>(true)),
       "force square (1:1 aspect) pixels")
      ("silence-gaps,g",
       PO::value<Validator<bool> >(&return_v.movie_builder_config.movie_attributes.silence_gaps)
       ->implicit_value(Validator<bool>(true)),
       "create silence gaps at segment boundaries")
      ("silence-gap-length,l",
       PO::value<Positive_int_validator>(&return_v.movie_builder_config.movie_attributes.silence_gap_length)
       ->default_value(return_v.movie_builder_config.movie_attributes.silence_gap_length),
       "set length of silence in video frames (must be > 0, requires --silence-gaps option)")
      ("full-resolution-scale,S",
       PO::value<Positive_double_validator>(&return_v.movie_builder_config.movie_attributes.full_resolution_scale)
       ->default_value(return_v.movie_builder_config.movie_attributes.full_resolution_scale),
       "set scale for output of movies at full resolution (0.0 < scale <= 1.0)")
      ("aspect-fill-color,c",
       PO::value<Color_validator>(&return_v.movie_builder_config.movie_attributes.aspect_fill_color)
       ->default_value(return_v.movie_builder_config.movie_attributes.aspect_fill_color),
       "set color for letterboxing/pillarboxing (format RRGGBB, in hexadecimal")
      ("PDF-fill-color,L",
       PO::value<Color_validator>(&return_v.movie_builder_config.movie_attributes.PDF_fill_color)
       ->default_value(return_v.movie_builder_config.movie_attributes.PDF_fill_color),
       "set fill color for PDF background (format RRGGBB, in hexadecimal")
      ("crop-image-to-frame,R",
       PO::value<Validator<bool> >(&return_v.movie_builder_config.movie_attributes.crop_image_to_frame)
       ->implicit_value(Validator<bool>(true)),
       "scale and fill, rather than fit, source image in video frame ")
      ("skip-build-warnings,k",
       PO::value<Validator<bool> >(&return_v.movie_builder_config.skip_build_warnings)
       ->implicit_value(Validator<bool>(true)),
       "do not output builder warnings (cannot use with --quit-on-prebuild-warnings)")
      ("quiet,Q", PO::value<Validator<bool> >(&return_v.movie_builder_config.quiet)
       ->implicit_value(Validator<bool>(true)),
       "little output, if any, from the build (cannot use with --verbose)")
      ("verbose,V", PO::value<Validator<bool> >(&return_v.movie_builder_config.verbose)
       ->implicit_value(Validator<bool>(true)),
       "talkative build (cannot use with --quiet)")
      ("detailed-error,e", PO::value<Validator<bool> >(&return_v.detailed_error)
       ->implicit_value(Validator<bool>(true)),
       "display detailed diagnostics on error")
      ("quit-on-prebuild-warnings,p", PO::value<Validator<bool> >(&return_v.quit_on_prebuild_warnings)
       ->implicit_value(Validator<bool>(true)),
       "quit, do not continue building, if there are prebuild warning(s) " // (check_code_ignore)
       "(cannot use with --skip-build-warnings)")
      ("config-directory,C", PO::value<Directory_validator>(&return_v.custom_installation_dir),
       "set custom program configuration directory")
      ;
  }
}

//
// Constructor/destructor/copy
//

Kinetophone_builder_config::Kinetophone_builder_config()
  // setup some reasonable defaults
  : help_message(""),
    session_file_path(),
    image_source_path(""), // left blank to indicate that the source in the session file should be used
    movie_builder_config(new Movie_builder_config),
    detailed_error(false),
    quit_on_prebuild_warnings(false),
    custom_installation_dir("")
{
}

Kinetophone_builder_config::~Kinetophone_builder_config()
{
  delete movie_builder_config;
}

//
//  Class constants
//

// commands
const int Kinetophone_builder_config::command_error;
const int Kinetophone_builder_config::command_build;
const int Kinetophone_builder_config::command_help;
const int Kinetophone_builder_config::command_output_version;

int Kinetophone_builder_config::parse_program_options(int argc,
                                                      const char** argv,
                                                      Error_param& return_error)
{
  precondition(!return_error());

  int return_value = Kinetophone_builder_config::command_error;

  help_message = "";

  start_error_block();

  try
  {
    PO::options_description cmd_line_desc("Command line only");
    stringstream basic_desc_label;
    basic_desc_label << "Command line and configuration file" << endl << "  Basic";
    PO::options_description basic_desc(basic_desc_label.str());
    PO::options_description advanced_desc("  Advanced");
    Ih_desc_params desc_params =
    {
      cmd_line_desc,
      basic_desc,
      advanced_desc
    };

    Ih_validator_struct validator(*this);
    string config_file;
    ih_setup_descriptions(desc_params,
                          validator,
                          config_file);

    // parse the command line to get config file
    PO::variables_map VM_config;
    PO::store(PO::command_line_parser(argc, const_cast<char**>(argv)).
              options(cmd_line_desc).allow_unregistered().run(), VM_config);
    notify(VM_config);

    PO::options_description config_desc;
    config_desc.add(basic_desc).add(advanced_desc);

    stringstream help_stream;
    help_stream << "Usage:  ";
    help_stream << "kinetophone_builder --help | [ session-file ] [ options ]" << endl;
    help_stream << endl;

    PO::options_description all_desc("Options");
    all_desc.add(cmd_line_desc).add(basic_desc).add(advanced_desc);
    help_stream << all_desc;
    help_message = help_stream.str();

    PO::variables_map VM;

    // parse the command line (overrides config file)
    PO::positional_options_description pos_desc;
    pos_desc.add("session-file", 1);
    PO::store(PO::command_line_parser(argc, const_cast<char**>(argv)).
              options(all_desc).positional(pos_desc).run(), VM);
    PO::notify(VM);

    // parse the config file
    if ("" != config_file)
    {
      ifstream ifs(config_file.c_str());
      bool have_error =  (!ifs && !VM_config["config"].defaulted());
      on_error(have_error, new Boost_error(error_location(), string("cannot open config file: ") + config_file));
      const bool allow_unregistered_options = true;
      PO::store(PO::parse_config_file(ifs, config_desc, allow_unregistered_options), VM);
      PO::notify(VM);
    }

    if (VM.count("help"))
    { // HELP
      return_value = Kinetophone_builder_config::command_help;
    }
    else if (VM.count("version"))
    {
      // PROGRAM VERSION
      return_value = Kinetophone_builder_config::command_output_version;
    }
    else
    { // BUILD
      // update the config from the validator
      validator.update(*this);
      ih_check_option_consistency(*this, VM);
      return_value = Kinetophone_builder_config::command_build;
    }
  }
  catch (const exception& e)
  {
    string diagnostic = e.what();
    stringstream s;
    s << "Reason: " << diagnostic << endl << endl;
    s << "Enter kinetophone_builder --help for valid program options.";
    Kinetophone_error* error = new Kinetophone_error(error_location(),
                                                     Kinetophone_error::command_line,
                                                     new Boost_error(error_location(), s.str()));
    on_error(true, error);
  }
  catch (...)
  {
    string diagnostic = "exception parsing command line";
    Kinetophone_error* error = new Kinetophone_error(error_location(),
                                                     Kinetophone_error::command_line,
                                                     diagnostic);
    on_error(true, error);
  }

  goto exit_point;

  end_error_block();

  default_error_handler(return_error);

 error_cleanup:
  if (!return_error.need_error())
  {
    delete handler_error;
  }
  return_value = Kinetophone_builder_config::command_error;
  goto exit_point;

 exit_point:
  return return_value;
}
