// Kinetophone_vox_model.cpp
//
// Copyright 2011-2012 Roan Trail, Inc.
//
// This file is part of Kinetophone.
//
// Kinetophone is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published
// by the Free Software Foundation, either version 2 of the License,
// or (at your option) any later version.
//
// Kinetophone is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// General Public License for more details.  You should have received
// a copy of the GNU General Public License along with Kinetophone. If
// not, see <http://www.gnu.org/licenses/>.

#include "Kinetophone_vox_model.hpp"

// include *mm first to avoid conflicts
#include <gdkmm/pixbuf.h>

#include "Kinetophone_vox_config.hpp"
#include "../narrator/Kinetophone_dir_slide_collection.hpp"
#include "../narrator/Kinetophone_PDF_slide_collection.hpp"
#include "../base/File_manager.hpp"
#include "../base/Narration_synthesizer.hpp"
#include "../base/Slide_collection.hpp"
#include "../base/error/Kinetophone_error.hpp"
#include "../base/error/Posix_error.hpp"
#include <libxml++/document.h>
#include <libxml++/exceptions/exception.h>
#include <libxml++/nodes/node.h>
#include <libxml++/parsers/domparser.h>
#include <fstream>
#include <string>
#include <cerrno> // (check_code_ignore)

using Gdk::Pixbuf;
using Glib::RefPtr;
using xmlpp::Document;
using xmlpp::DomParser;
using xmlpp::Node;
using std::ofstream;
using std::ios;
using std::ios_base;
using std::string;
using Roan_trail::Error_param;
using Roan_trail::File_manager;
using Roan_trail::Recorder::Narration_synthesizer;
using Roan_trail::Source::Slide_collection;
using Roan_trail::Source::source_type_from_XML;
using namespace Roan_trail::Kinetophone;

//
// Constructor/destructor
//

Kinetophone_vox_model::Kinetophone_vox_model(const Kinetophone_vox_config& vox_config)
  : m_vox_config(vox_config),
    m_slides(0),
    m_synthesizer(new Narration_synthesizer),
    m_slides_source(directory_source)
{
  postcondition(mf_invariant(false));
}

Kinetophone_vox_model::~Kinetophone_vox_model()
{
  precondition(mf_invariant(false));

  delete m_synthesizer;
  delete m_slides;
}

//
// Protected member functions
//

bool Kinetophone_vox_model::mf_invariant(bool check_base_class) const
{
  bool return_value = false;

  if (!m_synthesizer)
  {
    goto exit_point;
  }

  return_value = true;

 exit_point:
  return return_value;
}

//
// Commands
//

bool Kinetophone_vox_model::load_from_XML(const string& file_path, Error_param& return_error)
{
  precondition(!return_error()
               && mf_invariant());

  Slide_collection<RefPtr<Pixbuf> >* slides = 0;
  Slides_source_type slides_source;
  bool return_value = false;

  start_error_block();

  Error_param error;
  const string& diagnostic_prefix = "error loading XML file";

  try
  {
    DomParser parser(file_path); // no XML validation by default
    parser.set_substitute_entities(); // substitute text for entity references
    parser.parse_file(file_path);
    on_error(!parser, new Kinetophone_error(error_location(),
                                            Kinetophone_error::file_IO,
                                            diagnostic_prefix + string(", could not parse"),
                                            file_path));

    const Node* root_node = parser.get_document()->get_root_node();
    on_error(!root_node, new Kinetophone_error(error_location(),
                                               Kinetophone_error::file_IO,
                                               diagnostic_prefix + string(", could not parse"),
                                               file_path));

    string source_type_string;
    const bool got_source_type = source_type_from_XML(file_path,
                                                      root_node,
                                                      source_type_string,
                                                      error);
    on_error(!got_source_type, new Kinetophone_error(error_location(),
                                                     Kinetophone_error::file_IO,
                                                     error()));
    if ("directory" == source_type_string)
    {
      slides = new Kinetophone_dir_slide_collection;
      slides_source = directory_source;
    }
    else if ("PDF" == source_type_string)
    {
      slides = new Kinetophone_PDF_slide_collection;
      slides_source = PDF_source;
    }
    else
    {
      on_error(true, new Kinetophone_error(error_location(),
                                           Kinetophone_error::file_IO,
                                           diagnostic_prefix + string(", could not determine source type"),
                                           file_path));
    }

    // import into the slide collection
    const string custom_source_path = m_vox_config.image_source_path;

    const bool slides_loaded = slides->load_from_XML(file_path,
                                                     root_node,
                                                     custom_source_path,
                                                     error);
    on_error(!slides_loaded, new Kinetophone_error(error_location(),
                                                   Kinetophone_error::build,
                                                   error()));
  }
  catch (const xmlpp::exception& e)
  {
    const string diagnostic = diagnostic_prefix + string(": ") + e.what();
    on_error(true, new Kinetophone_error(error_location(),
                                         Kinetophone_error::file_IO,
                                         diagnostic,
                                         file_path));
  }

  delete m_slides;
  m_slides = slides;
  m_slides_source = slides_source;
  return_value = true;

  goto exit_point;
  end_error_block();

  default_error_handler(return_error);

 error_cleanup:
  if (!return_error.need_error())
  {
    delete handler_error;
  }
  if (slides)
  {
    delete slides;
  }
  goto exit_point;

 exit_point:
  postcondition(mf_invariant());
  return return_value;
}

bool Kinetophone_vox_model::load_from_PDF_source(const string& PDF_source_path, Error_param& return_error)
{
  precondition(!return_error()
               && mf_invariant());

  Kinetophone_PDF_slide_collection* slides = 0;
  bool return_value = false;

  start_error_block();

  slides = new Kinetophone_PDF_slide_collection;
  slides->set_import_PDF_notes(true);

  Error_param error;
  const bool slides_loaded = slides->load_from_source(PDF_source_path, error);
  on_error(!slides_loaded, new Kinetophone_error(error_location(),
                                                 Kinetophone_error::source_load,
                                                 error()));

  delete m_slides;
  m_slides = slides;
  m_slides_source = PDF_source;
  return_value = true;

  goto exit_point;
  end_error_block();

  default_error_handler(return_error);

 error_cleanup:
  if (!return_error.need_error())
  {
    delete handler_error;
  }
  if (slides)
  {
    delete slides;
  }
  goto exit_point;

 exit_point:
  postcondition(mf_invariant());
  return return_value;
}

bool Kinetophone_vox_model::save_as_XML(const string& file_path,
                                        bool overwrite,
                                        Error_param& return_error) const
{
  precondition(!return_error()
               && mf_invariant());

  bool return_value = false;

  start_error_block();

  Error_param error;

  Document kinetophone_narrator_document;
  kinetophone_narrator_document.create_root_node("kinetophone_narrator_model");
  Node* XML_root_node = kinetophone_narrator_document.get_root_node();
  assert(XML_root_node && "error getting root node from kinetophone narrator (vox) XML document");

  const bool slides_saved = m_slides->save_as_XML(file_path,
                                                  XML_root_node,
                                                  error);
  on_error(!slides_saved, new Kinetophone_error(error_location(),
                                                Kinetophone_error::general,
                                                error()));

  const bool synthesizer_saved = m_synthesizer->save_as_XML(file_path,
                                                            XML_root_node,
                                                            error);
  on_error(!synthesizer_saved, new Kinetophone_error(error_location(),
                                                     Kinetophone_error::general,
                                                     error()));

  const string XML = kinetophone_narrator_document.write_to_string_formatted(); // UTF-8 encoding by default

  ofstream XML_file;

  // Note: this isn't really a satisfying way to check for a file's existence
  //       because it allows a race condition, but...
  if (!overwrite)
  {
    XML_file.open(file_path.c_str(), ios_base::in);
    on_error(XML_file.is_open(), new Kinetophone_error(error_location(),
                                                       Kinetophone_error::file_IO,
                                                       "file exists and overwrite is not allowed",
                                                       file_path));
  }

  XML_file.open(file_path.c_str(), ios_base::out | ios_base::trunc);
  on_error(!XML_file.is_open(), new Kinetophone_error(error_location(),
                                                      Kinetophone_error::file_IO,
                                                      new Posix_error(errno, // (check_code_ignore)
                                                                      "open for write",
                                                                      file_path.c_str())));
  XML_file << XML;

  // check for write error
  on_error(!XML_file, new Kinetophone_error(error_location(),
                                            Kinetophone_error::file_IO,
                                            new Posix_error(errno, // (check_code_ignore)
                                                            "write",
                                                            file_path.c_str())));

  return_value = true;

  goto exit_point;
  end_error_block();

  default_error_handler_and_cleanup(return_error,
                                    return_value,
                                    false);
  goto exit_point;

 exit_point:
  postcondition(mf_invariant());
  return return_value;
}
