// Kinetophone_vox_manager.cpp
//
// Copyright 2011-2012 Roan Trail, Inc.
//
// This file is part of Kinetophone.
//
// Kinetophone is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published
// by the Free Software Foundation, either version 2 of the License,
// or (at your option) any later version.
//
// Kinetophone is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// General Public License for more details.  You should have received
// a copy of the GNU General Public License along with Kinetophone. If
// not, see <http://www.gnu.org/licenses/>.

#include "Kinetophone_vox_manager.hpp"

// include *mm first to avoid conflicts
#include <gdkmm/pixbuf.h>

#include "Kinetophone_vox_config.hpp"
#include "Kinetophone_vox_model.hpp"
#include "../base/File_manager.hpp"
#include "../base/Logger.hpp"
#include "../base/Segment.hpp"
#include "../base/Slide.hpp"
#include "../base/Slide_collection.hpp"
#include "../base/Narration_synthesizer.hpp"
#include "../base/Speech_synthesizer_config.hpp"
#include "../base/Sound_file_config.hpp"
#include "../base/error/Kinetophone_error.hpp"
#include "../base/error/Build_error.hpp"
#include "../base/error/Posix_error.hpp"
#include <fstream>
#include <iostream>
#include <sstream>
#include <iomanip>

using Glib::RefPtr;
using Gdk::Pixbuf;
using std::endl;
using std::ios_base;
using std::ofstream;
using std::setfill;
using std::setw;
using std::string;
using std::stringstream;
using Roan_trail::Logger;
using Roan_trail::Builder::Build_error;
using Roan_trail::Recorder::Segment;
using Roan_trail::Source::Slide;
using Roan_trail::Source::Slide_collection;
using Roan_trail::Recorder::Narration_synthesizer;
using Roan_trail::Recorder::Speech_synthesizer_config;
using Roan_trail::Recorder::Sound_file_config;
using namespace Roan_trail::Kinetophone;

//
// Internal helpers
//
namespace
{
  const double ic_estimate_factor = 1.01;
  const double ic_output_space_warning_threshold = 0.95;
}

//
// Constructor/destructor
//

Kinetophone_vox_manager::Kinetophone_vox_manager(const Kinetophone_vox_config& config,
                                                 Kinetophone_vox_model& model)
  : m_config(config),
    m_model(model),
    m_logger(Logger::default_logger()),
    m_running(0),
    m_cancelled(0),
    m_setup(false)
{
  postcondition(mf_invariant(false));
}

Kinetophone_vox_manager::~Kinetophone_vox_manager()
{
  precondition(mf_invariant(false));
}

//
// Control
//

bool Kinetophone_vox_manager::setup_for_generate(Error_param& return_error)
{
  precondition(!return_error()
               && mf_invariant());

  bool return_value = false;

  start_error_block();

  mf_log_config();

  m_logger << Logger::info << "Setting up for vox generation" << endl;

  Error_param error;
  Narration_synthesizer* synthesizer = m_model.narration_synthesizer();
  const bool synthesizer_started = synthesizer->startup(*m_config.speech_synthesizer_config,
                                                        error);
  on_error(!synthesizer_started, new Kinetophone_error(error_location(),
                                                       Kinetophone_error::vox,
                                                       error()));

  mf_log_model();

  // success, update data members
  m_setup = true;

  return_value = true;
  goto exit_point;

  end_error_block();

  default_error_handler_and_cleanup(return_error,
                                    return_value,
                                    false);

 exit_point:
  postcondition(return_error.is_valid_at_return(return_value)
                && mf_invariant());
  return return_value;
}

// Perform various checks before generating the session file/sound
// file.  The check is not a guarantee that this will work,
// however, because conditions might change after the check.
bool Kinetophone_vox_manager::pregenerate_checks(Error_param& return_error)
{
  precondition(m_setup
               && !return_error()
               && mf_invariant());

  bool return_value = false;
  bool have_warning = false;


  // check output session file existence
  if (File_manager::path_exists(m_config.vox_generation.output_session_file))
  {
    Kinetophone_error* exists_warning =
      new Kinetophone_error(error_location(),
                            Kinetophone_error::vox_warning,
                            new Build_error(error_location(),
                                            (m_config.vox_generation.overwrite_output_session_file ?
                                             Build_error::pregenerate_output_session_exists_overwrite
                                             : Build_error::pregenerate_output_session_exists_no_overwrite)));
    mf_log_warning(exists_warning);
    delete exists_warning;
    have_warning = true;
  }

  return_value = (!have_warning || !m_config.quit_on_pregenerate_warnings);

  goto exit_point;

 exit_point:
  postcondition(m_setup
                && !return_error()
                && mf_invariant());
  return return_value;
}

bool Kinetophone_vox_manager::generate(Error_param& return_error)
{
  precondition(!return_error()
               && mf_invariant());

  __sync_or_and_fetch(&m_running, 0x1);

  bool need_shutdown = true;
  bool cancelled = false;
  bool return_value = false;
  Narration_synthesizer* synthesizer = m_model.narration_synthesizer();

  start_error_block();

  m_logger << Logger::message << "Starting vox generation" << endl;

  Error_param error;

  // vox generation loop

  const Slide_collection<RefPtr<Pixbuf> >* slides = m_model.slides();
  const int slide_count = slides->slide_count();

  m_logger << Logger::message << "Generating sound file" << endl;

  // check for at least one slide
  bool have_slide = false;
  for (int i = 0; i < slide_count; ++i)
  {
    const Slide<RefPtr<Pixbuf> >* current_slide = m_model.slides()->operator[](i);;
    if (m_config.vox_generation.output_empty_slides
        || ("" != current_slide->notes()))
    {
      have_slide = true;
      goto check_exit;
    }
  }

 check_exit:
  on_error(!have_slide, new Kinetophone_error(error_location(),
                                              Kinetophone_error::vox,
                                              "found 0 slides to output"));

  for (int i = 0; i < slide_count; ++i)
  {
    // check for cancellation
    cancelled = is_cancelled();
    if (cancelled)
    {
      m_logger << Logger::message << "Vox generation cancelled" << endl;
      goto break_out;
    }

    const Slide<RefPtr<Pixbuf> >* current_slide = m_model.slides()->operator[](i);;

    // status
    m_logger << Logger::message << "Slide " << (i + 1) << " of " << slide_count;

    if (m_config.vox_generation.output_empty_slides
        || ("" != current_slide->notes()))
    {
      m_logger << Logger::message << endl;
      mf_log_slide(*current_slide);
      // add slide as a segment with generated speech
      const bool slide_added = synthesizer->speak(i,
                                                  current_slide->notes(),
                                                  error);
      on_error(!slide_added, new Kinetophone_error(error_location(),
                                                   Kinetophone_error::vox,
                                                   error()));
      string recorded_time;
      synthesizer->recorded_time(recorded_time);
      m_logger << Logger::info << "Total recorded time: " << recorded_time << " (HH:MM:SS)" << endl;
      m_logger << Logger::info << "Total audio frames written: " << synthesizer->frames_written_count() << endl;
    }
    else
    {
      m_logger << Logger::message << " (empty notes, SKIPPED)" << endl;
    }

    // check for cancellation
    cancelled = is_cancelled();
    if (cancelled)
    {
      m_logger << Logger::message << endl << "Vox generation cancelled" << endl;
      goto break_out;
    }
  }

 break_out: // normal loop exit point (complete or cancelled)

  const bool did_end = synthesizer->end(error);
  on_error(!did_end, new Kinetophone_error(error_location(),
                                           Kinetophone_error::vox,
                                           error()));

  // create output session file
  if (!cancelled)
  {
    m_logger << Logger::message << "Creating output session file" << endl;
    const bool saved = m_model.save_as_XML(m_config.vox_generation.output_session_file,
                                           m_config.vox_generation.overwrite_output_session_file,
                                           error);
    on_error(!saved, new Kinetophone_error(error_location(),
                                           Kinetophone_error::vox,
                                           error()));
  }

  m_logger << Logger::message << "Finishing" << endl;

  need_shutdown = false;
  const bool shutdown = synthesizer->shutdown(true, error);
  on_error(!shutdown, new Kinetophone_error(error_location(),
                                            Kinetophone_error::vox,
                                            error()));

  m_logger << Logger::message;
  m_logger << "----------------------------" << endl;
  m_logger << "| Vox generation completed |" << endl;
  m_logger << "----------------------------" << endl;

  return_value = true;
  goto exit_point;

  end_error_block();

  default_error_handler(return_error);

 error_cleanup:
  if (!return_error.need_error())
  {
    delete handler_error;
  }
  if (need_shutdown)
  {
    Error_param error(false);
    synthesizer->shutdown(false, error); // ignore error
  }

  m_logger << Logger::info << "Error encountered" << endl;
  return_value = false;
  goto exit_point;

 exit_point:
  postcondition(return_error.is_valid_at_return(return_value)
                && mf_invariant());

  __sync_and_and_fetch(&m_running, 0x0);

  return return_value;
}

//
// Protected member functions
//

bool Kinetophone_vox_manager::mf_invariant(bool check_base_class) const
{
  static_cast<void>(check_base_class); // avoid unused warning

  bool return_value = false;

  return_value = true;
  goto exit_point;

 exit_point:
  return return_value;
}

//
// Private member functions
//

//
//   messages
//

void Kinetophone_vox_manager::mf_log_slide(const Slide<RefPtr<Pixbuf> >& slide)
{
  // additional info
  m_logger << Logger::info;
  const Long_int slide_index = slide.index();
  m_logger << "Slide index: " << (slide_index + 1);
  m_logger << ", image: " << File_manager::file_name_for_file_path(slide.path());
  m_logger << endl;
}

void Kinetophone_vox_manager::mf_log_config()
{
  m_logger << Logger::info;
  m_logger << endl;
  // general
  m_logger << "Output session file: " << m_config.vox_generation.output_session_file << endl;
  // speech synthesizer
  const Speech_synthesizer_config& synth_config = *m_config.speech_synthesizer_config;
  m_logger << "Speech synthesizer configuration" << endl;
  m_logger << "--------------------------------" << endl;
  const string voice_setup_command = (("" == synth_config.voice) ? "N/A"
                                      : Speech_synthesizer_config::setup_command_for_voice(synth_config.voice));
  m_logger << "  voice: " << synth_config.voice << " [setup command: " << voice_setup_command << "]" << endl;
  m_logger << "  sound file overwrite: " << (synth_config.file_overwrite ? "yes" : "no") << endl;
  m_logger << "  sound file sample rate: " << synth_config.sound_file->sample_rate << endl;
}

void Kinetophone_vox_manager::mf_log_model()
{
  m_logger << Logger::info;

  m_logger << endl;
  m_logger << "Image directory: " << m_model.slides()->source() << endl;
  m_logger << "Output sound file: " << m_model.narration_synthesizer()->output_file() << endl;
  m_logger << endl;
}

void Kinetophone_vox_manager::mf_log_warning(const Error* warning, bool output_separator)
{
  precondition(m_setup
               && warning);

  if (!m_config.vox_generation.skip_generate_warnings)
  {
    stringstream warning_stream;
    warning_stream << "Vox generation warning: " << endl;

    const Error* user_warning = warning->user_error_for_chain();

    bool have_detail = false;

    if (user_warning)
    {
      const string detailed_description =
        user_warning->dictionary_entry_for_chain(Error::detailed_description_error_key);
      if ("" != detailed_description)
      {
        have_detail = true;
        warning_stream << detailed_description << endl;
      }

      const string recovery_description =
        user_warning->dictionary_entry_for_chain(Error::recovery_description_error_key);
      if ("" != recovery_description)
      {
        have_detail = true;
        warning_stream << recovery_description << endl;
      }

      const string diagnostic = user_warning->dictionary_entry_for_chain(Error::diagnostic_error_key);
      if ("" != recovery_description)
      {
        have_detail = true;
        warning_stream << diagnostic << endl;
      }
    }

    if (!user_warning || !have_detail)
    {
      warning_stream << "Warning class: " << warning->error_class() << endl;
      warning_stream << "Code: " << warning->code();
      const Error::Error_dictionary& dictionary = warning->error_dictionary();
      Error::Error_dictionary::const_iterator code_string = dictionary.find(Error::code_string_error_key);
      if (dictionary.end() != code_string)
      {
        if ("" != code_string->second)
        {
          warning_stream << " (" << code_string->second << ")";
        }
      }
      warning_stream << endl;
    }

    if (output_separator)
    {
      warning_stream << "----" << endl;
    }

    m_logger << Logger::warning << warning_stream.str();
  }

  postcondition(m_setup);
}
