// Kinetophone_narrator_window.cpp
//
// Copyright 2011-2012 Roan Trail, Inc.
//
// This file is part of Kinetophone.
//
// Kinetophone is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published
// by the Free Software Foundation, either version 2 of the License,
// or (at your option) any later version.
//
// Kinetophone is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// General Public License for more details.  You should have received
// a copy of the GNU General Public License along with Kinetophone. If
// not, see <http://www.gnu.org/licenses/>.
//
// The record start/stop icons:
//   kinetophone_record_start.png
//   kinetophone_record_stop.png
// were taken from the gtkmm 2.24.2 release's reference documentation
// (respectively):
//   gtkmm-2.24.2/docs/reference/html/gtk-yes.png
//   gtkmm-2.24.2/docs/reference/html/gtk-no.png
// which are licensed under the Lesser GPL version 2.1.

#include "Kinetophone_narrator_window.hpp"

// include *mm first to avoid conflicts
#include <gtkmm/accelgroup.h>
#include <gtkmm/alignment.h>
#include <gtkmm/aspectframe.h>
#include <gtkmm/box.h>
#include <gtkmm/button.h>
#include <gtkmm/buttonbox.h>
#include <gtkmm/comboboxtext.h>
#include <gtkmm/dialog.h>
#include <gtkmm/eventbox.h>
#include <gtkmm/frame.h>
#include <gtkmm/image.h>
#include <gtkmm/imagemenuitem.h>
#include <gtkmm/infobar.h>
#include <gtkmm/label.h>
#include <gtkmm/menu.h>
#include <gtkmm/menubar.h>
#include <gtkmm/menuitem.h>
#include <gtkmm/paned.h>
#include <gtkmm/scrolledwindow.h>
#include <gtkmm/separator.h>
#include <gtkmm/separatormenuitem.h>
#include <gtkmm/statusbar.h>
#include <gtkmm/stock.h>
#include <gtkmm/textview.h>
#include <gtkmm/togglebutton.h>
#include <gtkmm/toggletoolbutton.h>
#include <gtkmm/toolbar.h>
#include <gtkmm/toolbutton.h>
#include <gtkmm/toolitem.h>
#include <gtkmm/viewport.h>
#include <gtkmm/window.h>
#include <pangomm/fontdescription.h>

#include "Fraction_bar_widget.hpp"
#include "Level_meter_widget.hpp"
#include "Scaled_image_widget.hpp"
#include "Kinetophone_narrator_config.hpp"
#include "../base/Application.hpp"
#include "../base/common.hpp"
#include "../base/Sound_recorder_config.hpp"
#include "../base/Sound_file_config.hpp"
#include "../base/error/Kinetophone_error.hpp"
#include <string>

using Glib::RefPtr;
using Gtk::AccelGroup;
using Gtk::Alignment;
using Gtk::AspectFrame;
using Gtk::Box;
using Gtk::Button;
using Gtk::ButtonBox;
using Gtk::ComboBoxText;
using Gtk::EventBox;
using Gtk::Frame;
using Gtk::HBox;
using Gtk::HButtonBox;
using Gtk::HPaned;
using Gtk::HSeparator;
using Gtk::Image;
using Gtk::ImageMenuItem;
using Gtk::InfoBar;
using Gtk::Label;
using Gtk::Menu;
using Gtk::MenuBar;
using Gtk::MenuItem;
using Gtk::ScrolledWindow;
using Gtk::SeparatorMenuItem;
using Gtk::Statusbar;
using Gtk::TextView;
using Gtk::ToggleButton;
using Gtk::ToggleToolButton;
using Gtk::Toolbar;
using Gtk::ToolButton;
using Gtk::ToolItem;
using Gtk::VBox;
using Gtk::Viewport;
using Gtk::Widget;
using Pango::FontDescription;
using sigc::connection;
using sigc::mem_fun;
using std::string;
using namespace Roan_trail::Kinetophone;

//
// Internal constants
//
namespace
{
  const int ic_default_main_window_width = 800;
  const int ic_default_main_window_height = 600;
}

//
// Constructor/destructor
//

Kinetophone_narrator_window::Kinetophone_narrator_window(const Kinetophone_narrator_config& config)
  : Window(),
    m_setup(false),
    m_retake_tool_button(0),
    m_previous_tool_button(0),
    m_next_tool_button(0),
    m_mute_toggle_tool_button(0),
    m_pause_toggle_tool_button(0),
    m_record_toggle_tool_button(0),
    m_slide_combo_box_text(0),
    m_level_toggle_button(0),
    m_file_export_session_menu_item(0),
    m_file_quit_menu_item(0),
    m_edit_cut_menu_item(0),
    m_edit_copy_menu_item(0),
    m_edit_paste_menu_item(0),
    m_help_about_menu_item(0),
    m_session_info_bar(0),
    m_session_info_label(0),
    m_space_time_label(0),
    m_statusbar(0),
    m_image_aspectframe(0),
    m_record_time_label(0),
    m_notes_text_view(0),
    m_notes_editing_done_button(0),
    m_thumbnail_browser_hbutton_box(0),
    m_audio_level_meter_widget(0),
    m_space_available_widget(0),
    m_slide_image_widget(0),
    m_record_icon_start(0),
    m_record_icon_stop(0),
    m_record_icon(0)
{
  mf_setup(config);

  postcondition(mf_invariant(false));
}

Kinetophone_narrator_window::~Kinetophone_narrator_window()
{
  precondition(mf_invariant(false));

  // not part of the managed elements
  delete m_record_icon_stop;
  delete m_record_icon_start;
}

//
// Protected member functions
//

bool Kinetophone_narrator_window::mf_invariant(bool check_base_class) const
{
  bool return_value = false;

  if (!m_retake_tool_button
      || !m_previous_tool_button
      || !m_next_tool_button
      || !m_mute_toggle_tool_button
      || !m_pause_toggle_tool_button
      || !m_record_toggle_tool_button
      || !m_slide_combo_box_text
      || !m_level_toggle_button
      || !m_file_export_session_menu_item
      || !m_file_quit_menu_item
      || !m_edit_cut_menu_item
      || !m_edit_copy_menu_item
      || !m_edit_paste_menu_item
      || !m_help_about_menu_item
      || !m_session_info_bar
      || !m_session_info_label
      || !m_space_time_label
      || !m_statusbar
      || !m_image_aspectframe
      || !m_record_time_label
      || !m_notes_text_view
      || !m_notes_editing_done_button
      || !m_thumbnail_browser_hbutton_box
      || !m_audio_level_meter_widget
      || !m_space_available_widget
      || !m_slide_image_widget
      || !m_record_icon_start
      || !m_record_icon_stop
      || !m_record_icon)
  {
    goto exit_point;
  }

  // signal connections seemingly cannot be checked because the widget signal_... methods
  // return a Glib::SignalProxy... object which doesn't appear to have access to the slot

  return_value = true;

 exit_point:
  return return_value;
}

//
// Private member functions
//

//
// Window setup functions
//

void Kinetophone_narrator_window::mf_setup(const Kinetophone_narrator_config& config)
{
  // setup main window
  set_title(string("Kinetophone is narrating: ") + config.image_source_path);
  set_default_size(ic_default_main_window_width, ic_default_main_window_height);
  const string& installation_dir = Application::application()->installation_dir();
  try
  {
    set_icon_from_file(installation_dir + "/kinetophone_256.tiff");
    set_default_icon_from_file(installation_dir + "/kinetophone_256.tiff");
  }
  catch (...)
  {
    // couldn't find the program icons, ignore
  }

  VBox* main_vbox = manage(new VBox);
  add(*main_vbox);

  Widget* child_section = 0;
  // menu bar
  child_section = mf_setup_menu_bar();
  main_vbox->pack_start(*child_section, Gtk::PACK_SHRINK);
  // tool bar
  child_section = mf_setup_control_toolbar();
  main_vbox->pack_start(*child_section, Gtk::PACK_SHRINK);
  // info bar
  child_section = mf_setup_session_info_bar();
  main_vbox->pack_start(*child_section, Gtk::PACK_SHRINK);
 // the following has to be done after adding to main vbox to avoid Gtk-WARNING:
  m_session_info_bar->set_default_response(Gtk::RESPONSE_OK);

  Alignment* main_alignment = manage(new Alignment);
  main_alignment->set_padding(4, 4, 4, 4);
  main_vbox->pack_start(*main_alignment);
  VBox* middle_vbox = manage(new VBox);
  main_alignment->add(*middle_vbox);

  // viewer
  child_section = mf_setup_viewer();
  middle_vbox->pack_start(*child_section);
  // separator
  HSeparator* separator_1_hseparator = manage(new HSeparator);
  middle_vbox->pack_start(*separator_1_hseparator, Gtk::PACK_SHRINK);
  // accessories (levels, available space)
  child_section = mf_setup_accessories(config);
  middle_vbox->pack_start(*child_section, Gtk::PACK_SHRINK);
  // thumbnail browswer
  child_section = mf_setup_thumbnail_browser();
  middle_vbox->pack_start(*child_section, Gtk::PACK_SHRINK);
  // statusbar
  child_section = mf_setup_statusbar();
  main_vbox->pack_start(*child_section, Gtk::PACK_SHRINK);
  // accelerators
  mf_setup_accelerators();

  show_all_children();
  m_notes_editing_done_button->hide();
  m_session_info_bar->hide();
}

Widget* Kinetophone_narrator_window::mf_setup_menu_bar()
{
  MenuBar* menu_bar = manage(new MenuBar);
  // File...
  MenuItem* file_menu_item = manage(new MenuItem);
  file_menu_item->set_use_underline(true);
  file_menu_item->set_label("_File");
  menu_bar->append(*file_menu_item);
  Menu* file_menu = manage(new Menu);
  file_menu_item->set_submenu(*file_menu);
  //   ->Export...
  MenuItem* file_export_menu_item = manage(new MenuItem);
  file_export_menu_item->set_use_underline(true);
  file_export_menu_item->set_label("_Export");
  file_menu->append(*file_export_menu_item);
  Menu* file_export_menu = manage(new Menu);
  file_export_menu_item->set_submenu(*file_export_menu);
  //   ->Session
  m_file_export_session_menu_item = manage(new MenuItem); // exposed in class interface
  m_file_export_session_menu_item->set_use_underline(true);
  m_file_export_session_menu_item->set_label("_Session");
  file_export_menu->append(*m_file_export_session_menu_item);
  // separator
  SeparatorMenuItem* separator_1_menu_item = manage(new SeparatorMenuItem);
  file_menu->append(*separator_1_menu_item);
  //   ->Quit
  m_file_quit_menu_item = manage(new ImageMenuItem(Gtk::Stock::QUIT)); // exposed in class interface
  m_file_quit_menu_item->set_use_underline(true);
  m_file_quit_menu_item->set_label("_Quit");
  file_menu->append(*m_file_quit_menu_item);
  // Edit...
  MenuItem* edit_menu_item = manage(new MenuItem);
  edit_menu_item->set_use_underline(true);
  edit_menu_item->set_label("_Edit");
  menu_bar->append(*edit_menu_item);
  Menu* edit_menu = manage(new Menu);
  edit_menu_item->set_submenu(*edit_menu);
  //   ->Cut
  m_edit_cut_menu_item = manage(new ImageMenuItem(Gtk::Stock::CUT)); // exposed in the class interface
  m_edit_cut_menu_item->set_use_underline(true);
  m_edit_cut_menu_item->set_label("Cu_t");
  edit_menu->append(*m_edit_cut_menu_item);
  //   ->Copy
  m_edit_copy_menu_item = manage(new ImageMenuItem(Gtk::Stock::COPY)); // exposed in the class interface
  m_edit_copy_menu_item->set_use_underline(true);
  m_edit_copy_menu_item->set_label("_Copy");
  edit_menu->append(*m_edit_copy_menu_item);
  //   ->Paste
  m_edit_paste_menu_item = manage(new ImageMenuItem(Gtk::Stock::PASTE)); // exposed in the class interface
  m_edit_paste_menu_item->set_use_underline(true);
  m_edit_paste_menu_item->set_label("_Paste");
  edit_menu->append(*m_edit_paste_menu_item);
  // Help...
  MenuItem* help_menu_item = manage(new MenuItem);
  help_menu_item->set_right_justified();
  help_menu_item->set_use_underline(true);
  help_menu_item->set_label("_Help");
  menu_bar->append(*help_menu_item);
  Menu* help_menu = manage(new Menu);
  help_menu_item->set_submenu(*help_menu);
  //   ->About
  m_help_about_menu_item = manage(new MenuItem); // exposed in the class interface
  m_help_about_menu_item->set_use_underline(true);
  m_help_about_menu_item->set_label("_About");
  help_menu->append(*m_help_about_menu_item);

  return menu_bar;
}

Widget* Kinetophone_narrator_window::mf_setup_control_toolbar()
{
  Toolbar* control_toolbar = manage(new Toolbar);
  control_toolbar->set_toolbar_style(Gtk::TOOLBAR_ICONS);
  // retake
  m_retake_tool_button = manage(new ToolButton(Gtk::StockID("gtk-redo"))); // exposed in the class interface
  m_retake_tool_button->set_tooltip_text("Retake Slide (CTRL-<backspace>)");
  m_retake_tool_button->set_is_important();
  control_toolbar->append(*m_retake_tool_button);
  // previous
  m_previous_tool_button = manage(new ToolButton(Gtk::StockID("gtk-go-back"))); // exposed
  m_previous_tool_button->set_tooltip_text("Previous Slide (CTRL-<left arrow>)");
  m_previous_tool_button->set_is_important();
  control_toolbar->append(*m_previous_tool_button);
  // slide combo
  ToolItem* slide_combo_tool_item = manage(new ToolItem);
  slide_combo_tool_item->set_is_important();
  control_toolbar->append(*slide_combo_tool_item);
  m_slide_combo_box_text = manage(new ComboBoxText);  // exposed in the class interface
  slide_combo_tool_item->add(*m_slide_combo_box_text);
  // next
  m_next_tool_button = manage(new ToolButton(Gtk::StockID("gtk-go-forward"))); // exposed
  m_next_tool_button->set_tooltip_text("Next Slide (CTRL-<right arrow>)");
  m_next_tool_button->set_is_important();
  control_toolbar->append(*m_next_tool_button);
  // separator
  ToolItem* separator_1_tool_item = manage(new ToolItem);
  separator_1_tool_item->set_expand(true);
  control_toolbar->append(*separator_1_tool_item);
  // time
  ToolItem* time_tool_item = manage(new ToolItem);
  time_tool_item->set_is_important();
  control_toolbar->append(*time_tool_item);
  Gdk::Color LCD_green;
  LCD_green.set_rgb(0.80 * 65535, 0.81 * 65535, 0.68 * 65535);
  EventBox* record_time_event_box = manage(new EventBox());
  record_time_event_box->modify_bg(Gtk::STATE_NORMAL, LCD_green);
  time_tool_item->add(*record_time_event_box);
  Frame* record_time_frame = manage(new Frame());
  record_time_event_box->add(*record_time_frame);
  m_record_time_label = manage(new Label()); // exposed in class interface
  m_record_time_label->set_padding(2, 2);
  Pango::FontDescription font_description("monospace");
  m_record_time_label->modify_font(font_description);
  record_time_frame->add(*m_record_time_label);
  // mute
  m_mute_toggle_tool_button = manage(new ToggleToolButton); // exposed in class interface
  m_mute_toggle_tool_button->set_tooltip_text("Toggle Mute (CTRL-m)");
  m_mute_toggle_tool_button->set_is_important();
  m_mute_toggle_tool_button->set_icon_name("audio-volume-high");
  control_toolbar->append(*m_mute_toggle_tool_button);
  // pause
  m_pause_toggle_tool_button = manage(new ToggleToolButton); // exposed in class interface
  m_pause_toggle_tool_button->set_tooltip_text("Toggle Pause (CTRL-p)");
  m_pause_toggle_tool_button->set_is_important();
  m_pause_toggle_tool_button->set_icon_name("gtk-media-pause");
  control_toolbar->append(*m_pause_toggle_tool_button);
  // record
  //   icons
  const string& installation_dir = Application::application()->installation_dir();
  m_record_icon_start =
    new Image(installation_dir + "/kinetophone_record_start.png"); // exposed in class interface
  m_record_icon_stop =
    new Image(installation_dir + "/kinetophone_record_stop.png"); // exposed in class interface
  m_record_icon =
    manage(new Image(m_record_icon_start->get_pixbuf())); // exposed in class interface
    // button
  m_record_toggle_tool_button = manage(new ToggleToolButton); // exposed in class interface
  m_record_toggle_tool_button->set_tooltip_text("Toggle Record (CTRL-r)");
  m_record_toggle_tool_button->set_is_important();
  m_record_toggle_tool_button->set_icon_widget(*m_record_icon);
  control_toolbar->append(*m_record_toggle_tool_button);

  return control_toolbar;
}

Widget* Kinetophone_narrator_window::mf_setup_session_info_bar()
{
  m_session_info_bar = manage(new InfoBar); // exposed in class interface
  m_session_info_bar->set_message_type(Gtk::MESSAGE_INFO);
  m_session_info_bar->add_button(Gtk::Stock::OK, Gtk::RESPONSE_OK);
  m_session_info_bar->signal_response().
    connect(sigc::hide<0>(mem_fun(*m_session_info_bar, &Widget::hide)));
  Alignment* session_info_alignment = manage(new Alignment(0.0, 0.0, 0.0, 1.0));
  // why doesn't the info bar return a container rather that a widget?
  assert((g_type_from_name("GtkHBox") == G_OBJECT_TYPE(m_session_info_bar->get_content_area()->gobj()))
         && "error, content area of session info bar is not of type GtkHBox");
  Box* session_info_box =  reinterpret_cast<Box*>(m_session_info_bar->get_content_area());
  session_info_box->pack_start(*session_info_alignment, true, true);
  m_session_info_label = manage(new Label); // exposed in class interface
  m_session_info_label->set_selectable(true);
  m_session_info_label->set_line_wrap(false);
  session_info_alignment->add(*m_session_info_label);

  return m_session_info_bar;
}

Widget* Kinetophone_narrator_window::mf_setup_viewer()
{
  HPaned* viewer_hpaned = manage(new HPaned);
  viewer_hpaned->set_position(360);
  viewer_hpaned->set_can_focus(true);

  // image area
  VBox* image_vbox = manage(new VBox);
  viewer_hpaned->pack1(*image_vbox, true, false);
  //   label
  Label* image_label = manage(new Label("Image"));
  image_label->set_padding(1, 0);
  image_label->set_alignment(0.0, 0.0);
  image_vbox->pack_start(*image_label, Gtk::PACK_SHRINK);
  //   image
  Alignment* image_alignment = manage(new Alignment);
  image_vbox->pack_start(*image_alignment);
  m_image_aspectframe = manage(new AspectFrame("",
                                               0.0,
                                               0.0,
                                               1.0,
                                               false));
  m_image_aspectframe->set_shadow_type(Gtk::SHADOW_NONE);
  image_alignment->add(*m_image_aspectframe);
  m_slide_image_widget = manage(new Scaled_image_widget); // exposed in class interface
  m_image_aspectframe->add(*m_slide_image_widget);
  // notes area
  VBox* notes_vbox = manage(new VBox);
  viewer_hpaned->pack2(*notes_vbox, true, true);
  //   label
  HBox* notes_label_hbox = manage(new HBox);
  notes_vbox->pack_start(*notes_label_hbox, Gtk::PACK_SHRINK);
  Alignment* notes_alignment = manage(new Alignment(0.0, 0.0, 0.0, 0.0));
  notes_alignment->set_padding(0, 8, 0, 0);
  notes_label_hbox->pack_start(*notes_alignment, Gtk::PACK_SHRINK);
  Label* notes_label = manage(new Label("Notes"));
  notes_label->set_alignment(0.0, 0.0);
  notes_alignment->add(*notes_label);
  //   notes editing done button
  Alignment* notes_editing_done_alignment = manage(new Alignment(1.0, 1.0, 0.0, 0.0));
  notes_editing_done_alignment->set_padding(0, 1, 0, 0);
  notes_label_hbox->pack_start(*notes_editing_done_alignment);
  m_notes_editing_done_button = manage(new Button); // exposed in class interface
  notes_editing_done_alignment->add(*m_notes_editing_done_button);
  Label* notes_editing_done_label = manage(new Label("Done"));
  Pango::FontDescription notes_editing_done_font_description("8");
  notes_editing_done_label->modify_font(notes_editing_done_font_description);
  m_notes_editing_done_button->add(*notes_editing_done_label);

  Alignment* notes_widget_alignment = manage(new Alignment);
  notes_widget_alignment->set_padding(0, 0, 0, 2);
  notes_vbox->pack_start(*notes_widget_alignment);
  ScrolledWindow* notes_scrolled_window = manage(new ScrolledWindow);
  notes_scrolled_window->set_policy(Gtk::POLICY_NEVER, Gtk::POLICY_AUTOMATIC);
  notes_widget_alignment->add(*notes_scrolled_window);
  m_notes_text_view = manage(new TextView); // exposed in class interface
  m_notes_text_view->set_can_focus(true);
  m_notes_text_view->set_accepts_tab(false);
  m_notes_text_view->set_wrap_mode(Gtk::WRAP_WORD);
  notes_scrolled_window->add(*m_notes_text_view);

  return viewer_hpaned;
}

Widget* Kinetophone_narrator_window::mf_setup_accessories(const Kinetophone_narrator_config& config)
{
  HBox* accessories_hbox = manage(new HBox);
  Label* space_label = manage(new Label("Space"));
  accessories_hbox->pack_start(*space_label, Gtk::PACK_SHRINK);
  // space available widget
  m_space_available_widget = manage(new Fraction_bar_widget(true, // exposed in class interface
                                                            false,
                                                            -1.0,
                                                            -1.0,
                                                            -1.0,
                                                            -1.0,
                                                            -1,
                                                            -1,
                                                            3));
  accessories_hbox->pack_start(*m_space_available_widget, Gtk::PACK_SHRINK);
  // space/time label
  m_space_time_label = manage(new Label("", 0.0, 0.5)); // exposed in class interface
  accessories_hbox->pack_start(*m_space_time_label);
  // level toggle button
  Alignment* level_alignment = manage(new Alignment(0.5, 0.5, 0.0, 0.0));
  level_alignment->set_padding(4, 4, 0, 0);
  accessories_hbox->pack_start(*level_alignment, Gtk::PACK_SHRINK);
  m_level_toggle_button = manage(new ToggleButton("_Level", true)); // exposed in class interface
  m_level_toggle_button->set_tooltip_text("Toggle Level Display");
  level_alignment->add(*m_level_toggle_button);
  // level meters
  m_audio_level_meter_widget = manage(new Level_meter_widget(2,  // exposed in class interface
                                                             config.update_rate,
                                                             false,
                                                             true,
                                                             true,
                                                             config.level_meter.model_ballistics,
                                                             config.level_meter.attack_period,
                                                             config.level_meter.decay_period,
                                                             config.level_meter.peak_hold_period,
                                                             -1,
                                                             -1,
                                                             -1,
                                                             -1,
                                                             3));
  //   there is always at least 1 channel, check for number 2
  //   (channels over number 2 will not be shown)
  if (config.sound_recorder_config->sound_file->channels < 2)
  {
    m_audio_level_meter_widget->set_enable_channel(1, false);
  }
  accessories_hbox->pack_start(*m_audio_level_meter_widget, Gtk::PACK_SHRINK);

  return accessories_hbox;
}

Widget* Kinetophone_narrator_window::mf_setup_thumbnail_browser()
{
  ScrolledWindow* thumbnail_browser_scrolled_window = manage(new ScrolledWindow);
  thumbnail_browser_scrolled_window->set_policy(Gtk::POLICY_AUTOMATIC, Gtk::POLICY_NEVER);
  thumbnail_browser_scrolled_window->set_shadow_type(Gtk::SHADOW_IN);
  m_thumbnail_browser_hbutton_box = manage(new HButtonBox); // exposed in class interface
  m_thumbnail_browser_hbutton_box->set_spacing(1);
  m_thumbnail_browser_hbutton_box->set_homogeneous(true);
  m_thumbnail_browser_hbutton_box->set_layout(Gtk::BUTTONBOX_START);
  // autoscroll the thumbnail
  m_thumbnail_browser_hbutton_box->set_focus_hadjustment(*thumbnail_browser_scrolled_window->get_hadjustment());
  thumbnail_browser_scrolled_window->add(*m_thumbnail_browser_hbutton_box);

  return thumbnail_browser_scrolled_window;
}

Widget* Kinetophone_narrator_window::mf_setup_statusbar()
{
  m_statusbar = manage(new Statusbar); // exposed in class interface
  m_statusbar->set_spacing(2);

  return m_statusbar;
}

void Kinetophone_narrator_window::mf_setup_accelerators()
{
  RefPtr<AccelGroup> window_accel_group = get_accel_group();

  m_retake_tool_button->add_accelerator("clicked",
                                        window_accel_group,
                                        GDK_BackSpace,
                                        Gdk::CONTROL_MASK,
                                        Gtk::ACCEL_VISIBLE);
  m_previous_tool_button->add_accelerator("clicked",
                                          window_accel_group,
                                          GDK_Left,
                                          Gdk::CONTROL_MASK,
                                          Gtk::ACCEL_VISIBLE);
  m_next_tool_button->add_accelerator("clicked",
                                      window_accel_group,
                                      GDK_Right,
                                      Gdk::CONTROL_MASK,
                                      Gtk::ACCEL_VISIBLE);
}
