// Kinetophone_narrator_view.cpp
//
// Copyright 2011-2012 Roan Trail, Inc.
//
// This file is part of Kinetophone.
//
// Kinetophone is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published
// by the Free Software Foundation, either version 2 of the License,
// or (at your option) any later version.
//
// Kinetophone is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// General Public License for more details.  You should have received
// a copy of the GNU General Public License along with Kinetophone. If
// not, see <http://www.gnu.org/licenses/>.

#include "Kinetophone_narrator_view.hpp"

// include *mm first to avoid conflicts
#include <gtkmm/aboutdialog.h>
#include <gtkmm/aspectframe.h>
#include <gtkmm/comboboxtext.h>
#include <gtkmm/buttonbox.h>
#include <gtkmm/filechooserdialog.h>
#include <gtkmm/image.h>
#include <gtkmm/infobar.h>
#include <gtkmm/label.h>
#include <gtkmm/messagedialog.h>
#include <gtkmm/statusbar.h>
#include <gtkmm/stock.h>
#include <gtkmm/textview.h>
#include <gtkmm/togglebutton.h>
#include <gtkmm/toggletoolbutton.h>

#include "Kinetophone_narrator_model.hpp"
#include "Kinetophone_narrator_config.hpp"
#include "Kinetophone_narrator_window.hpp"
#include "Kinetophone_narrator_error_dialog.hpp"
#include "../base/Narration_recorder.hpp"
#include "../base/Slide_collection.hpp"
#include "../base/Sound_recorder_config.hpp"
#include "../base/Sound_file_config.hpp"
#include "Level_meter_widget.hpp"
#include "Fraction_bar_widget.hpp"
#include "Scaled_image_widget.hpp"
#include "../base/Slide_collection.hpp"
#include "../base/Device_enum.hpp"
#include "../base/error/Kinetophone_error.hpp"
#include "../base/error/Curses_error.hpp"
#include <iostream>
#include <iomanip>
#include <sstream>
#include <string>
#include <utility>
#include <vector>

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#ifndef PACKAGE_VERSION
#define PACKAGE_VERSION "unknown"
#endif

#ifndef PACKAGE_URL
#define PACKAGE_URL "unknown"
#endif

using Glib::RefPtr;
using Gdk::Pixbuf;
using Gtk::AboutDialog;
using Gtk::Clipboard;
using Gtk::ComboBoxText;
using Gtk::FileChooserDialog;
using Gtk::HButtonBox;
using Gtk::InfoBar;
using Gtk::Image;
using Gtk::MessageDialog;
using Gtk::TextView;
using Gtk::ToggleButton;
using Gtk::ToggleToolButton;
using std::cout;
using std::cerr;
using std::endl;
using std::pair;
using std::string;
using std::stringstream;
using std::vector;
using sigc::bind;
using sigc::connection;
using sigc::mem_fun;
using Roan_trail::Error_param;
using Roan_trail::Source::Slide_collection;
using Roan_trail::Recorder::Device_enum;
using Roan_trail::Recorder::Narration_recorder;
using Roan_trail::Recorder::Sound_recorder_config;
using namespace Roan_trail::Kinetophone;

//
// Internal variables
//

namespace
{
  Error_param iv_ignore_error(false);
}

//
// Constructor/destructor
//

Kinetophone_narrator_view::Kinetophone_narrator_view(const Kinetophone_narrator_model& model,
                                                     const Kinetophone_narrator_config& config)
  : m_model(model),
    m_config(config),
    m_main_window(new Kinetophone_narrator_window(config)),
    m_setup(false),
    m_levels(),
    m_file_size(),
    m_audio_info(),
    m_overflow_count(),
    m_retake_signal(),
    m_previous_signal(),
    m_slide_signal(),
    m_next_signal(),
    m_mute_signal(),
    m_pause_signal(),
    m_record_signal(),
    m_toggle_metering_signal(),
    m_reset_overflow_counts_signal(),
    m_reset_clipped_signal(),
    m_notes_changed_signal(),
    m_notes_editing_done_signal(),
    m_export_session_signal(),
    m_quit_signal(),
    m_pause_toggle_connection(),
    m_slide_connection(),
    m_slide_notes_connection(),
    m_thumbnail_browser_toggle()
{
  m_levels.resize(min(m_model.narration_recorder()->config()->sound_file->channels, 2));
  mf_setup_slide_combo();
  mf_setup_thumbnail_browser();
  mf_format_audio_info_string();
  mf_connect_signals();

  postcondition(mf_invariant(false));
}

Kinetophone_narrator_view::~Kinetophone_narrator_view()
{
  precondition(mf_invariant(false));

  delete m_main_window;
}

//
// Updates
//

void Kinetophone_narrator_view::model_slides_will_update()
{
  precondition(mf_invariant());

  const Narration_recorder* recorder = m_model.narration_recorder();
  if (Roan_trail::Recorder::Recorder::state_stopped == recorder->state())
  {
    mf_notes_editing_done();
  }

  const Slide_collection<RefPtr<Pixbuf> >* slides = m_model.slides();
  const int slide_index = slides->current_slide_index();

  // make current slide's button not active (not pushed in)
  pair<ToggleButton*, connection> &toggle_pair = m_thumbnail_browser_toggle[slide_index];
  toggle_pair.second.block(); // temporary block to prevent spurious update
  toggle_pair.first->set_active(false);
  toggle_pair.second.unblock(); // enable the connection again

  postcondition(mf_invariant());
}

void Kinetophone_narrator_view::model_recorder_will_update()
{
  precondition(mf_invariant());

  const Narration_recorder* recorder = m_model.narration_recorder();
  if (Roan_trail::Recorder::Recorder::state_stopped == recorder->state())
  {
    mf_notes_editing_done();
  }

  postcondition(mf_invariant());
}

void Kinetophone_narrator_view::model_slides_updated()
{
  precondition(mf_invariant());

  const Slide_collection<RefPtr<Pixbuf> >* slides = m_model.slides();
  const int slide_index = slides->current_slide_index();

  // update slide image
  RefPtr<Pixbuf> image;
  const Gdk::Rectangle& image_rect = m_main_window->slide_image_widget()->image_rect();
  const Rect_size image_size(image_rect.get_width(), image_rect.get_height());
  slides->image_for_slide_at(slide_index,
                             image_size,
                             image);
  m_main_window->image_aspectframe()->set(Gtk::ALIGN_CENTER,
                                          Gtk::ALIGN_CENTER,
                                          static_cast<float>(image->get_width())
                                          / static_cast<float>(image->get_height()),
                                          false);
  m_main_window->slide_image_widget()->set_image(image);

  // update slide combo box
  ComboBoxText* slide_combo = m_main_window->slide_combo_box_text();

  m_slide_connection.block(); // temporary block to prevent spurious update
  slide_combo->set_active(slide_index);
  m_slide_connection.unblock(); // enable the connection again

  // update slide notes
  TextView* notes_text_view = m_main_window->notes_text_view();
  const string& slide_notes = (*slides)[slide_index]->notes();
  m_slide_notes_connection.block(); // temporary block to prevent spurious update
  notes_text_view->get_buffer()->set_text(slide_notes);
  m_slide_notes_connection.unblock(); // enable the connection again

  // update the thumbnail browser
  HButtonBox* thumbnail_browser = m_main_window->thumbnail_browser_hbutton_box();
  ToggleButton* current_browser_button = m_thumbnail_browser_toggle[slide_index].first;
  //   make sure the current slide's button can be seen
  thumbnail_browser->set_focus_child(*current_browser_button);
  m_thumbnail_browser_toggle[slide_index].second.block(); // temporary block to prevent spurious update
  //   make the current slide's button active (pushed in)
  current_browser_button->set_active(true);
  m_thumbnail_browser_toggle[slide_index].second.unblock(); // enable the connection again

  postcondition(mf_invariant());
}

void Kinetophone_narrator_view::model_recorder_updated()
{
  precondition(mf_invariant());

  const Narration_recorder* recorder = m_model.narration_recorder();

  // TODO: make sure the "gtk-*" icons are available for install

  const int recorder_state = recorder->state();
  if (Roan_trail::Recorder::Recorder::state_stopped == recorder_state)
  {
    // menu
    m_main_window->file_export_session_menu_item()->set_sensitive();
    m_main_window->file_quit_menu_item()->set_sensitive();
    m_main_window->help_about_menu_item()->set_sensitive();
    // buttons
    m_main_window->retake_tool_button()->set_sensitive(false);
    ToggleToolButton* pause_button = m_main_window->pause_toggle_tool_button();
    pause_button->set_sensitive(false);
    m_pause_toggle_connection.block(); // temporary block to prevent spurious update
    pause_button->set_active(false);
    m_pause_toggle_connection.unblock(); // enable the connection again
    ToggleToolButton* record_button = m_main_window->record_toggle_tool_button();
    Image* window_start_icon = m_main_window->record_icon_start();
    m_main_window->record_icon()->set(window_start_icon->get_pixbuf());
    bool can_record = recorder->can_record();
    record_button->set_sensitive(can_record);
    m_main_window->mute_toggle_tool_button()->set_sensitive(can_record);
    m_main_window->level_toggle_button()->set_sensitive(can_record);
    m_main_window->audio_level_meter_widget()->set_sensitive(can_record);
    // notes
    m_main_window->notes_text_view()->set_editable();
  }
  else if (Roan_trail::Recorder::Recorder::state_paused == recorder_state)
  {
    // buttons
    m_main_window->retake_tool_button()->set_sensitive(false);
    m_main_window->pause_toggle_tool_button()->set_sensitive();
  }
  else if (Roan_trail::Recorder::Recorder::state_recording == recorder_state)
  {
    // menu
    m_main_window->file_export_session_menu_item()->set_sensitive(false);
    m_main_window->file_quit_menu_item()->set_sensitive(false);
    m_main_window->help_about_menu_item()->set_sensitive(false);
    // buttons
    m_main_window->retake_tool_button()->set_sensitive();
    m_main_window->pause_toggle_tool_button()->set_sensitive();
    Image* window_stop_icon = m_main_window->record_icon_stop();
    m_main_window->record_icon()->set(window_stop_icon->get_pixbuf());
    // session info
    m_main_window->session_info_bar()->hide();
    // notes
    m_main_window->notes_text_view()->set_editable(false);
  }
  else
  {
    assert(false && "error invalid recorder state when updating UI");
  }

  postcondition(mf_invariant());
}

void Kinetophone_narrator_view::update_levels()
{
  precondition(mf_invariant());

  const Narration_recorder* recorder = m_model.narration_recorder();

  recorder->sound_levels_RMS(m_levels);

  m_main_window->audio_level_meter_widget()->set_levels(m_levels);

  postcondition(mf_invariant());
}

void Kinetophone_narrator_view::update_available_space()
{
  precondition(mf_invariant());

  const Narration_recorder* recorder = m_model.narration_recorder();

  Long_int bytes_available = 0;
  double fraction_available = 0.0;
  double seconds_remaining = 0.0;
  recorder->space_available(bytes_available,
                            fraction_available,
                            seconds_remaining);

  // make 1MB reserve
  if (bytes_available > 1000000)
  {
    bytes_available -= 1000000;
  }
  else
  {
    bytes_available = 0;
  }

  stringstream space_time_stream;
  space_time_stream.setf(std::ios_base::fixed, std::ios_base::floatfield);
  space_time_stream.precision(3);
  if (bytes_available < 1000000000)
  {
    space_time_stream << "(" << bytes_available / 1000000.0 << " MB";
  }
  else
  {
    space_time_stream << "(" << bytes_available / 1000000000.0 << " GB";
  }

  space_time_stream.precision(0);
  if (seconds_remaining < 60.0)
  {
    space_time_stream << " / " << seconds_remaining << " Sec)";
  }
  else if (seconds_remaining < 3600.0)
  {
    space_time_stream << " / " << seconds_remaining / 60.0 << " Min)";
  }
  else
  {
    space_time_stream << " / " << seconds_remaining / 3600.0 << " Hr ";
    space_time_stream << fmod(seconds_remaining, 3600.0) / 60.0 << " Min)";
  }
  m_main_window->space_time_label()->set_text(space_time_stream.str());
  m_main_window->space_available_widget()->set_fraction(fraction_available);

  // TODO: click record button if no space available

  postcondition(mf_invariant());
}

void Kinetophone_narrator_view::update_statusbar()
{
  precondition(mf_invariant());

  if (m_config.show_file_size || m_config.show_audio_info || m_config.show_overflow_count)
  {
    mf_update_file_size();
    mf_update_overflow_count();

    string statusbar = m_audio_info;
    if (m_overflow_count.size())
    {
      if (m_audio_info.size())
      {
        statusbar += " | ";
      }
      statusbar += m_overflow_count;
      if (m_file_size.size())
      {
        statusbar += " | ";
        statusbar += m_file_size;
      }
    }
    else
    {
      if (m_audio_info.size())
      {
        statusbar += " | ";
      }
      statusbar += m_file_size;
    }

    m_main_window->statusbar()->pop();
    m_main_window->statusbar()->push(statusbar);
  }

  postcondition(mf_invariant());
}

void Kinetophone_narrator_view::update_time(bool is_flashing)
{
  precondition(mf_invariant());

  Long_int total_frames = 0;
  Long_int slide_frames = 0;

  const Narration_recorder* recorder = m_model.narration_recorder();
  recorder->frames(total_frames,
                   slide_frames,
                   iv_ignore_error);

  stringstream record_time;
  if (Roan_trail::Recorder::Recorder::state_stopped == recorder->state())
  {
    record_time << "Total: ";
    if (!recorder->have_recording())
    {
      record_time << "--:--:--";
    }
    else
    {
      string total_record_time;
      recorder->recorded_time(total_record_time);
      record_time << total_record_time;
    }
    record_time << "\n";
    record_time << "Slide: --:--:--";
  }
  else
  {
    string total_record_time;
    string slide_record_time;
    if (is_flashing)
    {
      recorder->recorded_time(total_record_time, ' ');
      recorder->recorded_time_for_current_index(slide_record_time, ' ');
    }
    else
    {
      recorder->recorded_time(total_record_time, ':');
      recorder->recorded_time_for_current_index(slide_record_time, ':');
    }
    record_time << "Total: " << total_record_time << "\nSlide: " << slide_record_time;
  }

  m_main_window->record_time_label()->set_text(record_time.str());

  postcondition(mf_invariant());
}

//
// Commands
//

void Kinetophone_narrator_view::toggle_metering()
{
  precondition(mf_invariant());

  const Narration_recorder* recorder = m_model.narration_recorder();
  m_main_window->audio_level_meter_widget()->set_sensitive(recorder->is_RMS_metering_enabled());

  postcondition(mf_invariant());
}

void Kinetophone_narrator_view::reset_clipped()
{
  precondition(mf_invariant());

  m_main_window->audio_level_meter_widget()->reset_clipped();

  postcondition(mf_invariant());
}

void Kinetophone_narrator_view::show_session_info(const string& message)
{
  m_main_window->session_info_label()->set_text(message);
  m_main_window->session_info_bar()->show();
}

int Kinetophone_narrator_view::show_error(const Error& error, bool is_warning)
{
  Kinetophone_narrator_error_dialog error_dialog(error,
                                                 *m_main_window,
                                                 is_warning);
  error_dialog.add_button(Gtk::Stock::OK, Gtk::RESPONSE_OK);
  error_dialog.set_default_response(Gtk::RESPONSE_OK);
  int return_value = error_dialog.run();

  return return_value;
}

bool Kinetophone_narrator_view::ask_overwrite_recording()
{
  bool return_value = false;

  MessageDialog dialog(*m_main_window,
                       "A recording already exists, overwrite?",
                       false,
                       Gtk::MESSAGE_QUESTION,
                       Gtk::BUTTONS_YES_NO,
                       true);
  const string recording_file = m_model.narration_recorder()->output_file();
  string message = string("(File: " + recording_file + ")");
  dialog.set_secondary_text(message);
  dialog.set_default_response(Gtk::RESPONSE_NO);
  const int result = dialog.run();
  switch(result)
  {
  case Gtk::RESPONSE_YES:
    return_value = true;
    break;
  default:
    // do nothing
    break;
  }

  return return_value;
}

void Kinetophone_narrator_view::output_console(const string& message)
{
  cout << message;
}

void Kinetophone_narrator_view::output_error(const string& message)
{
  cerr << message;
}

bool Kinetophone_narrator_view::list_devices(Error_param& return_error)
{
  precondition(!return_error());

  bool return_value = false;

  start_error_block();

  string device_info_string;
  Error_param error;
  bool got_info = Device_enum::list_devices(paFloat32, // Note: paFloat32 declaration uses an old-style cast
                                            device_info_string,
                                            error);
  on_error(!got_info, new Kinetophone_error(error_location(),
                                            Kinetophone_error::list_devices,
                                            error()));

  cout << device_info_string;

  return_value = true;
  goto exit_point;

  end_error_block();

  default_error_handler_and_cleanup(return_error,
                                    return_value,
                                    false);
  goto exit_point;

 exit_point:
  return return_value;
}

//
// Protected member functions
//

bool Kinetophone_narrator_view::mf_invariant(bool check_base_class) const
{
  bool return_value = false;

  // check data member allocations
  if (!m_main_window)
  {
    goto exit_point;
  }

  if (!m_levels.size())
  {
    goto exit_point;
  }


  if (m_setup)
  {
    // all signals should be connected and connections active after setup
    if (m_retake_signal.empty()
        || m_previous_signal.empty()
        || m_slide_signal.empty()
        || m_next_signal.empty()
        || m_mute_signal.empty()
        || m_pause_signal.empty()
        || m_record_signal.empty()
        || m_toggle_metering_signal.empty()
        || m_reset_overflow_counts_signal.empty()
        || m_reset_clipped_signal.empty()
        || m_notes_changed_signal.empty()
        || m_notes_editing_done_signal.empty()
        || m_export_session_signal.empty()
        || m_quit_signal.empty()
        || m_pause_toggle_connection.empty()
        || m_slide_connection.empty()
        || m_slide_notes_connection.empty())
    {
      goto exit_point;
    }
  }

  return_value = true;

 exit_point:
  return return_value;
}

//
// Private member functions
//

void Kinetophone_narrator_view::mf_format_audio_info_string()
{
  if (m_config.show_audio_info)
  {
    const Sound_recorder_config* config = m_config.sound_recorder_config;
    stringstream s;
    s << "DEV " << config->input_device << " | ";
    s << config->sound_file->format_string_brief() << " | ";
    double VBR_quality;
    if (config->sound_file->has_VBR(VBR_quality))
    {
      s << "VBRQ=" << VBR_quality << " | ";
    }
    s << "FPB=" << config->frames_per_buffer << " | ";
    s << "WBF=" << config->write_buffer_factor << " | ";
    s << (config->file_overwrite ? "OVWRT" : "NO OVWRT");
    m_audio_info = s.str();
  }
}


string& Kinetophone_narrator_view::mf_update_file_size()
{
  if (m_config.show_file_size)
  {
    Long_int file_size = 0;
    double data_rate = 0.0;

    const Narration_recorder* recorder = m_model.narration_recorder();
    recorder->output_file_size(file_size, data_rate);
    stringstream file_size_stream;

    file_size_stream << "SIZE: ";
    file_size_stream.setf(std::ios_base::fixed, std::ios_base::floatfield);
    if (file_size < 1000L)
    {
      file_size_stream.precision(0);
      file_size_stream << file_size << " B";
    }
    else if (file_size < 1000000L)
    {
      file_size_stream.precision(0);
      file_size_stream << static_cast<double>(file_size) / 1000.0 << " KB";
    }
    else if (file_size < 1000000000L)
    {
      file_size_stream.precision(2);
      file_size_stream << static_cast<double>(file_size) / 1000000.0 << " MB";
    }
    else
    {
      file_size_stream.precision(3);
      file_size_stream << static_cast<double>(file_size) / 1000000000.0 << " GB";
    }

    file_size_stream.precision(2);
    file_size_stream << " (" << data_rate / 1000.0 << " KB/SEC)";

    m_file_size = file_size_stream.str();
  }

  return m_file_size;
}

string& Kinetophone_narrator_view::mf_update_overflow_count()
{
  if (m_config.show_overflow_count)
  {
    const Narration_recorder* recorder = m_model.narration_recorder();
    stringstream overflow;
    Long_int input_overflow_count = recorder->input_overflow_count();
    overflow << "OFLW IN: " << std::setw(2) << std::setfill('0') << input_overflow_count % 100;
    if (input_overflow_count > 100)
    {
      overflow << "+ ";
    }
    else
    {
      overflow << "  ";
    }
    Long_int output_overflow_count = recorder->output_overflow_count();
    overflow <<  "/ OUT: " <<std::setw(2) << std::setfill('0') << output_overflow_count % 100;
    if (output_overflow_count > 100)
    {
      overflow << "+";
    }
    else
    {
      overflow << " ";
    }
    m_overflow_count = overflow.str();
  }

  return m_overflow_count;
}

void Kinetophone_narrator_view::mf_setup_slide_combo()
{
  ComboBoxText* slide_combo = m_main_window->slide_combo_box_text();
  const Slide_collection<RefPtr<Pixbuf> >* slides = m_model.slides();
  for (int i = 0; i < slides->slide_count(); ++i)
  {
    stringstream item_title;
    item_title << "Slide " << (i + 1);
    slide_combo->append_text(item_title.str());
  }
  slide_combo->set_active(0);
}

void Kinetophone_narrator_view::mf_connect_signals()
{
  // connect signals which need simple "forwarding"
  m_main_window->retake_tool_button()->signal_clicked().connect(m_retake_signal.make_slot());
  m_main_window->previous_tool_button()->signal_clicked().connect(m_previous_signal.make_slot());
  m_main_window->next_tool_button()->signal_clicked().connect(m_next_signal.make_slot());
  m_pause_toggle_connection = m_main_window->pause_toggle_tool_button()->signal_toggled().
    connect(m_pause_signal.make_slot());
  m_main_window->record_toggle_tool_button()->signal_toggled().connect(m_record_signal.make_slot());
  m_main_window->level_toggle_button()->signal_toggled().connect(m_toggle_metering_signal.make_slot());
  // connect signals which need custom slots
  m_main_window->signal_key_press_event().
    connect(mem_fun(*this, &Kinetophone_narrator_view::mf_main_window_key_pressed));
  m_main_window->mute_toggle_tool_button()->signal_toggled().
    connect(mem_fun(*this, &Kinetophone_narrator_view::mf_mute));
  m_slide_connection = m_main_window->slide_combo_box_text()->signal_changed().
    connect(mem_fun(*this, &Kinetophone_narrator_view::mf_slide));
  m_slide_notes_connection = m_main_window->notes_text_view()->get_buffer()->signal_changed().
    connect(mem_fun(*this, &Kinetophone_narrator_view::mf_notes_changed));
  m_main_window->notes_editing_done_button()->signal_clicked().
    connect(mem_fun(*this, &Kinetophone_narrator_view::mf_notes_editing_done));
  m_main_window->file_export_session_menu_item()->signal_activate().
    connect(mem_fun(*this, &Kinetophone_narrator_view::mf_export_session));
  m_main_window->file_quit_menu_item()->signal_activate().
    connect(mem_fun(*this, &Kinetophone_narrator_view::mf_quit));
  m_main_window->edit_cut_menu_item()->signal_activate().
    connect(mem_fun(*this, &Kinetophone_narrator_view::mf_cut));
  m_main_window->edit_copy_menu_item()->signal_activate().
    connect(mem_fun(*this, &Kinetophone_narrator_view::mf_copy));
  m_main_window->edit_paste_menu_item()->signal_activate().
    connect(mem_fun(*this, &Kinetophone_narrator_view::mf_paste));
  m_main_window->help_about_menu_item()->signal_activate().
    connect(mem_fun(*this, &Kinetophone_narrator_view::mf_about));
  m_main_window->signal_delete_event().
     connect(mem_fun(*this, &Kinetophone_narrator_view::mf_delete_main_window));

  m_main_window->setup();
}

void Kinetophone_narrator_view::mf_setup_thumbnail_browser()
{
  HButtonBox* thumbnail_browser = m_main_window->thumbnail_browser_hbutton_box();
  const Slide_collection<RefPtr<Pixbuf> >* slides = m_model.slides();
  m_thumbnail_browser_toggle.clear();
  const int slide_count = slides->slide_count();
  m_thumbnail_browser_toggle.reserve(slide_count);
  for (int i = 0; i < slide_count; ++i)
  {
    stringstream item_title;
    item_title << "Slide " << (i + 1);
    ToggleButton* browser_button = manage(new ToggleButton(item_title.str()));
    browser_button->modify_bg(Gtk::STATE_NORMAL,
                              thumbnail_browser->get_style()->get_bg(Gtk::STATE_NORMAL));
    browser_button->set_can_focus(false);
    RefPtr<Pixbuf> thumbnail_pixbuf;
    slides->thumbnail_for_slide_at(i, thumbnail_pixbuf);
    Image* thumbnail_image = manage(new Image(thumbnail_pixbuf));
    browser_button->set_image_position(Gtk::POS_TOP);
    browser_button->set_image(*thumbnail_image);
    connection current_connection = browser_button->signal_toggled().
      connect(bind<0>(m_thumbnail_choose_signal.make_slot(), i));
    m_thumbnail_browser_toggle.push_back(std::make_pair(browser_button, current_connection));
    thumbnail_browser->add(*browser_button);
    thumbnail_image->show();
    browser_button->show();
  }
}

//
// slots
//

bool Kinetophone_narrator_view::mf_main_window_key_pressed(GdkEventKey* event)
{
  bool return_value = false;

  const guint modifiers = gtk_accelerator_get_default_mod_mask();

  if (GDK_CONTROL_MASK == (event->state & modifiers))
  {
    switch (event->keyval)
    {
    case GDK_l: // CTRL-l resets level meter clipped indicator
      m_reset_clipped_signal.emit();
      return_value = true;
      break;
    case GDK_m: // CTRL-m toggles mute
      {
        ToggleToolButton* mute_button = m_main_window->mute_toggle_tool_button();
        if (mute_button->get_sensitive())
        {
          // simulate click of button, because the mute button isn't updated when the recorder
          // is updated
          mute_button->set_active(!mute_button->get_active());
        }
      }
      return_value = true;
      break;
    case GDK_o: // CTRL-o resets overflow counts
      m_reset_overflow_counts_signal.emit();
      return_value = true;
      break;
    case GDK_p: // CTRL-p toggles pause
      {
        ToggleToolButton* pause_button = m_main_window->pause_toggle_tool_button();
        if (pause_button->get_sensitive())
        {
          // simulate click of button, because the pause button isn't updated when the recorder
          // is updated
          pause_button->set_active(!pause_button->get_active());
        }
      }
      return_value = true;
      break;
    case GDK_r: // CTRL-r toggles recording
      if (m_main_window->record_toggle_tool_button()->get_sensitive())
      {
        // just need to emit the record signal, because the record button is updated when the recorder
        // is updated
        m_record_signal.emit();
      }
      return_value = true;
      break;
    default:
      break;
    }
  }

  return return_value;
}

void Kinetophone_narrator_view::mf_slide()
{
  int slide_index = m_main_window->slide_combo_box_text()->get_active_row_number();
  m_slide_signal.emit(slide_index);
}

void Kinetophone_narrator_view::mf_mute()
{
  m_mute_signal.emit();

  // TODO: make sure the "audio-volumes-*" icons are available for install
  if (m_model.narration_recorder()->is_muted())
  {
    m_main_window->mute_toggle_tool_button()->set_icon_name("audio-volume-muted");
  }
  else
  {
    m_main_window->mute_toggle_tool_button()->set_icon_name("audio-volume-high");
  }
}

void Kinetophone_narrator_view::mf_notes_changed()
{
  const Narration_recorder* recorder = m_model.narration_recorder();
  if (Roan_trail::Recorder::Recorder::state_stopped == recorder->state())
  {
    m_main_window->notes_editing_done_button()->show();
    m_notes_changed_signal.emit();
  }
}

void Kinetophone_narrator_view::mf_notes_editing_done()
{
  const string& current_notes = m_main_window->notes_text_view()->get_buffer()->get_text();
  m_main_window->notes_editing_done_button()->hide();
  m_main_window->set_focus_child(*m_main_window->previous_tool_button());
  m_notes_editing_done_signal.emit(current_notes);
}

void Kinetophone_narrator_view::mf_export_session()
{
  mf_notes_editing_done();

  FileChooserDialog dialog(*m_main_window,
                           "Choose a file to export session (in XML):",
                           Gtk::FILE_CHOOSER_ACTION_SAVE);
  dialog.set_do_overwrite_confirmation();
  dialog.add_button(Gtk::Stock::CANCEL, Gtk::RESPONSE_CANCEL);
  dialog.add_button("Select", Gtk::RESPONSE_OK);
  dialog.set_default_response(Gtk::RESPONSE_OK);

  const int result = dialog.run();
  switch(result)
  {
  case Gtk::RESPONSE_OK:
    m_export_session_signal.emit(dialog.get_filename());
    break;
  default:
    // do nothing
    break;
  }
}

bool Kinetophone_narrator_view::mf_delete_main_window(GdkEventAny* event)
{
  static_cast<void>(event); // avoid unused warning

  bool return_value = true;

  // don't allow window delete if recording
  const Narration_recorder* recorder = m_model.narration_recorder();
  if (Roan_trail::Recorder::Recorder::state_stopped == recorder->state())
  {
    return_value = !mf_verify_quit();
  }

  return return_value;
}

void Kinetophone_narrator_view::mf_quit()
{
  if (mf_verify_quit())
  {
    m_quit_signal.emit();
  }
}

void Kinetophone_narrator_view::mf_cut()
{
  m_main_window->notes_text_view()->get_buffer()->cut_clipboard(Clipboard::get());
}

void Kinetophone_narrator_view::mf_copy()
{
  m_main_window->notes_text_view()->get_buffer()->copy_clipboard(Clipboard::get());
}

void Kinetophone_narrator_view::mf_paste()
{
  m_main_window->notes_text_view()->get_buffer()->paste_clipboard(Clipboard::get());
}

void Kinetophone_narrator_view::mf_about()
{
  AboutDialog about_dialog;
  about_dialog.set_transient_for(*m_main_window);
  about_dialog.set_program_name("Kinetophone Narrator");
  about_dialog.set_copyright("Copyright 2011-2012 Roan Trail, Inc.");

  stringstream license;
  license << "Kinetophone is free software: you can redistribute it and/or modify" << endl;
  license << "it under the terms of the GNU General Public License as published" << endl;
  license << "by the Free Software Foundation, either version 2 of the License," << endl;
  license << "or (at your option) any later version." << endl;
  license << endl;
  license << "Kinetophone is distributed in the hope that it will be useful, but" << endl;
  license << "WITHOUT ANY WARRANTY; without even the implied warranty of" << endl;
  license << "MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU" << endl;
  license << "General Public License for more details.  You should have received" << endl;
  license << "a copy of the GNU General Public License along with Kinetophone. If" << endl;
  license << "not, see <http://www.gnu.org/licenses/>." << endl;
  about_dialog.set_license(license.str());

  about_dialog.set_version(PACKAGE_VERSION);
  about_dialog.set_comments("Program to narrate over still images.");
  about_dialog.set_website(PACKAGE_URL);
  about_dialog.set_website_label("Kinetophone Website");
  vector<string> authors;
  authors.push_back("Roan Trail, Inc. Development Team");
  about_dialog.set_authors(authors);
  RefPtr<Pixbuf> logo;
  about_dialog.set_logo(logo);
  about_dialog.set_wrap_license(false);

  about_dialog.run();
}

//
// Helpers
//

bool Kinetophone_narrator_view::mf_verify_quit()
{
  bool return_value = false;

  if (m_model.is_dirty())
  {
    string message = "You have not exported changes to your session.  ";
    message += "Do you want to quit without saving?";
    MessageDialog dialog(*m_main_window,
                         "There are unsaved changes.",
                         false,
                         Gtk::MESSAGE_QUESTION,
                         Gtk::BUTTONS_YES_NO,
                         true);
    dialog.set_secondary_text(message);
    dialog.set_default_response(Gtk::RESPONSE_NO);
    const int result = dialog.run();
    switch(result)
    {
    case Gtk::RESPONSE_YES:
      return_value = true;
      break;
    default:
      // do nothing
      break;
    }
  }
  else
  {
    return_value = true;
  }

  return return_value;
}
