// Kinetophone_narrator_controller.cpp
//
// Copyright 2011-2012 Roan Trail, Inc.
//
// This file is part of Kinetophone.
//
// Kinetophone is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published
// by the Free Software Foundation, either version 2 of the License,
// or (at your option) any later version.
//
// Kinetophone is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// General Public License for more details.  You should have received
// a copy of the GNU General Public License along with Kinetophone. If
// not, see <http://www.gnu.org/licenses/>.

#include "Kinetophone_narrator_controller.hpp"

// include *mm first to avoid conflicts
#include <glibmm/main.h>
#include <gtkmm/main.h>
#include <gdkmm/pixbuf.h>

#include "Kinetophone_narrator_model.hpp"
#include "Kinetophone_narrator_view.hpp"
#include "Kinetophone_narrator_config.hpp"
#include "../base/Narration_recorder.hpp"
#include "../base/Slide_collection.hpp"
#include "../base/Sound_recorder_config.hpp"
#include "../base/Sound_file_config.hpp"
#include "../base/error/Kinetophone_error.hpp"
#include <string>
#include <sstream>
#include <iomanip>
#include <cmath>

using Gdk::Pixbuf;
using Glib::RefPtr;
using std::endl;
using std::string;
using std::stringstream;
using sigc::mem_fun;
using Roan_trail::Recorder::Narration_recorder;
using Roan_trail::Source::Slide_collection;
using namespace Roan_trail::Kinetophone;

//
// Constructor/destructor/copy
//

Kinetophone_narrator_controller::Kinetophone_narrator_controller(Kinetophone_narrator_model& model,
                                                                 Kinetophone_narrator_config& config)
  : m_model(model),
    m_config(config),
    m_update_rate(config.update_rate),
    m_update_tick_count(0),
    m_update_period_for_record(config.update_rate / 3),
    m_time_update_period_for_pause((config.update_rate / 3) * 4),
    m_flash_period_for_pause((config.update_rate / 3) * 8),
    m_space_update_period_for_nonrecord((config.update_rate / 3) * 10),
    m_setup(false),
    m_signal_model_slides_will_update(),
    m_signal_model_recorder_will_update(),
    m_signal_model_slides_updated(),
    m_signal_model_recorder_updated(),
    m_signal_update_time(),
    m_signal_update_levels(),
    m_signal_update_available_space(),
    m_signal_update_statusbar(),
    m_signal_toggle_metering(),
    m_signal_reset_clipped(),
    m_signal_overwrite_recording(),
    m_signal_session_info(),
    m_signal_error(),
    m_update_tick_connection()
{
  // make a singleton by convention
  precondition(!m_instance);

  m_instance = this;

  Narration_recorder* recorder = m_model.narration_recorder();
  recorder->set_metering_channels(min(2, config.sound_recorder_config->sound_file->channels));
  recorder->enable_RMS_metering(true);

  postcondition(mf_invariant(false));
}

//
// Public member functions
//

void Kinetophone_narrator_controller::setup()
{
  precondition(!m_setup
               && mf_invariant());

  // initial update
  mf_initial_update();

  // setup timer for update
  m_update_tick_connection
    = Glib::signal_timeout().connect(mem_fun(*this, &Kinetophone_narrator_controller::mf_update_tick),
                                     1000 / m_update_rate);

  m_setup = true;

  postcondition(m_setup
                && mf_invariant());
}

//
// Slots
//

void Kinetophone_narrator_controller::retake()
{
  Narration_recorder* recorder = m_model.narration_recorder();
  if (Roan_trail::Recorder::Recorder::state_recording == recorder->state())
  {
    Error_param error;
    recorder->retake(error);
    assert(!error() && "unexpected error calling retake"); // this would be a programming error
  }
  else
  {
    assert(false && "error, retake when not recording");
  }
}

void Kinetophone_narrator_controller::previous()
{
  m_signal_model_slides_will_update.emit();

  Narration_recorder* recorder = m_model.narration_recorder();
  if (Roan_trail::Recorder::Recorder::state_recording == recorder->state())
  {
    m_model.slides()->previous_slide();
    Error_param error;
    recorder->new_index(m_model.slides()->current_slide_index(), error);
    assert(!error() && "unexpected error calling new index (previous)"); // this would be a programming error
  }
  else
  {
    m_model.slides()->previous_slide();
  }

  m_signal_model_slides_updated.emit();
}

void Kinetophone_narrator_controller::slide(int slide_index)
{
  precondition((slide_index >= 0)
               && (slide_index < m_model.slides()->slide_count())
               && mf_invariant());

  m_signal_model_slides_will_update.emit();

  Slide_collection<RefPtr<Pixbuf> >* slides = m_model.slides();
  Narration_recorder* recorder = m_model.narration_recorder();
  if (Roan_trail::Recorder::Recorder::state_recording == recorder->state())
  {
    slides->set_current_slide_index(slide_index);
    Error_param error;
    recorder->new_index(m_model.slides()->current_slide_index(), error);
    assert(!error() && "unexpected error calling new index (slide)"); // this would be a programming error
  }
  else
  {
    m_model.slides()->set_current_slide_index(slide_index);
  }

  m_signal_model_slides_updated.emit();

  postcondition(mf_invariant());
}

void Kinetophone_narrator_controller::next()
{
  precondition(mf_invariant());

  m_signal_model_slides_will_update.emit();

  Narration_recorder* recorder = m_model.narration_recorder();
  if (Roan_trail::Recorder::Recorder::state_recording == recorder->state())
  {
    m_model.slides()->next_slide();
    Error_param error;
    recorder->new_index(m_model.slides()->current_slide_index(), error);
    assert(!error() && "unexpected error calling new index (next)"); // this would be a programming error
  }
  else
  {
    m_model.slides()->next_slide();
  }

  m_signal_model_slides_updated.emit();

  postcondition(mf_invariant());
}

void Kinetophone_narrator_controller::thumbnail_choose(int slide_index)
{
  precondition(mf_invariant());

  m_signal_model_slides_will_update.emit();

  Slide_collection<RefPtr<Pixbuf> >* slides = m_model.slides();
  Narration_recorder* recorder = m_model.narration_recorder();
  if (Roan_trail::Recorder::Recorder::state_recording == recorder->state())
  {
    slides->set_current_slide_index(slide_index);
    Error_param error;
    recorder->new_index(m_model.slides()->current_slide_index(), error);
    assert(!error() && "unexpected error calling new index (thumbnail)"); // this would be a programming error
  }
  else
  {
    m_model.slides()->set_current_slide_index(slide_index);
  }

  m_signal_model_slides_updated.emit();

  postcondition(mf_invariant());
}


void Kinetophone_narrator_controller::mute()
{
  precondition(mf_invariant());

  Narration_recorder* recorder = m_model.narration_recorder();

  Error_param error(false);
  recorder->mute(error); // ignore error

  postcondition(mf_invariant());
}

void Kinetophone_narrator_controller::record()
{
  precondition(mf_invariant());

  Error_param error;

  start_error_block();

  // TODO: beep beep beep countdown
  m_signal_model_recorder_will_update.emit();

  Narration_recorder* recorder = m_model.narration_recorder();
  if (Roan_trail::Recorder::Recorder::state_stopped == recorder->state())
  {
    if (recorder->have_recording())
    {
      const bool proceed_with_overwrite = m_signal_overwrite_recording.emit();
      if (!proceed_with_overwrite)
      {
        goto exit_point;
      }
    }
    recorder->clear_record_error();
    const bool recording_started = recorder->record(m_model.slides()->current_slide_index(), error);
    on_error(!recording_started, new Kinetophone_error(error_location(),
                                                       Kinetophone_error::record,
                                                       error()));
    m_model.set_dirty();
  }
  else
  {
    const bool recording_stopped = recorder->stop(error);
    on_error(!recording_stopped, new Kinetophone_error(error_location(),
                                                       Kinetophone_error::record,
                                                       error()));
    m_signal_update_time.emit(false);
    // one last check for record error after stop
    const Error* record_error = recorder->record_error();
    if (record_error)
    {
      recorder->clear_record_error();
      on_error(record_error, new Kinetophone_error(error_location(),
                                                 Kinetophone_error::record,
                                                 record_error));
    }
    stringstream session_info_message;
    session_info_message << "Session complete. The audio recording is at: " << endl;;
    session_info_message << "  " << recorder->output_file();
    if (!recorder->config()->file_overwrite)
    {
      session_info_message << endl << "No further recording is allowed because overwrite is off.";
    }
    m_signal_session_info.emit(session_info_message.str());
  }

  m_signal_model_recorder_updated.emit();

  goto exit_point;

  end_error_block();

 error_handler:
  m_signal_error.emit(*handler_error, false);
  goto error_cleanup;

 error_cleanup:
  delete handler_error;
  m_signal_model_recorder_updated.emit();
  goto exit_point;

 exit_point:
  postcondition(mf_invariant());
}

void Kinetophone_narrator_controller::pause()
{
  precondition(mf_invariant());

  start_error_block();

  m_signal_model_recorder_will_update.emit();

  Narration_recorder* recorder = m_model.narration_recorder();
  Error_param error;
  if (Roan_trail::Recorder::Recorder::state_recording == recorder->state())
  {
    bool recording_paused = recorder->pause(error);
    on_error(!recording_paused, new Kinetophone_error(error_location(),
                                                      Kinetophone_error::record,
                                                      error()));
  }
  else if (Roan_trail::Recorder::Recorder::state_paused == recorder->state())
  {
    bool recording_resumed = recorder->record(m_model.slides()->current_slide_index(), error);
    on_error(!recording_resumed, new Kinetophone_error(error_location(),
                                                       Kinetophone_error::record,
                                                       error()));
  }
  else
  {
    assert(false && "error, invalid recorder state when pausing");
  }

  m_signal_model_recorder_updated.emit();

  goto exit_point;

  end_error_block();

 error_handler:
  m_signal_error.emit(*handler_error, false);
  goto error_cleanup;

 error_cleanup:
  delete handler_error;
  m_signal_model_recorder_updated.emit();
  goto exit_point;

 exit_point:
  postcondition(mf_invariant());
}

void Kinetophone_narrator_controller::toggle_metering()
{
  precondition(mf_invariant());

  Narration_recorder* recorder = m_model.narration_recorder();
  recorder->enable_RMS_metering(!recorder->is_RMS_metering_enabled());
  m_signal_toggle_metering.emit();

  postcondition(mf_invariant());
}

void Kinetophone_narrator_controller::reset_overflow_counts()
{
  precondition(mf_invariant());

  Narration_recorder* recorder = m_model.narration_recorder();
  recorder->reset_overflow_count();
  m_signal_update_statusbar.emit();

  postcondition(mf_invariant());
}

void Kinetophone_narrator_controller::reset_clipped()
{
  precondition(mf_invariant());

  m_signal_reset_clipped.emit();

  postcondition(mf_invariant());
}

void Kinetophone_narrator_controller::notes_changed()
{
  precondition(mf_invariant());

  m_model.set_dirty();

  postcondition(mf_invariant());
}

void Kinetophone_narrator_controller::notes_editing_done(const string &current_notes)
{
  precondition(mf_invariant());

  Slide_collection<RefPtr<Pixbuf> >* slides = m_model.slides();
  const int slide_index = m_model.slides()->current_slide_index();
  (*slides)[slide_index]->set_notes(current_notes);

  postcondition(mf_invariant());
}

void Kinetophone_narrator_controller::export_session(const string& session_file)
{
  precondition((Roan_trail::Recorder::Recorder::state_stopped == m_model.narration_recorder()->state())
               && mf_invariant());

  start_error_block();

  Error_param error;

  bool session_saved = m_model.export_as_XML(session_file, error);
  on_error(!session_saved, new Kinetophone_error(error_location(),
                                                 Kinetophone_error::general,
                                                 error()));

  m_model.set_dirty(false);

  goto exit_point;

  end_error_block();

 error_handler:
  m_signal_error.emit(*handler_error, false);
  goto error_cleanup;

 error_cleanup:
  delete handler_error;
  goto exit_point;

 exit_point:
  postcondition(mf_invariant());
  return;
}

void Kinetophone_narrator_controller::quit()
{
  precondition(mf_invariant());

  m_setup = false;

  m_update_tick_connection.disconnect();

  Gtk::Main::quit();
}

//
// Protected member functions
//

bool Kinetophone_narrator_controller::mf_invariant(bool check_base_class) const
{
  bool return_value = false;

  // range checking
  if ((m_update_rate < 1)
      || (m_update_tick_count < 0)
      || (m_update_period_for_record < 1)
      || (m_time_update_period_for_pause < 1)
      || (m_flash_period_for_pause < 1)
      || (m_space_update_period_for_nonrecord < 1))
  {
    goto exit_point;
  }

  // all signals should be connected and connections active when setup
  if (m_setup
      && (m_signal_model_slides_will_update.empty()
          || m_signal_model_recorder_will_update.empty()
          || m_signal_model_slides_updated.empty()
          || m_signal_model_recorder_updated.empty()
          || m_signal_update_time.empty()
          || m_signal_update_levels.empty()
          || m_signal_update_available_space.empty()
          || m_signal_update_statusbar.empty()
          || m_signal_toggle_metering.empty()
          || m_signal_reset_clipped.empty()
          || m_signal_overwrite_recording.empty()
          || m_signal_session_info.empty()
          || m_signal_error.empty()
          || m_update_tick_connection.empty()))
  {
    goto exit_point;
  }

  return_value = true;

 exit_point:
  return return_value;
}

//
// Private data members
//

Kinetophone_narrator_controller* Kinetophone_narrator_controller::m_instance = 0;

//
// Private member functions
//

//
// Update
//

void Kinetophone_narrator_controller::mf_initial_update()
{
  m_signal_update_available_space.emit();
  m_signal_update_time.emit(false);
  m_signal_update_statusbar.emit();
  m_signal_model_slides_updated.emit();
  m_signal_model_recorder_updated.emit();
}

bool Kinetophone_narrator_controller::mf_update_tick()
{
  start_error_block();

  ++m_update_tick_count;
  Narration_recorder* recorder = m_model.narration_recorder();
  const int recorder_state = recorder->state();

  // update time display
  if (!(m_update_tick_count % m_update_period_for_record))
  {
    if (Roan_trail::Recorder::Recorder::state_recording == recorder_state)
    {
      m_signal_update_time.emit(false);
    }
    else if (Roan_trail::Recorder::Recorder::state_paused == recorder_state)
    {
      if (!(m_update_tick_count % m_time_update_period_for_pause))
      {
        m_signal_update_time.emit(m_update_tick_count % m_flash_period_for_pause);
      }
    }
  }

  // update available space
  if (Roan_trail::Recorder::Recorder::state_recording == recorder_state)
  {
    if (!(m_update_tick_count % m_update_period_for_record))
    {
      m_signal_update_available_space.emit();
    }
    m_signal_update_statusbar.emit();
  }
  else if (!(m_update_tick_count % m_space_update_period_for_nonrecord))
  {
    // update more leisurely when stopped/paused
    m_signal_update_available_space.emit();
    m_signal_update_statusbar.emit();
  }

  // update audio levels if enabled
  if (recorder->is_RMS_metering_enabled())
  {
    m_signal_update_levels.emit();
  }

  // check for an record error between updates, if recording
  if (Roan_trail::Recorder::Recorder::state_recording == recorder_state)
  {
    const Error* record_error = recorder->record_error();
    if (record_error)
    {
      Error_param error(false);
      recorder->stop(error); // ignore error
      recorder->clear_record_error();
      m_signal_update_time.emit(false);
      m_signal_model_recorder_updated.emit();
      on_error(true, new Kinetophone_error(error_location(),
                                           Kinetophone_error::record,
                                           record_error));
    }
  }

  goto exit_point;

  end_error_block();

 error_handler:
  m_signal_error.emit(*handler_error, false);
  goto error_cleanup;

 error_cleanup:
  delete handler_error;
  goto exit_point;

 exit_point:
  return true; // BTW, returning false stops the timer here
}
