// Kinetophone_narrator_config.cpp
//
// Copyright 2011-2013 Roan Trail, Inc.
//
// This file is part of Kinetophone.
//
// Kinetophone is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published
// by the Free Software Foundation, either version 2 of the License,
// or (at your option) any later version.
//
// Kinetophone is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// General Public License for more details.  You should have received
// a copy of the GNU General Public License along with Kinetophone. If
// not, see <http://www.gnu.org/licenses/>.

#include "Kinetophone_narrator_config.hpp"
#include "../base/common.hpp"
#include "../base/File_manager.hpp"
#include "../base/Image_types.hpp"
#include "../base/Level_meter_view.hpp"
#include "../base/option_checks.hpp"
#include "../base/Option_validators.hpp"
#include "../base/Sound_file_config.hpp"
#include "../base/Sound_recorder_config.hpp"
#include "../base/error/Boost_error.hpp"
#include "../base/error/Kinetophone_error.hpp"
#include <boost/program_options.hpp>
#include <exception>
#include <fstream>
#include <sstream>
#include <string>
#include <cstdlib>

using std::endl;
using std::exception;
using std::ifstream;
using std::logic_error;
using std::string;
using std::stringstream;
using Roan_trail::Color;
using Roan_trail::File_manager;
using Roan_trail::Options::check_conflicting_options;
using Roan_trail::Options::check_option_dependency;
using Roan_trail::Options::Color_validator;
using Roan_trail::Options::Directory_validator;
using Roan_trail::Options::Fraction_validator;
using Roan_trail::Options::Positive_int_validator;
using Roan_trail::Options::Str_to_int_lookup_validator;
using Roan_trail::Options::Validator;
using Roan_trail::Recorder::Sound_file_config;
using namespace Roan_trail::Kinetophone;

namespace PO = boost::program_options;

//
// Internal helpers
//

namespace
{
  using Roan_trail::Recorder::Sound_recorder_config;

  //
  // some custom validation code for this configuration
  //

  bool ih_filter_file_type_string(const string& file_type_string, int& check_file_type)
  {
    bool return_value = false;

    if (Sound_file_config::file_type_for_string(file_type_string, check_file_type))
    {
      switch (check_file_type)
      {
      // of the ones supported by Kinetophone, only these are allowed on the command line
      case Sound_file_config::file_type_wav:
      case Sound_file_config::file_type_aiff:
      case Sound_file_config::file_type_flac:
        return_value = true;
        break;
      default:
        break;
      }
    }

    return return_value;
  }

  bool ih_filter_data_format_string(const string& data_format_string, int& check_data_format)
  {
    bool return_value = false;

    if (Sound_file_config::data_format_for_string(data_format_string, check_data_format))
    {
      switch (check_data_format)
      {
      // of the ones supported by Kinetophone, only these are allowed on the command line
      case Sound_file_config::data_format_PCMS8:
      case Sound_file_config::data_format_PCM16:
      case Sound_file_config::data_format_PCM24:
      case Sound_file_config::data_format_PCM32:
      case Sound_file_config::data_format_float:
      case Sound_file_config::data_format_double:
        return_value = true;
        break;
      default:
        break;
      }
    }

    return return_value;
  }

  typedef struct Str_to_int_lookup_validator<string,
                                             ih_filter_file_type_string> Ih_file_type_validator;
  typedef struct Str_to_int_lookup_validator<string,
                                             ih_filter_data_format_string> Ih_data_format_validator;
  typedef struct Str_to_int_lookup_validator<string,
                                             Sound_file_config::endianness_for_string> Ih_endianness_validator;

  //
  // the main validator struct for this configuration
  //

  struct Ih_validator_struct
  {
    Ih_validator_struct(const Kinetophone_narrator_config& s)
      : image_source_path(s.image_source_path),
        image_directory(),
        PDF_fill_color(Color_validator(*s.PDF_fill_color)),
        import_PDF_notes(Validator<bool>(s.import_PDF_notes)),
        session_file_path(s.session_file_path),
        level_meter(s.level_meter),
        update_rate(Positive_int_validator(s.update_rate)),
        show_file_size(Validator<bool>(s.show_file_size)),
        show_audio_info(Validator<bool>(s.show_audio_info)),
        show_overflow_count(Validator<bool>(s.show_overflow_count)),
        detailed_error(Validator<bool>(s.detailed_error)),
        custom_installation_dir(Directory_validator(s.custom_installation_dir)),
        sound_recorder_config(*s.sound_recorder_config) {}
    void update(Kinetophone_narrator_config& return_cfg)
    {
      *return_cfg.PDF_fill_color = PDF_fill_color.v;
      return_cfg.import_PDF_notes = import_PDF_notes.v;
      return_cfg.session_file_path = session_file_path;
      level_meter.update(return_cfg.level_meter);
      return_cfg.update_rate = update_rate.v;
      return_cfg.show_file_size = show_file_size.v;
      return_cfg.show_audio_info = show_audio_info.v;
      return_cfg.show_overflow_count = show_overflow_count.v;
      return_cfg.detailed_error = detailed_error.v;
      return_cfg.custom_installation_dir = custom_installation_dir.v;
      sound_recorder_config.update(*return_cfg.sound_recorder_config);
      // calculate and assign remaining members
      if (("" == image_source_path))
      {
        if ("" != image_directory.v)
        {
          return_cfg.image_source_path = image_directory.v;
        }
        else
        {
          // if not specified, use the current directory for the image source path
          return_cfg.image_source_path = ".";
        }
      }
      else
      {
        return_cfg.image_source_path = image_source_path;
      }
      string use_file_type = sound_recorder_config.sound_file.file_type.v;
      if ("" == use_file_type)
      {
        if ("" != sound_recorder_config.sound_file.file_name)
        {
          // file type not specified, but output file is
          // try to determine from output file extension
          use_file_type = File_manager::file_extension_for_file(sound_recorder_config.sound_file.file_name);
        }
        else
        {
          // neither specified, use default
          use_file_type =
            Sound_file_config::string_for_file_type(Sound_file_config::default_file_type);
        }
      }
      if (!Sound_file_config::file_type_for_string(use_file_type,
                                                   return_cfg.sound_recorder_config->sound_file->file_type))
      {
        // not an an understood file type, this will be an error
        return_cfg.sound_recorder_config->sound_file->file_type = Sound_file_config::file_type_unknown;
      }
      if (!Sound_file_config::data_format_for_string(sound_recorder_config.sound_file.data_format.v,
                                                     return_cfg.sound_recorder_config->sound_file->data_format))
      { // use default data format
        return_cfg.sound_recorder_config->sound_file->data_format = Sound_file_config::default_data_format;
      }
      if (!Sound_file_config::endianness_for_string(sound_recorder_config.sound_file.endianness.v,
                                                    return_cfg.sound_recorder_config->sound_file->endianness))
      { // use default endianness
        return_cfg.sound_recorder_config->sound_file->endianness = Sound_file_config::default_endianness;
      }
    }
    string image_source_path;
    Directory_validator image_directory;
    Color_validator PDF_fill_color;
    Validator<bool> import_PDF_notes;
    string session_file_path;
    struct Level_meter_validator_struct
    {
      Level_meter_validator_struct(const Kinetophone_narrator_config::Level_meter_struct& s)
        : model_ballistics(Validator<bool>(s.model_ballistics)),
          attack_period(Positive_int_validator(s.attack_period)),
          decay_period(Positive_int_validator(s.decay_period)),
          peak_hold_period(Positive_int_validator(s.peak_hold_period)) {}
      void update(Kinetophone_narrator_config::Level_meter_struct& return_s)
      {
        return_s.model_ballistics = model_ballistics.v;
        return_s.attack_period = attack_period.v;
        return_s.decay_period = decay_period.v;
        return_s.peak_hold_period = peak_hold_period.v;
      }
      //
      Validator<bool> model_ballistics;
      Positive_int_validator attack_period;
      Positive_int_validator decay_period;
      Positive_int_validator peak_hold_period;
    } level_meter;
    Positive_int_validator update_rate;
    Validator<bool> show_file_size;
    Validator<bool> show_audio_info;
    Validator<bool> show_overflow_count;
    Validator<bool> detailed_error;
    Directory_validator custom_installation_dir;
    struct Sound_recorder_config_validator_struct
    {
      Sound_recorder_config_validator_struct(const Sound_recorder_config& s)
        : input_device(s.input_device),
          frames_per_buffer(Positive_int_validator(s.frames_per_buffer)),
          write_buffer_factor(Positive_int_validator(s.write_buffer_factor)),
          RMS_level_integration_period(Positive_int_validator(s.RMS_level_integration_period)),
          sound_file(*s.sound_file),
          file_overwrite(s.file_overwrite) {}
      void update(Sound_recorder_config& return_s)
      {
        return_s.input_device = input_device;
        return_s.frames_per_buffer = frames_per_buffer.v;
        return_s.write_buffer_factor = write_buffer_factor.v;
        return_s.RMS_level_integration_period = RMS_level_integration_period.v;
        sound_file.update(*return_s.sound_file);
        return_s.file_overwrite = file_overwrite.v;
      }
      //
      int input_device;
      Positive_int_validator frames_per_buffer;
      Positive_int_validator write_buffer_factor;
      Positive_int_validator RMS_level_integration_period;
      struct Sound_file_config_validator_struct
      {
        Sound_file_config_validator_struct(const Sound_file_config& s)
          : channels(Positive_int_validator(s.channels)),
            sample_rate(Positive_int_validator(static_cast<int>(s.sample_rate))),
            VBR_quality(Fraction_validator(s.VBR_quality)),
            file_name(s.file_name),
            file_type(Ih_file_type_validator("")),
            data_format(Ih_data_format_validator(Sound_file_config::string_for_data_format(s.data_format))),
            endianness(Ih_endianness_validator(Sound_file_config::string_for_endianness(s.endianness))) {}
        void update(Sound_file_config& return_s)
        {
          return_s.channels = channels.v;
          return_s.sample_rate = sample_rate.v;
          return_s.VBR_quality = VBR_quality.v;
          return_s.file_name = file_name;
          if (!Sound_file_config::file_type_for_string(file_type.v, return_s.file_type))
          {
            return_s.file_type = Sound_file_config::file_type_unknown;
          }
          if (!Sound_file_config::data_format_for_string(data_format.v, return_s.data_format))
          {
            assert(false && "invalid data format string");
          }
          if (!Sound_file_config::endianness_for_string(endianness.v, return_s.endianness))
          {
            assert(false && "invalid endianness string");
          }
        }
        //
        Positive_int_validator channels;
        Positive_int_validator sample_rate;
        Fraction_validator VBR_quality;
        string file_name;
        Ih_file_type_validator file_type;
        Ih_data_format_validator data_format;
        Ih_endianness_validator endianness;
      } sound_file;
      Validator<bool> file_overwrite;
    } sound_recorder_config;
  };

  void ih_check_option_consistency(const Kinetophone_narrator_config& cfg,
                                   const PO::variables_map& VM)
  {
    // conflicting option checks
    check_conflicting_options(VM,
                              "file-type",
                              "overwrite",
                              (cfg.sound_recorder_config->file_overwrite
                               && (Sound_file_config::file_type_flac
                                   == cfg.sound_recorder_config->sound_file->file_type)),
                              "FLAC file format cannot be specified with file overwrite");
    check_conflicting_options(VM,
                              "image-directory",
                              "image-source-path");
    check_conflicting_options(VM,
                              "import-notes-from-file",
                              "import-PDF-notes",
                              cfg.import_PDF_notes);
    // option dependency checks
    check_option_dependency(VM,
                            "level-meter-attack-period",
                            "level-meter-ballistics",
                            cfg.level_meter.model_ballistics);
    check_option_dependency(VM,
                            "level-meter-decay-period",
                            "level-meter-ballistics",
                            cfg.level_meter.model_ballistics);
    // invalid file type
    if (Sound_file_config::file_type_unknown == cfg.sound_recorder_config->sound_file->file_type)
    {
      throw logic_error("could not determine a valid sound file type");
    }
  }

  typedef struct
  {
    PO::options_description& cmd_line_desc;
    PO::options_description& basic_desc;
    PO::options_description& advanced_desc;
  } Ih_desc_params;

  // sets up the program option description object
  void ih_setup_descriptions(Ih_desc_params& return_desc_params,
                             Ih_validator_struct& return_v,
                             string& return_config_file)
  {
    const char *home_dir = getenv("HOME"); // (check_code_ignore)
    string default_config_file = (!home_dir || (*home_dir == 0) || !File_manager::path_is_directory(home_dir)) ?
      string("") : string(home_dir) + string("/.kinetophone");

    return_desc_params.cmd_line_desc.add_options()
      ("help,h", "output help message")
      ("list-devices,l", "generate a list of devices")
      ("config,i", PO::value<string>(&return_config_file)->default_value(default_config_file),
       "specify configuration file")
      ("version,v", "print program version")
      ;

    return_desc_params.basic_desc.add_options()
      ("audio-info,I", PO::value<Validator<bool> >(&return_v.show_audio_info)
       ->implicit_value(Validator<bool>(true)),
       "show audio information when recording")
      ("device,d", PO::value<int>(&return_v.sound_recorder_config.input_device)
       ->default_value(return_v.sound_recorder_config.input_device),
       "set input device number (from --list-devices option)")
      ("image-directory,D", PO::value<Directory_validator>(&return_v.image_directory),
       "set image source directory")
      ("image-source-path,S", PO::value<string>(&return_v.image_source_path)
       ->default_value(return_v.image_source_path),
       "set image source path (directory or PDF file)")
      ("import-notes-from-file,n", PO::value<string>(&return_v.session_file_path)
       ->default_value(return_v.session_file_path),
       "import notes from session XML file (cannot use with --import-PDF-notes yes")
      ("sound-file,f", PO::value<string>(&return_v.sound_recorder_config.sound_file.file_name)
       ->default_value(return_v.sound_recorder_config.sound_file.file_name),
       "set destination sound file path")
      ("overwrite,o", PO::value<Validator<bool> >(&return_v.sound_recorder_config.file_overwrite)
       ->implicit_value(Validator<bool>(true)),
       "overwrite sound recording file")
      ("channels,c", PO::value<Positive_int_validator>(&return_v.sound_recorder_config.sound_file.channels)
       ->default_value(return_v.sound_recorder_config.sound_file.channels),
       "set channels to record (must be > 0)")
      ("sample-rate,s", PO::value<Positive_int_validator>(&return_v.sound_recorder_config.sound_file.sample_rate)
       ->default_value(return_v.sound_recorder_config.sound_file.sample_rate),
       "set samples per second (must be > 0)")
      ("file-type,t", PO::value<Ih_file_type_validator>(&return_v.sound_recorder_config.sound_file.file_type),
       "set audio file type\n"
       "{ wav | aiff | flac }") // (check_code_ignore)
      ("data-format,a", PO::value<Ih_data_format_validator>(&return_v.sound_recorder_config.sound_file
                                                            .data_format)
       ->default_value(return_v.sound_recorder_config.sound_file.data_format),
       "set format of samples in the audio file (signed values)\n"
       "{ pcm8 | pcm16 | pcm24 | pcm32 | float | double }")
      ("show-file-size,F", PO::value<Validator<bool> >(&return_v.show_file_size)
       ->implicit_value(Validator<bool>(true)),
       "display the size of the output file")
      ("PDF-fill-color,L",
       PO::value<Color_validator>(&return_v.PDF_fill_color)
       ->default_value(return_v.PDF_fill_color),
       "set fill color for PDF background (format RRGGBB, in hexadecimal")
      ("import-PDF-notes,N", PO::value<Validator<bool> >(&return_v.import_PDF_notes)
       ->default_value(return_v.import_PDF_notes),
       "import PDF text annotations as slide notes (value cannot be yes with --import-notes-from-file)")
      ;

    return_desc_params.advanced_desc.add_options()
      ("endian,E", PO::value<Ih_endianness_validator>(&return_v.sound_recorder_config.sound_file.endianness)
       ->default_value(return_v.sound_recorder_config.sound_file.endianness),
       "set endian order in the output file\n"
       "\t{ file | little | big | cpu }")
      ("frames-per-buffer,b", PO::value<Positive_int_validator>(&return_v.sound_recorder_config.frames_per_buffer)
       ->default_value(return_v.sound_recorder_config.frames_per_buffer),
       "set number of samples processed at a time (must be > 0)")
      ("write-buffer-factor,W",
       PO::value<Positive_int_validator>(&return_v.sound_recorder_config.write_buffer_factor)
       ->default_value(return_v.sound_recorder_config.write_buffer_factor),
       "set factor to multiply frames per buffer for sizing write buffer (must be > 0)")
      ("RMS-integration-period,R",
       PO::value<Positive_int_validator>(&return_v.sound_recorder_config.RMS_level_integration_period)
       ->default_value(return_v.sound_recorder_config.RMS_level_integration_period),
       "set the period in milliseconds used for RMS level averaging (must be > 0)")
      ("update-rate,U", PO::value<Positive_int_validator>(&return_v.update_rate)
       ->default_value(return_v.update_rate),
       "set number of times display is refreshed per second (must be > 0)")
      ("level-meter-ballistics,B", PO::value<Validator<bool> >(&return_v.level_meter.model_ballistics)
       ->implicit_value(Validator<bool>(return_v.level_meter.model_ballistics)),
       "model ballistics in level meter (requires --level-meter or -m)")
      ("level-meter-attack-period,P", PO::value<Positive_int_validator>(&return_v.level_meter.attack_period)
       ->default_value(return_v.level_meter.attack_period),
       "set the number of milliseconds for the level meter to travel forward 0.99X full scale (must be > 0, "
       "requires --level-meter-ballistics)")
      ("level-meter-decay-period,p", PO::value<Positive_int_validator>(&return_v.level_meter.decay_period)
       ->default_value(return_v.level_meter.decay_period),
       "set the number of milliseconds for the level meter to decay back from 0.99X full scale (must be > 0, "
       "requires --level-meter-ballistics)")
      ("level-meter-peak-hold-period,H", PO::value<Positive_int_validator>(&return_v.level_meter.peak_hold_period)
       ->default_value(return_v.level_meter.peak_hold_period),
       "set the number of milliseconds for the level meter to maintain peak (must be > 0)")
      ("detailed-error,e", PO::value<Validator<bool> >(&return_v.detailed_error)
       ->implicit_value(Validator<bool>(true)),
       "display detailed diagnostics on error")
      ("overflow-count,O", PO::value<Validator<bool> >(&return_v.show_overflow_count)
       ->implicit_value(Validator<bool>(true)),
       "show input/output overflow count")
      ("VBR-quality,q", PO::value<Fraction_validator>(&return_v.sound_recorder_config.sound_file.VBR_quality)
       ->default_value(Sound_file_config::highest_VBR_quality),
       "set variable bit rate (VBR) quality (0.0 = lowest to 1.0 = highest)")
      ("config-directory,C", PO::value<Directory_validator>(&return_v.custom_installation_dir),
       "set custom program configuration directory")
      ;
  }
}

//
// Constructor/destructor/copy
//

Kinetophone_narrator_config::Kinetophone_narrator_config()
  // setup some reasonable defaults
  : help_message(""),
    image_source_path(""),
    PDF_fill_color(new Color(default_PDF_fill_color, false)),
    import_PDF_notes(true),
    session_file_path(),
    level_meter(Level_meter_struct()),
    update_rate(default_update_rate),
    show_file_size(false),
    show_audio_info(false),
    show_overflow_count(false),
    detailed_error(false),
    custom_installation_dir(""),
    sound_recorder_config(new Sound_recorder_config)
{
}

Kinetophone_narrator_config::~Kinetophone_narrator_config()
{
  delete sound_recorder_config;
  delete PDF_fill_color;
}

Kinetophone_narrator_config::Level_meter_struct::Level_meter_struct()
  // setup some reasonable defaults
  : model_ballistics(true),
    attack_period(Level_meter_view::default_attack_period),
    decay_period(Level_meter_view::default_decay_period),
    peak_hold_period(Level_meter_view::default_peak_hold_period)
{
}

//
//  Class constants
//

//   default
const int Kinetophone_narrator_config::default_update_rate = 30;
const uint32_t Kinetophone_narrator_config::default_PDF_fill_color = 0xFFFFFFFF; // white
//   command
const int Kinetophone_narrator_config::command_error;
const int Kinetophone_narrator_config::command_narrate;
const int Kinetophone_narrator_config::command_help;
const int Kinetophone_narrator_config::command_list_devices;
const int Kinetophone_narrator_config::command_output_version;

int Kinetophone_narrator_config::parse_program_options(int argc,
                                                       const char** argv,
                                                       Error_param& return_error)
{
  precondition(!return_error());

  int return_value = Kinetophone_narrator_config::command_error;

  help_message = "";

  start_error_block();

  try
  {
    PO::options_description cmd_line_desc("Command line only");
    stringstream basic_desc_label;
    basic_desc_label << "Command line and configuration file" << endl << "  Basic";
    PO::options_description basic_desc(basic_desc_label.str());
    PO::options_description advanced_desc("  Advanced");
    Ih_desc_params desc_params =
    {
      cmd_line_desc,
      basic_desc,
      advanced_desc
    };

    Ih_validator_struct validator(*this);
    string config_file;
    ih_setup_descriptions(desc_params,
                          validator,
                          config_file);

    // parse the command line to get config file
    PO::variables_map VM_config;
    PO::store(PO::command_line_parser(argc, const_cast<char**>(argv)).
              options(cmd_line_desc).allow_unregistered().run(), VM_config);
    notify(VM_config);

    PO::options_description config_desc;
    config_desc.add(basic_desc).add(advanced_desc);

    stringstream help_stream;
    help_stream << "Usage: " << endl;
    help_stream << "kinetophone_narrator --help" << endl << endl;
    help_stream << "kinetophone_narrator --list-devices" << endl << endl;
    help_stream << "kinetophone_narrator [image-directory [sound_file ]] [ options ]" << endl << endl;
    help_stream << "kinetophone_narrator [PDF-file-path [sound_file ]] [ options ]" << endl << endl;

    PO::options_description all_desc("Options");
    all_desc.add(cmd_line_desc).add(basic_desc).add(advanced_desc);
    help_stream << all_desc;
    help_message = help_stream.str();

    PO::variables_map VM;

    // parse the command line (overrides config file)
    PO::positional_options_description pos_desc;
    pos_desc.add("image-source-path", 1);
    pos_desc.add("sound-file", 1);
    PO::store(PO::command_line_parser(argc, const_cast<char**>(argv)).
              options(all_desc).positional(pos_desc).run(), VM);
    PO::notify(VM);

    // parse the config file
    if ("" != config_file)
    {
      ifstream ifs(config_file.c_str());
      bool have_error =  (!ifs && !VM_config["config"].defaulted());
      on_error(have_error, new Boost_error(error_location(), string("cannot open config file: ") + config_file));
      const bool allow_unregistered_options = true;
      PO::store(PO::parse_config_file(ifs, config_desc, allow_unregistered_options), VM);
      PO::notify(VM);
    }

    if (VM.count("help"))
    { // HELP
      return_value = Kinetophone_narrator_config::command_help;
    }
    else if (VM.count("list-devices"))
    { // LIST DEVICES
      return_value = Kinetophone_narrator_config::command_list_devices;
    }
    else if (VM.count("version"))
    {
      // PROGRAM VERSION
      return_value = Kinetophone_narrator_config::command_output_version;
    }
    else
    { // NARRATE
      // update the config from the validator
      validator.update(*this);
      ih_check_option_consistency(*this, VM);
      return_value = Kinetophone_narrator_config::command_narrate;
    }
  }
  catch (const exception& e)
  {
    string diagnostic = e.what();
    stringstream s;
    s << "Reason: " << diagnostic << std::endl << std::endl;
    s << "Enter kinetophone_narrator --help for valid program options.";
    Kinetophone_error* error = new Kinetophone_error(error_location(),
                                                     Kinetophone_error::command_line,
                                                     new Boost_error(error_location(), s.str()));
    on_error(true, error);
  }
  catch (...)
  {
    string diagnostic = "exception parsing command line";
    Kinetophone_error* error = new Kinetophone_error(error_location(),
                                                     Kinetophone_error::command_line,
                                                     diagnostic);
    on_error(true, error);
  }

  goto exit_point;

  end_error_block();

  default_error_handler(return_error);

 error_cleanup:
  if (!return_error.need_error())
  {
    delete handler_error;
  }
  return_value = Kinetophone_narrator_config::command_error;
  goto exit_point;

 exit_point:
  return return_value;
}
